// public/js/suppliers/modules/export-manager.js
// Simple export manager (CSV / XLSX stub).
(function (global) {
    class ExportManager {
        constructor(app) {
            this.app = app;
        }

        init() {
            this.bind();
        }

        bind() {
            // export buttons use .export-btn or id #export-vendors
            document.addEventListener('click', (e) => {
                const btn = e.target.closest('.export-btn, #export-vendors');
                if (!btn) return;
                const type = btn.getAttribute('data-type') || this.app.currentVendorType || 'service_providers';
                this.export(type);
            });
        }

        async export(type) {
            try {
                SuppliersHelpers.showGlobalLoading(true);
                // call a server endpoint; fallback: build CSV from table
                const url = `/service-providers/export?type=${encodeURIComponent(type)}`;
                try {
                    const res = await SuppliersHelpers.request(url, { method: 'GET' });
                    // if server returns fileUrl
                    if (res && res.fileUrl) {
                        window.open(res.fileUrl, '_blank');
                        this.app.showAlert('Export started', 'success');
                        return;
                    }
                } catch (err) {
                    // ignore server export failure — fallback to table CSV
                }
                // fallback: build csv from current table
                const csv = this.buildCsvFromTable();
                if (!csv) {
                    this.app.showAlert('Nothing to export', 'warning');
                    return;
                }
                const blob = new Blob([csv], { type: 'text/csv' });
                const link = document.createElement('a');
                link.href = URL.createObjectURL(blob);
                link.download = `${type}-export-${new Date().toISOString().slice(0,10)}.csv`;
                document.body.appendChild(link);
                link.click();
                link.remove();
                URL.revokeObjectURL(link.href);
                this.app.showAlert('Export downloaded', 'success');
            } catch (err) {
                this.app.showAlert('Export failed: ' + (err.message || ''), 'error');
            } finally {
                SuppliersHelpers.showGlobalLoading(false);
            }
        }

        buildCsvFromTable() {
            const table = document.querySelector('#vendors-table');
            if (!table) return '';
            const rows = Array.from(table.querySelectorAll('thead tr, tbody tr'));
            const lines = rows.map(row => {
                const cells = Array.from(row.querySelectorAll('th,td')).map(cell => {
                    // remove inner action buttons text
                    const txt = cell.innerText.replace(/\t/g, ' ').replace(/\n/g, ' ').trim();
                    const escaped = String(txt).replace(/"/g, '""');
                    return `"${escaped}"`;
                });
                return cells.join(',');
            });
            return lines.join('\n');
        }
    }

    global.ExportManager = ExportManager;
})(window);
