// public/js/suppliers/modules/app-shell.js
// App shell that wires modules and global behaviors.
// Attaches SuppliersApp to window.

(function (global) {

    class SuppliersApp {
        constructor() {
            this.modules = {};
            this.currentVendorType = 'service_providers'; // default view (matches radio values)
            this.init();
        }

        init() {
            this.registerBuiltins();
            this.initModules();
            this.setupVendorToggle();
            this.setupGlobalEvents();
        }

        registerBuiltins() {
            // Modules will be constructed if available on window
            this.moduleConstructors = {
                dashboard: global.DashboardModule || null,
                searchManager: global.SearchManager || null,
                exportManager: global.ExportManager || null,
                modalManager: global.ModalManager || null,
                vendorManager: global.VendorManager || null
            };
        }

        initModules() {
            Object.entries(this.moduleConstructors).forEach(([name, Ctor]) => {
                if (!Ctor) return;
                try {
                    const instance = new Ctor(this);
                    this.modules[name] = instance;
                    if (typeof instance.init === 'function') {
                        instance.init();
                    }
                    console.log(`SuppliersApp: module '${name}' initialized`);
                } catch (err) {
                    console.error(`SuppliersApp: failed to init module ${name}`, err);
                }
            });
        }

        setupVendorToggle() {
            // Support radio inputs or custom toggle buttons
            document.addEventListener('change', (e) => {
                if (e.target && e.target.name === 'vendor-type') {
                    const value = e.target.value;
                    this.currentVendorType = value;
                    document.dispatchEvent(new CustomEvent('suppliers:viewChanged', { detail: { view: value } }));
                }
            });

            // also expose programmatic toggle via buttons with data-view
            document.addEventListener('click', (e) => {
                const btn = e.target.closest('[data-suppliers-toggle]');
                if (!btn) return;
                const view = btn.getAttribute('data-suppliers-toggle');
                if (view) {
                    this.currentVendorType = view;
                    // set radio if present
                    const radio = document.querySelector(`input[name="vendor-type"][value="${view}"]`);
                    if (radio) radio.checked = true;
                    document.dispatchEvent(new CustomEvent('suppliers:viewChanged', { detail: { view } }));
                }
            });
        }

        setupGlobalEvents() {
            // allow modules to call app.showAlert(...)
            this.container = document.querySelector('.page-title-box') || document.body;
        }

        getModule(name) {
            return this.modules[name];
        }

        showAlert(message, type = 'info') {
            const map = {
                success: 'alert-success',
                error: 'alert-danger',
                warning: 'alert-warning',
                info: 'alert-info'
            };
            const cls = map[type] || 'alert-info';
            // remove previous
            document.querySelectorAll('.suppliers-app-alert').forEach(n => n.remove());
            const el = document.createElement('div');
            el.className = `alert ${cls} suppliers-app-alert`;
            el.innerHTML = `${message} <button type="button" class="btn-close" data-bs-dismiss="alert"></button>`;
            if (this.container && this.container.parentNode) {
                this.container.parentNode.insertBefore(el, this.container.nextSibling);
            } else {
                document.body.prepend(el);
            }
            setTimeout(() => el.remove(), 5000);
        }
    }

    global.SuppliersApp = SuppliersApp;
})(window);
