// Shipment Form Main Class
class ShipmentForm {
    constructor() {
        this.modules = {};
        this.baseCurrency = document.querySelector('input[name="base_currency"]') ? document.querySelector('input[name="base_currency"]').value : 'USD';
        this.init();
    }

    init() {
        // Check if we're on a shipment form page
        if (!document.getElementById('shipmentForm')) {
            console.log('Shipment form not found on this page');
            return;
        }

        this.initializeModules();
        this.setupGlobalEventListeners();
        this.generateTrackingNumber();
        
        console.log('Shipment form initialized successfully');
    }

    initializeModules() {
        try {
            // Define module classes in order of dependency
            const moduleClasses = {
                'utilities': Utilities,
                'stepManager': StepManager,
                'customerManager': CustomerManager,
                'packageManager': PackageManager,
                'routeManager': RouteManager,
                'pricingManager': PricingManager,
                'servicesManager': ServicesManager,
                'verificationManager': VerificationManager,
                'modalManager': ModalManager
            };

            // Initialize modules only if their classes are defined
            Object.entries(moduleClasses).forEach(([name, Class]) => {
                if (typeof Class !== 'undefined') {
                    this.modules[name] = new Class(this);
                    
                    // Initialize the module if it has an init method
                    if (this.modules[name] && typeof this.modules[name].init === 'function') {
                        this.modules[name].init();
                    }
                } else {
                    console.warn(`Module class ${name} is not defined`);
                }
            });

        } catch (error) {
            console.error('Error initializing modules:', error);
            this.showAlert('Error initializing form modules', 'error');
        }
    }

    setupGlobalEventListeners() {
        const form = document.getElementById('shipmentForm');
        if (form) {
            form.addEventListener('submit', (e) => this.handleFormSubmit(e));
        }

        // Generate tracking number button
        const generateBtn = document.getElementById('generateTrackingBtn');
        if (generateBtn) {
            generateBtn.addEventListener('click', () => this.generateTrackingNumber());
        }
    }

    generateTrackingNumber() {
        const trackingInput = document.getElementById('tracking_number');
        if (!trackingInput) return;

        const prefix = 'TRK';
        const randomNum = Math.floor(100000 + Math.random() * 900000);
        const trackingNumber = `${prefix}${randomNum}`;
        
        trackingInput.value = trackingNumber;

        const finalTrackingDisplay = document.getElementById('finalTrackingDisplay');
        if (finalTrackingDisplay) {
            finalTrackingDisplay.textContent = trackingNumber;
        }

        // Update provider tracking link if provider is selected
        const pricingManager = this.getModule('pricingManager');
        if (pricingManager && pricingManager.updateServiceProviderDetails) {
            pricingManager.updateServiceProviderDetails();
        }
    }

    async handleFormSubmit(e) {
        e.preventDefault();
        
        if (!this.validateForm()) {
            this.showAlert('Please complete all required fields', 'error');
            return;
        }

        this.showLoading(true);

        try {
            await this.submitForm();
        } catch (error) {
            console.error('Form submission error:', error);
            this.showAlert('Error submitting form. Please try again.', 'error');
        } finally {
            this.showLoading(false);
        }
    }

    validateForm() {
        // Basic validation - modules will handle specific section validation
        
        // Check if form exists
        if (!document.getElementById('shipmentForm')) {
            return false;
        }

        // Check OTP verification
        const otpVerified = document.getElementById('otpVerified') ? document.getElementById('otpVerified').value === '1' : false;
        if (!otpVerified) {
            this.showAlert('Please verify OTP before submission', 'error');
            return false;
        }

        return true;
    }

    async submitForm() {
        const form = document.getElementById('shipmentForm');
        const formData = new FormData(form);

        try {
            const response = await fetch(form.action, {
                method: form.method,
                body: formData,
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'X-CSRF-TOKEN': this.getCsrfToken()
                }
            });

            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            const result = await response.json();

            if (result.success) {
                this.showAlert('Shipment created successfully!', 'success');
                
                // Redirect after success
                setTimeout(() => {
                    window.location.href = result.redirect_url || '/shipments';
                }, 2000);
            } else {
                throw new Error(result.message || 'Submission failed');
            }
        } catch (error) {
            throw error;
        }
    }

    getCsrfToken() {
        const csrfMeta = document.querySelector('meta[name="csrf-token"]');
        return csrfMeta ? csrfMeta.getAttribute('content') : '';
    }

    showAlert(message, type = 'info') {
        // Try to use utilities module first
        const utilities = this.getModule('utilities');
        if (utilities && utilities.showAlert) {
            utilities.showAlert(message, type);
        } else {
            // Fallback to basic alert
            const alertClass = {
                'success': 'alert-success',
                'error': 'alert-danger',
                'warning': 'alert-warning',
                'info': 'alert-info'
            }[type] || 'alert-info';

            // Create simple alert
            const alertDiv = document.createElement('div');
            alertDiv.className = `alert ${alertClass} alert-dismissible fade show`;
            alertDiv.innerHTML = `
                ${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            `;

            const container = document.querySelector('.shipping-application');
            if (container) {
                container.insertBefore(alertDiv, container.firstChild);
                
                // Auto remove after 5 seconds
                setTimeout(() => {
                    if (alertDiv.parentElement) {
                        alertDiv.remove();
                    }
                }, 5000);
            }
        }
    }

    showLoading(show = true) {
        const utilities = this.getModule('utilities');
        if (utilities && utilities.showLoading) {
            utilities.showLoading(show);
        } else {
            const submitBtn = document.getElementById('finalSubmitBtn');
            if (submitBtn) {
                submitBtn.disabled = show;
                submitBtn.innerHTML = show ? 
                    '<i class="ri-loader-4-line ri-spin"></i> Processing...' : 
                    '<i class="ri-check-double-line"></i> Complete Shipment Booking';
            }
        }
    }

    getModule(moduleName) {
        return this.modules[moduleName];
    }
}

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    // Check if Bootstrap is available
    if (typeof bootstrap === 'undefined') {
        console.error('Bootstrap is not loaded. Please make sure Bootstrap is included on the page.');
        return;
    }

    // Initialize the shipment form
    window.shipmentForm = new ShipmentForm();
});