class VerificationManager {
    constructor(shipmentForm) {
        this.shipmentForm = shipmentForm;
        this.otpVerified = false;
    }

    init() {
        this.setupEventListeners();
        this.checkFinalSubmission();
    }

    setupEventListeners() {
        // OTP send button
        const sendOtpBtn = document.getElementById('sendOtpBtn');
        if (sendOtpBtn) {
            sendOtpBtn.addEventListener('click', () => this.sendOTP());
        }
        
        // OTP verification button
        const verifyOtpBtn = document.querySelector('button[onclick*="verifyOTP"]');
        if (verifyOtpBtn) {
            verifyOtpBtn.addEventListener('click', () => this.verifyOTP());
        }
        
        // Payment options
        document.querySelectorAll('.payment-option').forEach(option => {
            option.addEventListener('click', (e) => this.selectPaymentOption(e));
        });
    }

    async sendOTP() {
        const phone = document.getElementById('otp_phone') ? document.getElementById('otp_phone').value : '';
        if (!phone) {
            this.showAlert('Please enter a phone number for OTP', 'error');
            return;
        }

        // Show loading state
        const sendBtn = document.getElementById('sendOtpBtn');
        const originalText = sendBtn.innerHTML;
        sendBtn.innerHTML = '<i class="ri-loader-4-line ri-spin"></i> Sending...';
        sendBtn.disabled = true;

        try {
            const response = await fetch('/api/send-otp', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': this.getCsrfToken()
                },
                body: JSON.stringify({ 
                    phone: phone,
                    purpose: 'parcel_dropoff_verification'
                })
            });
            
            const data = await response.json();
            
            if (data.success) {
                const otpVerificationCode = document.getElementById('otpVerificationCode');
                if (otpVerificationCode) otpVerificationCode.value = data.otp_code;
                
                const otpInputSection = document.getElementById('otpInputSection');
                if (otpInputSection) otpInputSection.classList.remove('d-none');
                
                const otpStatus = document.getElementById('otpStatus');
                if (otpStatus) otpStatus.innerHTML = '<div class="alert alert-success">OTP sent successfully! Valid for 10 minutes.</div>';
                
                this.showAlert('OTP sent successfully!', 'success');
            } else {
                throw new Error(data.message || 'Failed to send OTP');
            }
        } catch (error) {
            this.showAlert('Error sending OTP: ' + error.message, 'error');
        } finally {
            // Restore button state
            sendBtn.innerHTML = originalText;
            sendBtn.disabled = false;
        }
    }

    verifyOTP() {
        const enteredOtp = document.getElementById('otp_code') ? document.getElementById('otp_code').value : '';
        const actualOtp = document.getElementById('otpVerificationCode') ? document.getElementById('otpVerificationCode').value : '';
        
        if (!enteredOtp) {
            const otpStatus = document.getElementById('otpStatus');
            if (otpStatus) otpStatus.innerHTML = '<div class="alert alert-warning">Please enter OTP</div>';
            return;
        }
        
        if (enteredOtp === actualOtp) {
            const otpVerifiedInput = document.getElementById('otpVerified');
            if (otpVerifiedInput) otpVerifiedInput.value = '1';
            
            const otpStatus = document.getElementById('otpStatus');
            if (otpStatus) otpStatus.innerHTML = '<div class="alert alert-success">OTP verified successfully!</div>';
            
            // Update status displays
            const statusDisplays = document.querySelectorAll('#otpStatusDisplay');
            statusDisplays.forEach(display => {
                if (display) {
                    display.textContent = 'Verified';
                    display.className = 'badge bg-success';
                }
            });
            
            this.otpVerified = true;
            this.checkFinalSubmission();
            this.showAlert('OTP verified successfully! Digital signature recorded.', 'success');
        } else {
            const otpStatus = document.getElementById('otpStatus');
            if (otpStatus) otpStatus.innerHTML = '<div class="alert alert-danger">Invalid OTP. Please try again.</div>';
            this.showAlert('Invalid OTP', 'error');
        }
    }

    selectPaymentOption(event) {
        document.querySelectorAll('.payment-option').forEach(opt => {
            opt.classList.remove('active');
        });
        event.currentTarget.classList.add('active');
        
        const paymentType = event.currentTarget.dataset.payment;
        const paymentDetails = document.getElementById('paymentDetails');
        
        if (paymentType === 'now') {
            if (paymentDetails) paymentDetails.classList.remove('d-none');
            const paymentStatusDisplay = document.getElementById('paymentStatusDisplay');
            if (paymentStatusDisplay) {
                paymentStatusDisplay.textContent = 'Pending';
                paymentStatusDisplay.className = 'badge bg-warning';
            }
        } else {
            if (paymentDetails) paymentDetails.classList.add('d-none');
            const paymentStatusDisplay = document.getElementById('paymentStatusDisplay');
            if (paymentStatusDisplay) {
                paymentStatusDisplay.textContent = 'On Delivery';
                paymentStatusDisplay.className = 'badge bg-info';
            }
        }
    }

    checkFinalSubmission() {
        const submitBtn = document.getElementById('finalSubmitBtn');
        if (submitBtn) {
            if (this.otpVerified) {
                submitBtn.disabled = false;
                submitBtn.innerHTML = '<i class="ri-check-double-line"></i> Complete Shipment Booking';
            } else {
                submitBtn.disabled = true;
                submitBtn.innerHTML = '<i class="ri-shield-keyhole-line"></i> Verify OTP First';
            }
        }
    }

    validateSection() {
        const stepManager = this.shipmentForm.getModule('stepManager');
        if (stepManager && stepManager.validateStep(6)) {
            if (this.otpVerified) {
                return true;
            } else {
                this.showAlert('Please verify OTP before final submission', 'error');
                return false;
            }
        }
        return false;
    }

    getCsrfToken() {
        const csrfMeta = document.querySelector('meta[name="csrf-token"]');
        return csrfMeta ? csrfMeta.getAttribute('content') : '';
    }

    showAlert(message, type = 'info') {
        if (window.shipmentForm) {
            window.shipmentForm.showAlert(message, type);
        } else {
            alert(message);
        }
    }
}