class StepManager {
    constructor(shipmentForm) {
        this.shipmentForm = shipmentForm;
        this.currentStep = 1;
        this.totalSteps = 6;
    }

    init() {
        this.setupStepNavigation();
        this.updateStepIndicator();
    }

    setupStepNavigation() {
        const accordions = document.querySelectorAll('.card-section .card-header button');
        accordions.forEach((accordion, index) => {
            accordion.addEventListener('click', () => {
                this.currentStep = index + 1;
                this.updateStepIndicator();
            });
        });
    }

    updateStepIndicator() {
        const indicator = document.getElementById('formStepIndicator');
        if (indicator) {
            indicator.textContent = `Step ${this.currentStep} of ${this.totalSteps}`;
        }
        
        const steps = ['customer', 'package', 'route', 'pricing', 'services', 'verification'];
        steps.forEach((step, index) => {
            const statusBadge = document.getElementById(`${step}StepStatus`);
            if (!statusBadge) return;
            
            if (index + 1 < this.currentStep) {
                statusBadge.className = 'badge bg-success ms-2';
                statusBadge.textContent = '✓';
            } else if (index + 1 === this.currentStep) {
                statusBadge.className = 'badge bg-primary ms-2';
                statusBadge.textContent = index + 1;
            } else {
                statusBadge.className = 'badge bg-secondary ms-2';
                statusBadge.textContent = index + 1;
            }
        });
    }

    moveToStep(stepNumber) {
        // Collapse current step
        const currentStepName = this.getStepName(this.currentStep);
        const currentCollapse = document.querySelector(`#${currentStepName}InfoCollapse`);
        if (currentCollapse) {
            currentCollapse.classList.remove('show');
        }

        // Expand target step
        const targetStepName = this.getStepName(stepNumber);
        const targetCollapse = document.querySelector(`#${targetStepName}InfoCollapse`);
        if (targetCollapse) {
            targetCollapse.classList.add('show');
        }

        this.currentStep = stepNumber;
        this.updateStepIndicator();
    }

    getStepName(stepNumber) {
        const steps = {
            1: 'customer',
            2: 'package',
            3: 'route',
            4: 'pricing',
            5: 'services',
            6: 'verification'
        };
        return steps[stepNumber] || '';
    }

    validateStep(stepNumber) {
        switch (stepNumber) {
            case 1:
                return this.validateCustomerStep();
            case 2:
                return this.validatePackageStep();
            case 3:
                return this.validateRouteStep();
            case 4:
                return this.validatePricingStep();
            case 5:
                return this.validateServicesStep();
            case 6:
                return this.validateVerificationStep();
            default:
                return true;
        }
    }

    validateCustomerStep() {
        const customerId = document.getElementById('customer_id').value;
        if (!customerId) {
            this.showAlert('Please select or create a customer', 'error');
            return false;
        }
        return true;
    }

    validatePackageStep() {
        const packageCount = document.getElementById('numPieces').value;
        if (!packageCount || parseInt(packageCount) === 0) {
            this.showAlert('Please enter the number of packages', 'error');
            return false;
        }
        return true;
    }

    validateRouteStep() {
        const origin = document.getElementById('origin_country_id').value;
        const destination = document.getElementById('destination_country_id').value;
        if (!origin || !destination) {
            this.showAlert('Please select origin and destination countries', 'error');
            return false;
        }
        return true;
    }

    validatePricingStep() {
        const providerCharges = document.getElementById('service_provider_charges').value;
        const myPrice = document.getElementById('my_price').value;
        if (!providerCharges || !myPrice || parseFloat(providerCharges) <= 0 || parseFloat(myPrice) <= 0) {
            this.showAlert('Please set service provider charges and your price', 'error');
            return false;
        }
        return true;
    }

    validateServicesStep() {
        return true; // Services are optional
    }

    validateVerificationStep() {
        const otpVerified = document.getElementById('otpVerified').value === '1';
        if (!otpVerified) {
            this.showAlert('Please verify OTP before proceeding', 'error');
            return false;
        }
        return true;
    }

    showAlert(message, type = 'error') {
        if (window.shipmentForm) {
            window.shipmentForm.showAlert(message, type);
        } else {
            alert(message);
        }
    }
}