class ServicesManager {
    constructor(shipmentForm) {
        this.shipmentForm = shipmentForm;
        this.services = [];
        this.baseCurrency = document.querySelector('input[name="base_currency"]') ? document.querySelector('input[name="base_currency"]').value : 'KES';
        this.serviceCounter = document.querySelectorAll('#servicesTable tbody tr:not(#noServicesRow)').length;
    }

    init() {
        this.setupEventListeners();
        this.calculateTotals(); // Initial calculation
    }

    setupEventListeners() {
        // Service selection change
        const serviceSelect = document.getElementById('service_id');
        if (serviceSelect) {
            serviceSelect.addEventListener('change', (e) => this.handleServiceSelection(e));
        }

        // Cost and charge inputs
        const costInput = document.getElementById('service_cost');
        const chargeInput = document.getElementById('service_charge');
        if (costInput) costInput.addEventListener('input', () => this.calculateProfit());
        if (chargeInput) chargeInput.addEventListener('input', () => this.calculateProfit());

        // Currency change
        const currencySelect = document.getElementById('service_currency');
        if (currencySelect) {
            currencySelect.addEventListener('change', () => this.calculateProfit());
        }

        // Add service button
        const addButton = document.getElementById('addService');
        if (addButton) {
            addButton.addEventListener('click', () => this.addService());
        }

        // Clear services button
        const clearButton = document.getElementById('clearServices');
        if (clearButton) {
            clearButton.addEventListener('click', () => this.clearAllServices());
        }

        // Remove service buttons (for existing services)
        document.querySelectorAll('.remove-service').forEach(button => {
            button.addEventListener('click', (e) => this.removeService(e));
        });

        // Enter key to add service
        document.addEventListener('keypress', (e) => {
            if (e.key === 'Enter' && e.target.closest('#servicesSection')) {
                this.addService();
            }
        });
    }

    handleServiceSelection(event) {
        const selectedOption = event.target.selectedOptions[0];
        const descriptionDiv = document.getElementById('serviceDescription');
        
        if (selectedOption && selectedOption.value) {
            const description = selectedOption.getAttribute('data-description');
            const defaultCost = selectedOption.getAttribute('data-default-cost');
            const defaultCharge = selectedOption.getAttribute('data-default-charge');
            const currency = selectedOption.getAttribute('data-currency');

            // Show description
            if (descriptionDiv) {
                descriptionDiv.textContent = description;
                descriptionDiv.classList.remove('d-none');
            }

            // Pre-fill cost and charge
            const costInput = document.getElementById('service_cost');
            const chargeInput = document.getElementById('service_charge');
            const currencySelect = document.getElementById('service_currency');

            if (costInput && defaultCost) costInput.value = defaultCost;
            if (chargeInput && defaultCharge) chargeInput.value = defaultCharge;
            if (currencySelect && currency) {
                currencySelect.value = currency;
            }

            // Calculate profit
            this.calculateProfit();
        } else {
            if (descriptionDiv) {
                descriptionDiv.classList.add('d-none');
            }
        }
    }

    calculateProfit() {
        const cost = parseFloat(document.getElementById('service_cost').value) || 0;
        const charge = parseFloat(document.getElementById('service_charge').value) || 0;
        const currency = document.getElementById('service_currency').value;
        const selectedOption = document.getElementById('service_id').selectedOptions[0];
        
        if (!selectedOption || !selectedOption.value) return;

        // Get exchange rates
        const exchangeRate = this.getExchangeRate(currency);
        const costKES = cost * exchangeRate;
        const chargeKES = charge * exchangeRate;
        const profit = chargeKES - costKES;

        // Update display
        const profitDisplay = document.getElementById('estimatedProfitDisplay');
        if (profitDisplay) {
            profitDisplay.innerHTML = `Estimated Profit: ${profit.toFixed(2)} ${this.baseCurrency}`;
            profitDisplay.classList.remove('d-none');
            profitDisplay.className = `alert ${profit >= 0 ? 'alert-success' : 'alert-danger'} py-1 px-2 mb-0`;
        }
    }

    getExchangeRate(currency) {
        if (currency === this.baseCurrency) return 1;
        
        const currencyOption = document.querySelector(`#service_currency option[value="${currency}"]`);
        if (currencyOption) {
            return parseFloat(currencyOption.getAttribute('data-sell-rate')) || 1;
        }
        return 1;
    }

    addService() {
        const serviceSelect = document.getElementById('service_id');
        const costInput = document.getElementById('service_cost');
        const chargeInput = document.getElementById('service_charge');
        const currencySelect = document.getElementById('service_currency');

        if (!serviceSelect.value) {
            this.showAlert('Please select a service', 'error');
            serviceSelect.focus();
            return;
        }

        if (!costInput.value || parseFloat(costInput.value) <= 0) {
            this.showAlert('Please enter a valid cost', 'error');
            costInput.focus();
            return;
        }

        if (!chargeInput.value || parseFloat(chargeInput.value) <= 0) {
            this.showAlert('Please enter a valid charge', 'error');
            chargeInput.focus();
            return;
        }

        const selectedOption = serviceSelect.selectedOptions[0];
        const serviceId = serviceSelect.value;
        const serviceName = selectedOption.text;
        const cost = parseFloat(costInput.value);
        const charge = parseFloat(chargeInput.value);
        const currency = currencySelect.value;
        const exchangeRate = this.getExchangeRate(currency);
        const costKES = cost * exchangeRate;
        const chargeKES = charge * exchangeRate;
        const profit = chargeKES - costKES;

        // Add to services array
        const service = {
            id: serviceId,
            name: serviceName,
            cost: cost,
            charge: charge,
            currency: currency,
            cost_kes: costKES,
            charge_kes: chargeKES,
            profit: profit
        };

        this.services.push(service);
        this.addServiceToTable(service);
        this.calculateTotals();
        this.clearServiceForm();

        this.showAlert('Service added successfully', 'success');
    }

    addServiceToTable(service) {
        const tableBody = document.querySelector('#servicesTable tbody');
        const noServicesRow = document.getElementById('noServicesRow');

        // Remove "no services" row if it exists
        if (noServicesRow) {
            noServicesRow.remove();
        }

        const newRow = document.createElement('tr');
        newRow.setAttribute('data-service-id', service.id);
        newRow.innerHTML = `
            <td>
                ${service.name}
                <input type="hidden" name="services[${this.serviceCounter}][id]" value="${service.id}">
            </td>
            <td>
                ${service.currency}
                <input type="hidden" name="services[${this.serviceCounter}][currency]" value="${service.currency}">
            </td>
            <td>
                ${service.cost.toFixed(2)}
                <input type="hidden" name="services[${this.serviceCounter}][cost]" value="${service.cost}">
            </td>
            <td>
                ${service.charge.toFixed(2)}
                <input type="hidden" name="services[${this.serviceCounter}][charge]" value="${service.charge}">
            </td>
            <td>${service.cost_kes.toFixed(2)}</td>
            <td>${service.charge_kes.toFixed(2)}</td>
            <td class="${service.profit >= 0 ? 'text-success' : 'text-danger'}">
                ${service.profit.toFixed(2)}
            </td>
            <td>
                <button type="button" class="btn btn-sm btn-danger remove-service" title="Remove">
                    <i class="ri-delete-bin-line"></i>
                </button>
            </td>
        `;

        tableBody.appendChild(newRow);

        // Add event listener to remove button
        newRow.querySelector('.remove-service').addEventListener('click', (e) => this.removeService(e));

        this.serviceCounter++;
    }

    removeService(event) {
        const button = event.target.closest('.remove-service');
        const row = button.closest('tr');
        const serviceId = row.getAttribute('data-service-id');

        // Remove from services array
        this.services = this.services.filter(service => service.id !== serviceId);

        // Remove from table
        row.remove();

        // Add "no services" row if table is empty
        const tableBody = document.querySelector('#servicesTable tbody');
        if (tableBody.children.length === 0) {
            const noServicesRow = document.createElement('tr');
            noServicesRow.id = 'noServicesRow';
            noServicesRow.innerHTML = `
                <td colspan="8" class="text-center py-4 text-muted">No services added yet</td>
            `;
            tableBody.appendChild(noServicesRow);
        }

        this.calculateTotals();
        this.showAlert('Service removed', 'info');
    }

    clearAllServices() {
        if (this.services.length === 0) return;

        if (!confirm('Are you sure you want to remove all services?')) {
            return;
        }

        this.services = [];
        const tableBody = document.querySelector('#servicesTable tbody');
        tableBody.innerHTML = `
            <tr id="noServicesRow">
                <td colspan="8" class="text-center py-4 text-muted">No services added yet</td>
            </tr>
        `;

        this.calculateTotals();
        this.serviceCounter = 0;
        this.showAlert('All services cleared', 'info');
    }

    clearServiceForm() {
        document.getElementById('service_id').value = '';
        document.getElementById('service_cost').value = '';
        document.getElementById('service_charge').value = '';
        document.getElementById('service_currency').value = this.baseCurrency;
        document.getElementById('serviceDescription').classList.add('d-none');
        document.getElementById('estimatedProfitDisplay').classList.add('d-none');
    }

    calculateTotals() {
        const totalCost = this.services.reduce((sum, service) => sum + service.cost, 0);
        const totalCharge = this.services.reduce((sum, service) => sum + service.charge, 0);
        const totalCostKES = this.services.reduce((sum, service) => sum + service.cost_kes, 0);
        const totalChargeKES = this.services.reduce((sum, service) => sum + service.charge_kes, 0);
        const totalProfit = totalChargeKES - totalCostKES;

        // Update footer
        const totalCostElement = document.getElementById('totalCost');
        const totalChargeElement = document.getElementById('totalCharge');
        const totalCostKESElement = document.getElementById('totalCostKES');
        const totalChargeKESElement = document.getElementById('totalChargeKES');
        const totalProfitElement = document.getElementById('totalProfit');

        if (totalCostElement) totalCostElement.textContent = totalCost.toFixed(2);
        if (totalChargeElement) totalChargeElement.textContent = totalCharge.toFixed(2);
        if (totalCostKESElement) totalCostKESElement.textContent = totalCostKES.toFixed(2);
        if (totalChargeKESElement) totalChargeKESElement.textContent = totalChargeKES.toFixed(2);
        if (totalProfitElement) {
            totalProfitElement.textContent = totalProfit.toFixed(2);
            totalProfitElement.className = totalProfit >= 0 ? 'text-success' : 'text-danger';
        }

        // Update services cost for pricing manager
        this.servicesCost = totalChargeKES;
        this.updateFinalAmount();
    }

    updateFinalAmount() {
        const pricingManager = this.shipmentForm.getModule('pricingManager');
        if (pricingManager && pricingManager.updateFinalAmount) {
            pricingManager.updateFinalAmount();
        }
    }

    validateSection() {
        // Services are optional, so always valid
        const stepManager = this.shipmentForm.getModule('stepManager');
        if (stepManager && stepManager.validateStep(5)) {
            stepManager.moveToStep(6);
            return true;
        }
        return false;
    }

    getServicesCost() {
        return this.servicesCost;
    }

    getServices() {
        return this.services;
    }

    showAlert(message, type = 'info') {
        if (window.shipmentForm && window.shipmentForm.showAlert) {
            window.shipmentForm.showAlert(message, type);
        } else {
            // Fallback alert
            const alertClass = {
                'success': 'alert-success',
                'error': 'alert-danger',
                'warning': 'alert-warning',
                'info': 'alert-info'
            }[type] || 'alert-info';

            alert(`${type.toUpperCase()}: ${message}`);
        }
    }
}