class RouteManager {
    constructor(shipmentForm) {
        this.shipmentForm = shipmentForm;
    }

    init() {
        this.setupEventListeners();
    }

    setupEventListeners() {
        // Route changes
        const originSelect = document.getElementById('origin_country_id');
        const destinationSelect = document.getElementById('destination_country_id');
        const modeSelect = document.getElementById('shipment_mode_select');
        const categorySelect = document.getElementById('cargo_category_id');
        
        if (originSelect) originSelect.addEventListener('change', () => this.calculateRouteRates());
        if (destinationSelect) destinationSelect.addEventListener('change', () => this.calculateRouteRates());
        if (modeSelect) modeSelect.addEventListener('change', () => this.calculateRouteRates());
        if (categorySelect) categorySelect.addEventListener('change', () => this.calculateRouteRates());
    }

    async calculateRouteRates() {
        const packageManager = this.shipmentForm.getModule('packageManager');
        const origin = document.getElementById('origin_country_id') ? document.getElementById('origin_country_id').value : '';
        const destination = document.getElementById('destination_country_id') ? document.getElementById('destination_country_id').value : '';
        const mode = document.getElementById('shipment_mode_select') ? document.getElementById('shipment_mode_select').value : '';
        const category = document.getElementById('cargo_category_id') ? document.getElementById('cargo_category_id').value : '';
        const weight = packageManager ? packageManager.getTotalWeight() : 0;
        const packages = packageManager ? packageManager.getPackageCount() : 0;
        
        if (!origin || !destination || !mode || !category || weight <= 0) {
            return;
        }
        
        try {
            const response = await fetch('/api/calculate-rates', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': this.getCsrfToken()
                },
                body: JSON.stringify({
                    origin_country_id: origin,
                    destination_country_id: destination,
                    shipment_mode: mode,
                    cargo_category_id: category,
                    weight: weight,
                    packages: packages
                })
            });
            
            const data = await response.json();
            
            if (data.success) {
                this.displayRates(data.rates);
            } else {
                this.displayRates([]);
            }
        } catch (error) {
            console.error('Error calculating rates:', error);
            this.displayRates([]);
        }
    }

    displayRates(rates) {
        const ratesContainer = document.getElementById('rateDisplay');
        if (!ratesContainer) return;
        
        ratesContainer.innerHTML = '';
        
        if (rates.length === 0) {
            ratesContainer.innerHTML = '<p class="text-muted">No rates available for this route</p>';
            return;
        }
        
        rates.forEach(rate => {
            const rateElement = document.createElement('div');
            rateElement.className = 'rate-option mb-2 p-2 border rounded';
            rateElement.innerHTML = `
                <div class="form-check">
                    <input class="form-check-input rate-radio" type="radio" name="selected_rate" 
                           value="${rate.id}" data-provider-charges="${rate.provider_charge}"
                           data-currency="${rate.currency}" data-exchange-rate="${rate.exchange_rate}">
                    <label class="form-check-label">
                        <strong>${rate.service_provider_name}</strong> - 
                        ${rate.currency} ${rate.provider_charge.toFixed(2)} 
                    </label>
                </div>
            `;
            ratesContainer.appendChild(rateElement);
        });
        
        // Add event listeners to rate radios
        document.querySelectorAll('.rate-radio').forEach(radio => {
            radio.addEventListener('change', (e) => this.selectRate(e.target));
        });
    }

    selectRate(radio) {
        const pricingManager = this.shipmentForm.getModule('pricingManager');
        if (pricingManager) {
            pricingManager.setProviderRate(
                parseFloat(radio.getAttribute('data-provider-charges')),
                radio.getAttribute('data-currency'),
                parseFloat(radio.getAttribute('data-exchange-rate'))
            );
        }
    }

    validateSection() {
        const stepManager = this.shipmentForm.getModule('stepManager');
        if (stepManager && stepManager.validateStep(3)) {
            stepManager.moveToStep(4);
        }
    }

    getCsrfToken() {
        const csrfMeta = document.querySelector('meta[name="csrf-token"]');
        return csrfMeta ? csrfMeta.getAttribute('content') : '';
    }
}