class PricingManager {
    constructor(shipmentForm) {
        this.shipmentForm = shipmentForm;
        this.providerCharges = 0;
        this.myPrice = 0;
        this.exchangeRate = 1;
        this.currency = document.querySelector('input[name="base_currency"]') ? document.querySelector('input[name="base_currency"]').value : 'USD';
    }

    init() {
        this.setupEventListeners();
    }

    setupEventListeners() {
        // Service provider selection
        const providerSelect = document.getElementById('preferred_service_provider_id');
        if (providerSelect) {
            providerSelect.addEventListener('change', () => this.updateServiceProviderDetails());
        }
        
        // Price inputs
        const myPriceInput = document.getElementById('my_price');
        if (myPriceInput) {
            myPriceInput.addEventListener('input', () => this.calculateProfit());
        }
    }

    setProviderRate(providerCharges, currency, exchangeRate) {
        this.providerCharges = providerCharges;
        this.currency = currency;
        this.exchangeRate = exchangeRate;
        
        const providerChargesInput = document.getElementById('service_provider_charges');
        const originalCurrencyInput = document.getElementById('originalCurrency');
        const exchangeRateInput = document.getElementById('exchangeRate');
        
        if (providerChargesInput) providerChargesInput.value = providerCharges.toFixed(2);
        if (originalCurrencyInput) originalCurrencyInput.value = currency;
        if (exchangeRateInput) exchangeRateInput.value = exchangeRate;
        
        this.calculateProfit();
        this.updateServiceProviderDetails();
    }

    updateServiceProviderDetails() {
        const providerId = document.getElementById('preferred_service_provider_id') ? document.getElementById('preferred_service_provider_id').value : '';
        const selectedOption = document.getElementById('preferred_service_provider_id') ? document.getElementById('preferred_service_provider_id').selectedOptions[0] : null;
        
        if (providerId && selectedOption) {
            const baseUrl = selectedOption.getAttribute('data-base-url');
            const trackingNumber = document.getElementById('tracking_number') ? document.getElementById('tracking_number').value : '';
            const providerName = selectedOption.text;
            
            if (baseUrl && trackingNumber) {
                const trackingLink = `${baseUrl}${trackingNumber}`;
                const trackingLinkInput = document.getElementById('provider_tracking_link');
                if (trackingLinkInput) trackingLinkInput.value = trackingLink;
            }
            
            const finalProviderDisplay = document.getElementById('finalProviderDisplay');
            if (finalProviderDisplay) finalProviderDisplay.textContent = providerName;
        }
    }

    calculateProfit() {
        this.myPrice = parseFloat(document.getElementById('my_price') ? document.getElementById('my_price').value : 0) || 0;
        const baseCharges = this.providerCharges * this.exchangeRate;
        const profit = this.myPrice - baseCharges;
        const profitMargin = baseCharges > 0 ? ((profit / baseCharges) * 100) : 0;
        
        this.updateDisplays(profit, profitMargin);
        this.updateFinalAmount();
    }

    updateDisplays(profit, profitMargin) {
        const profitDisplay = document.getElementById('profitDisplay');
        const profitAmountInput = document.getElementById('profitAmount');
        const marginDisplay = document.getElementById('marginDisplay');
        const finalProfitDisplay = document.getElementById('finalProfitDisplay');
        
        if (profitDisplay) profitDisplay.textContent = `${this.shipmentForm.baseCurrency} ${profit.toFixed(2)}`;
        if (profitAmountInput) profitAmountInput.value = profit.toFixed(2);
        
        if (marginDisplay) {
            marginDisplay.innerHTML = `<p class="mb-0">${profitMargin.toFixed(1)}%</p>`;
        }
        
        if (finalProfitDisplay) {
            finalProfitDisplay.textContent = profit.toFixed(2);
        }
    }

    updateFinalAmount() {
        const servicesManager = this.shipmentForm.getModule('servicesManager');
        const servicesCost = servicesManager ? servicesManager.getServicesCost() : 0;
        const totalAmount = this.myPrice + servicesCost;
        
        const finalAmountDisplay = document.getElementById('finalAmountDisplay');
        if (finalAmountDisplay) {
            finalAmountDisplay.textContent = totalAmount.toFixed(2);
        }
    }

    applyManualRate() {
        const manualCharges = parseFloat(document.getElementById('manual_service_provider_charges') ? document.getElementById('manual_service_provider_charges').value : 0) || 0;
        const manualCurrency = document.getElementById('manual_currency') ? document.getElementById('manual_currency').value : 'USD';
        const manualExchangeRate = parseFloat(document.getElementById('manual_exchange_rate') ? document.getElementById('manual_exchange_rate').value : 1) || 1;
        
        this.setProviderRate(manualCharges, manualCurrency, manualExchangeRate);
        
        const modal = bootstrap.Modal.getInstance(document.getElementById('manualRateModal'));
        if (modal) modal.hide();
        
        this.showAlert('Manual rate applied successfully', 'success');
    }

    validateSection() {
        const stepManager = this.shipmentForm.getModule('stepManager');
        if (stepManager && stepManager.validateStep(4)) {
            stepManager.moveToStep(5);
        }
    }

    showAlert(message, type = 'info') {
        if (window.shipmentForm) {
            window.shipmentForm.showAlert(message, type);
        } else {
            alert(message);
        }
    }
}