// public/js/shipment/modules/customer-manager.js

class CustomerManager {
    constructor(shipmentForm) {
        this.shipmentForm = shipmentForm;
        this.isCustomerSelected = false;
        this.urls = {
            search: window.CUSTOMER_SEARCH_URL || '/customer-ajax/search',
            store: window.CUSTOMER_STORE_URL || '/customer-ajax/store'
        };
    }

    init() {
        this.setupEventListeners();
        this.initializeFromExistingData();
        console.log('CustomerManager initialized with URLs:', this.urls);
    }

    setupEventListeners() {
        // Event listeners are now handled in the Blade template
    }

    initializeFromExistingData() {
        const customerIdInput = document.getElementById('customer_id');
        if (customerIdInput && customerIdInput.value) {
            this.isCustomerSelected = true;
            this.enableConfirmButton();
        }
    }

    async searchCustomerByPhone() {
        const phone = document.getElementById('customer_phone').value.trim();
        
        if (!phone) {
            this.showAlert('Please enter a phone number', 'error');
            return;
        }

        // Basic phone validation
        if (phone.length < 3) {
            this.showAlert('Please enter a valid phone number (at least 3 digits)', 'error');
            return;
        }

        try {
            this.showLoading(true);
            this.clearCustomerSelection();
            
            const searchUrl = `${this.urls.search}?phone=${encodeURIComponent(phone)}`;
            console.log('Searching customer with URL:', searchUrl);
            
            const response = await fetch(searchUrl, {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': this.getCsrfToken(),
                    'X-Requested-With': 'XMLHttpRequest'
                },
                credentials: 'same-origin'
            });

            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            const result = await response.json();
            console.log('Search result:', result);

            if (result.success && result.customer) {
                this.displayCustomerDetails(result.customer);
                this.isCustomerSelected = true;
                this.enableConfirmButton();
                this.hideNewCustomerNotice();
                this.showAlert(result.message || 'Customer found successfully!', 'success');
            } else {
                this.showNewCustomerNotice();
                this.isCustomerSelected = false;
                this.disableConfirmButton();
                this.showAlert(result.message || 'No customer found with this phone number', 'info');
            }
        } catch (error) {
            console.error('Error searching customer:', error);
            this.showAlert('Error searching for customer: ' + error.message, 'error');
            this.clearCustomerSelection();
        } finally {
            this.showLoading(false);
        }
    }

    displayCustomerDetails(customer) {
        const nameDisplay = document.getElementById('customerNameDisplay');
        const phoneDisplay = document.getElementById('customerPhoneDisplay');
        const emailDisplay = document.getElementById('customerEmailDisplay');
        const idDisplay = document.getElementById('customerIdDisplay');
        const addressDisplay = document.getElementById('customerAddressDisplay');
        const customerDetails = document.getElementById('customerDetails');
        const customerIdInput = document.getElementById('customer_id');
        
        if (nameDisplay) nameDisplay.textContent = customer.name || 'N/A';
        if (phoneDisplay) phoneDisplay.textContent = customer.phone;
        if (emailDisplay) emailDisplay.textContent = customer.email || 'N/A';
        if (idDisplay) idDisplay.textContent = customer.id_number || 'N/A';
        
        // Format address
        const addressParts = [];
        if (customer.address) addressParts.push(customer.address);
        if (customer.city) addressParts.push(customer.city);
        if (customer.country && customer.country.name) addressParts.push(customer.country.name);
        if (addressDisplay) addressDisplay.textContent = addressParts.join(', ') || 'N/A';
        
        if (customerDetails) customerDetails.classList.remove('d-none');
        if (customerIdInput) customerIdInput.value = customer.id;

        // Update verification fields
        this.updateVerificationFields(customer.phone);
        
        // Load customer stats
        this.loadCustomerStats(customer.id);
    }

    async loadCustomerStats(customerId) {
        try {
            // You can implement customer stats loading here
            const statsContainer = document.getElementById('customerStats');
            if (statsContainer) {
                statsContainer.innerHTML = `
                    <div class="row text-center">
                        <div class="col-4">
                            <small class="text-muted">Total Shipments</small>
                            <div class="fw-bold">${customer.total_shipments || 0}</div>
                        </div>
                        <div class="col-4">
                            <small class="text-muted">Total Revenue</small>
                            <div class="fw-bold">$${customer.total_revenue || 0}</div>
                        </div>
                        <div class="col-4">
                            <small class="text-muted">Status</small>
                            <div class="fw-bold text-capitalize">${customer.status || 'active'}</div>
                        </div>
                    </div>
                `;
            }
        } catch (error) {
            console.error('Error loading customer stats:', error);
        }
    }

    clearCustomerSelection() {
        const customerDetails = document.getElementById('customerDetails');
        const customerIdInput = document.getElementById('customer_id');
        const newCustomerNotice = document.getElementById('newCustomerNotice');
        
        if (customerDetails) customerDetails.classList.add('d-none');
        if (customerIdInput) customerIdInput.value = '';
        if (newCustomerNotice) newCustomerNotice.classList.add('d-none');
        
        this.isCustomerSelected = false;
        this.disableConfirmButton();
    }

    showNewCustomerNotice() {
        const newCustomerNotice = document.getElementById('newCustomerNotice');
        if (newCustomerNotice) newCustomerNotice.classList.remove('d-none');
    }

    hideNewCustomerNotice() {
        const newCustomerNotice = document.getElementById('newCustomerNotice');
        if (newCustomerNotice) newCustomerNotice.classList.add('d-none');
    }

    updateVerificationFields(phone) {
        const otpPhone = document.getElementById('otp_phone');
        const whatsappNumber = document.getElementById('whatsapp_number');
        
        if (otpPhone) otpPhone.value = phone;
        if (whatsappNumber) whatsappNumber.value = phone;
    }

    enableConfirmButton() {
        const confirmBtn = document.getElementById('confirmCustomerBtn');
        if (confirmBtn) {
            confirmBtn.disabled = false;
        }
    }

    disableConfirmButton() {
        const confirmBtn = document.getElementById('confirmCustomerBtn');
        if (confirmBtn) {
            confirmBtn.disabled = true;
        }
    }

    validateSection() {
        if (!this.isCustomerSelected) {
            this.showAlert('Please search and select a customer first', 'error');
            return false;
        }

        const stepManager = this.shipmentForm?.getModule('stepManager');
        if (stepManager && typeof stepManager.validateStep === 'function') {
            if (stepManager.validateStep(1)) {
                stepManager.moveToStep(2);
                return true;
            }
        } else {
            // Fallback: just mark as completed
            this.showAlert('Customer details confirmed!', 'success');
            return true;
        }
        
        return false;
    }

    async saveNewCustomer() {
        const formData = {
            name: document.getElementById('new_customer_name')?.value.trim() || '',
            phone: document.getElementById('new_customer_phone')?.value.trim() || '',
            id_number: document.getElementById('new_customer_id')?.value.trim() || '',
            email: document.getElementById('new_customer_email')?.value.trim() || '',
            address: document.getElementById('new_customer_address')?.value.trim() || '',
            city: document.getElementById('new_customer_city')?.value.trim() || '',
            country_id: document.getElementById('new_customer_country')?.value || '',
            customer_type: document.getElementById('new_customer_type')?.value || 'individual',
            company_name: document.getElementById('new_company_name')?.value || '',
            preferred_communication: document.getElementById('new_preferred_communication')?.value || 'whatsapp',
            status: 'active'
        };

        // Enhanced validation
        if (!formData.name || !formData.phone) {
            this.showAlert('Customer name and phone are required', 'error');
            return;
        }

        if (formData.phone.length < 3) {
            this.showAlert('Please enter a valid phone number', 'error');
            return;
        }

        try {
            const response = await fetch(this.urls.store, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': this.getCsrfToken(),
                    'X-Requested-With': 'XMLHttpRequest'
                },
                body: JSON.stringify(formData)
            });

            const result = await response.json();

            if (!response.ok) {
                throw new Error(result.message || `HTTP error! status: ${response.status}`);
            }

            if (result.success) {
                // Update phone input and search for the new customer
                document.getElementById('customer_phone').value = formData.phone;
                await this.searchCustomerByPhone();
                
                this.showAlert('Customer created successfully!', 'success');
                
                // Close modal
                const modal = bootstrap.Modal.getInstance(document.getElementById('newCustomerModal'));
                if (modal) modal.hide();
            } else {
                throw new Error(result.message || 'Failed to create customer');
            }
        } catch (error) {
            console.error('Error creating customer:', error);
            this.showAlert('Error creating customer: ' + error.message, 'error');
        }
    }

    showLoading(show) {
        const searchBtn = document.getElementById('searchCustomerBtn');
        if (searchBtn) {
            if (show) {
                searchBtn.innerHTML = '<i class="ri-loader-4-line ri-spin"></i> Searching...';
                searchBtn.disabled = true;
            } else {
                searchBtn.innerHTML = '<i class="ri-search-line"></i> Search';
                searchBtn.disabled = false;
            }
        }
    }

    getCsrfToken() {
        const csrfMeta = document.querySelector('meta[name="csrf-token"]');
        return csrfMeta ? csrfMeta.getAttribute('content') : '';
    }

    showAlert(message, type = 'info') {
        // Use the shipment form's alert system if available
        if (this.shipmentForm && typeof this.shipmentForm.showAlert === 'function') {
            this.shipmentForm.showAlert(message, type);
        } else {
            // Fallback alert system
            this.showFallbackAlert(message, type);
        }
    }

    showFallbackAlert(message, type = 'info') {
        const alertClass = {
            error: 'alert-danger',
            success: 'alert-success',
            info: 'alert-info',
            warning: 'alert-warning'
        }[type] || 'alert-info';

        // Create temporary alert
        const alertDiv = document.createElement('div');
        alertDiv.className = `alert ${alertClass} alert-dismissible fade show`;
        alertDiv.innerHTML = `
            ${message}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        `;
        
        const statusBar = document.querySelector('.system-status-bar');
        if (statusBar) {
            statusBar.parentNode.insertBefore(alertDiv, statusBar.nextSibling);
            
            // Auto remove after 5 seconds
            setTimeout(() => {
                if (alertDiv.parentNode) {
                    alertDiv.remove();
                }
            }, 5000);
        } else {
            // Ultimate fallback
            alert(message);
        }
    }
}

// Global function wrappers for backward compatibility
function searchCustomerByPhone() {
    if (window.shipmentForm && window.shipmentForm.getModule('customerManager')) {
        window.shipmentForm.getModule('customerManager').searchCustomerByPhone();
    } else {
        console.error('Customer manager not available');
        alert('System not ready. Please refresh the page.');
    }
}

function validateCustomerSection() {
    if (window.shipmentForm && window.shipmentForm.getModule('customerManager')) {
        return window.shipmentForm.getModule('customerManager').validateSection();
    } else {
        console.error('Customer manager not available');
        return false;
    }
}

function saveNewCustomer() {
    if (window.shipmentForm && window.shipmentForm.getModule('customerManager')) {
        window.shipmentForm.getModule('customerManager').saveNewCustomer();
    } else {
        console.error('Customer manager not available');
        alert('System not ready. Please refresh the page.');
    }
}

// Export for module system
if (typeof module !== 'undefined' && module.exports) {
    module.exports = CustomerManager;
}