<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('customers', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->string('phone')->unique();
            $table->string('email')->nullable();
            $table->string('id_number')->nullable()->comment('National ID/Passport');
            $table->text('address')->nullable();
            $table->string('city')->nullable();
            $table->foreignId('country_id')->nullable()->constrained()->onDelete('set null');
            $table->string('customer_type')->default('individual')->comment('individual, business, corporate');
            $table->string('company_name')->nullable()->comment('For business customers');
            $table->string('tax_number')->nullable()->comment('For business customers');
            $table->decimal('total_revenue', 15, 2)->default(0)->comment('Total revenue from all shipments');
            $table->integer('total_shipments')->default(0)->comment('Total number of shipments');
            $table->decimal('average_shipment_value', 10, 2)->default(0);
            $table->date('first_shipment_date')->nullable();
            $table->date('last_shipment_date')->nullable();
            $table->string('status')->default('active')->comment('active, inactive, blocked');
            $table->text('notes')->nullable();
            $table->string('preferred_communication')->default('whatsapp')->comment('whatsapp, email, sms');
            $table->json('metadata')->nullable()->comment('Additional customer data');
            $table->timestamps();
            $table->softDeletes();

            // Indexes for better performance
            $table->index('phone');
            $table->index('email');
            $table->index('customer_type');
            $table->index('status');
            $table->index('total_revenue');
            $table->index('created_at');
        });

        // Create customer_activities table for tracking
        Schema::create('customer_activities', function (Blueprint $table) {
            $table->id();
            $table->foreignId('customer_id')->constrained()->onDelete('cascade');
            $table->string('activity_type')->comment('shipment_created, payment_received, customer_updated, etc.');
            $table->text('description');
            $table->json('metadata')->nullable();
            $table->timestamp('performed_at');
            $table->foreignId('performed_by')->nullable()->constrained('users')->onDelete('set null');
            $table->timestamps();

            $table->index('customer_id');
            $table->index('activity_type');
            $table->index('performed_at');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('customer_activities');
        Schema::dropIfExists('customers');
    }
};