<?php

namespace App\Services;

use App\Models\Currency;

class PricingService
{
    protected $baseCurrency;
    protected $currentCurrency;
    protected $exchangeRate = 1;

    public function __construct()
    {
        $this->baseCurrency = Currency::base()->code;
        $this->currentCurrency = $this->baseCurrency;
    }

    /**
     * Convert amount to base currency
     */
    public function toBaseCurrency($amount, $fromCurrency)
    {
        return Currency::convertAmount($amount, $fromCurrency, $this->baseCurrency);
    }

    /**
     * Convert amount from base currency to target currency
     */
    public function fromBaseCurrency($amount, $toCurrency)
    {
        return Currency::convertAmount($amount, $this->baseCurrency, $toCurrency);
    }

    /**
     * Format amount with currency
     */
    public function format($amount, $currencyCode = null, $includeSymbol = true)
    {
        $currencyCode = $currencyCode ?: $this->currentCurrency;
        return Currency::formatAmount($amount, $currencyCode, $includeSymbol);
    }

    /**
     * Set current display currency
     */
    public function setDisplayCurrency($currencyCode)
    {
        if ($currencyCode !== $this->currentCurrency) {
            $this->exchangeRate = $this->getExchangeRate($currencyCode);
            $this->currentCurrency = $currencyCode;
        }
        return $this;
    }

    /**
     * Get exchange rate for currency
     */
    protected function getExchangeRate($currencyCode)
    {
        if ($currencyCode === $this->baseCurrency) {
            return 1;
        }

        $currency = Currency::where('code', $currencyCode)->firstOrFail();
        return 1 / $currency->selling_exchange_rate;
    }

    /**
     * Calculate total with proper currency handling
     */
    public function calculateTotal($freightAmount, $freightCurrency, $servicesAmount)
    {
        // Convert freight to base currency
        $freightInBase = $this->toBaseCurrency($freightAmount, $freightCurrency);
        
        // Services are already in base currency
        $totalInBase = $freightInBase + $servicesAmount;
        
        // Convert to current display currency
        return $totalInBase * $this->exchangeRate;
    }
}