<?php

namespace App\Repositories;

use App\Models\Vendors\ServiceProvider;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

class VendorRepository
{
    public function getPaginated(
        array $filters = [],
        string $sortBy = 'name',
        string $sortOrder = 'asc',
        int $perPage = 15,
        ?string $type = null
    ): LengthAwarePaginator {
        $query = ServiceProvider::query();

        // Apply filters
        if (!empty($filters['search'])) {
            $query->search($filters['search']);
        }

        if (!empty($filters['status'])) {
            $query->where('status', $filters['status']);
        }

        if ($type) {
            $query->where('type', $type);
        }

        // Apply sorting
        $validSortColumns = ['name', 'vendor_code', 'onboarding_date', 'open_balance', 'created_at'];
        $sortBy = in_array($sortBy, $validSortColumns) ? $sortBy : 'name';
        $sortOrder = $sortOrder === 'desc' ? 'desc' : 'asc';

        return $query->withCount(['routes', 'payments'])
            ->orderBy($sortBy, $sortOrder)
            ->paginate($perPage)
            ->appends($filters + ['sort_by' => $sortBy, 'sort_order' => $sortOrder]);
    }

    public function create(array $data): ServiceProvider
    {
        return DB::transaction(function () use ($data) {
            $vendor = ServiceProvider::create($data);

            if ($vendor->is_default) {
                $vendor->makeDefault();
            }

            // Create audit log
            activity()
                ->performedOn($vendor)
                ->withProperties($data)
                ->log('Vendor created');

            return $vendor->loadCount(['routes', 'payments']);
        });
    }

    public function update(ServiceProvider $vendor, array $data): bool
    {
        return DB::transaction(function () use ($vendor, $data) {
            $original = $vendor->getOriginal();

            $updated = $vendor->update($data);

            if ($updated) {
                // Create audit log
                activity()
                    ->performedOn($vendor)
                    ->withProperties([
                        'old' => $original,
                        'new' => $data
                    ])
                    ->log('Vendor updated');
            }

            return $updated;
        });
    }

    public function delete(ServiceProvider $vendor): bool
    {
        return DB::transaction(function () use ($vendor) {
            // Create audit log before deletion
            activity()
                ->performedOn($vendor)
                ->withProperties($vendor->toArray())
                ->log('Vendor deleted');

            return $vendor->delete();
        });
    }

    public function toggleStatus(ServiceProvider $vendor): string
    {
        $newStatus = $vendor->status === 'active' ? 'inactive' : 'active';
        
        $vendor->update(['status' => $newStatus]);

        activity()
            ->performedOn($vendor)
            ->withProperties(['new_status' => $newStatus])
            ->log('Vendor status changed');

        return $newStatus;
    }

    public function getDashboardStats(): array
    {
        return [
            'total_vendors' => ServiceProvider::count(),
            'active_vendors' => ServiceProvider::active()->count(),
            'total_service_providers' => ServiceProvider::serviceProviders()->count(),
            'total_suppliers' => ServiceProvider::suppliers()->count(),
            'total_balance' => ServiceProvider::sum('open_balance'),
            'recent_vendors' => ServiceProvider::latest()->limit(5)->get()
        ];
    }

    public function getForSelect(?string $type = null, ?string $search = null): Collection
    {
        $query = ServiceProvider::active();

        if ($type) {
            $query->where('type', $type);
        }

        if ($search) {
            $query->search($search);
        }

        return $query->select('id', 'name', 'vendor_code', 'type')
            ->orderBy('name')
            ->limit(50)
            ->get()
            ->map(function ($vendor) {
                return [
                    'id' => $vendor->id,
                    'text' => "{$vendor->name} ({$vendor->vendor_code})",
                    'vendor_code' => $vendor->vendor_code,
                    'type' => $vendor->type
                ];
            });
    }

    public function export(array $filters, string $format = 'csv')
    {
        $vendors = ServiceProvider::query();

        if (!empty($filters['type'])) {
            $vendors->where('type', $filters['type']);
        }

        if (!empty($filters['status'])) {
            $vendors->where('status', $filters['status']);
        }

        if (!empty($filters['start_date'])) {
            $vendors->whereDate('onboarding_date', '>=', $filters['start_date']);
        }

        if (!empty($filters['end_date'])) {
            $vendors->whereDate('onboarding_date', '<=', $filters['end_date']);
        }

        $vendors = $vendors->withCount(['routes', 'payments'])->get();

        if ($format === 'pdf') {
            return $this->generatePdfExport($vendors, $filters);
        }

        return $this->generateCsvExport($vendors, $filters);
    }

    private function generateCsvExport($vendors, $filters)
    {
        $fileName = 'vendors-export-' . now()->format('Y-m-d-H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$fileName}\"",
        ];

        $callback = function() use ($vendors, $filters) {
            $file = fopen('php://output', 'w');
            
            // Add BOM for UTF-8
            fwrite($file, "\xEF\xBB\xBF");
            
            // Headers
            fputcsv($file, [
                'Vendor Code', 'Name', 'Type', 'Status', 'Onboarding Date',
                'Balance', 'Contact Person', 'Email', 'Phone', 'Routes Count',
                'Payments Count', 'Address'
            ]);

            // Data
            foreach ($vendors as $vendor) {
                fputcsv($file, [
                    $vendor->vendor_code,
                    $vendor->name,
                    ucfirst($vendor->type),
                    ucfirst($vendor->status),
                    $vendor->onboarding_date->format('Y-m-d'),
                    $vendor->display_balance,
                    $vendor->contact_person,
                    $vendor->email,
                    $vendor->phone,
                    $vendor->routes_count,
                    $vendor->payments_count,
                    $vendor->address
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}