<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Customer extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'name',
        'phone',
        'email',
        'id_number',
        'address',
        'city',
        'country_id',
        'customer_type',
        'company_name',
        'tax_number',
        'total_revenue',
        'total_shipments',
        'average_shipment_value',
        'first_shipment_date',
        'last_shipment_date',
        'status',
        'notes',
        'preferred_communication',
        'metadata'
    ];

    protected $casts = [
        'total_revenue' => 'decimal:2',
        'average_shipment_value' => 'decimal:2',
        'first_shipment_date' => 'date',
        'last_shipment_date' => 'date',
        'metadata' => 'array'
    ];

    // Relationships
    public function country()
    {
        return $this->belongsTo(Country::class);
    }

    public function shipments()
    {
        return $this->hasMany(Shipment::class);
    }

    public function activities()
    {
        return $this->hasMany(CustomerActivity::class);
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    public function scopeBusiness($query)
    {
        return $query->where('customer_type', 'business');
    }

    public function scopeIndividual($query)
    {
        return $query->where('customer_type', 'individual');
    }

    public function scopeByPhone($query, $phone)
    {
        $cleanPhone = preg_replace('/\D+/', '', $phone);
        return $query->where('phone', 'LIKE', '%' . $phone . '%')
                    ->orWhereRaw('REPLACE(REPLACE(phone, " ", ""), "-", "") LIKE ?', ['%' . $cleanPhone . '%']);
    }

    public function scopeHighValue($query, $threshold = 1000)
    {
        return $query->where('total_revenue', '>=', $threshold);
    }

    public function scopeRecent($query, $days = 30)
    {
        return $query->where('created_at', '>=', now()->subDays($days));
    }

    // Methods
    public function updateRevenueStats()
    {
        $shipments = $this->shipments()->where('shipment_status', 'delivered')->get();
        
        $this->total_shipments = $shipments->count();
        $this->total_revenue = $shipments->sum('customer_amount');
        $this->average_shipment_value = $this->total_shipments > 0 ? $this->total_revenue / $this->total_shipments : 0;
        
        $firstShipment = $shipments->sortBy('created_at')->first();
        $lastShipment = $shipments->sortByDesc('created_at')->first();
        
        $this->first_shipment_date = $firstShipment ? $firstShipment->created_at->toDateString() : null;
        $this->last_shipment_date = $lastShipment ? $lastShipment->created_at->toDateString() : null;
        
        $this->save();
    }

    public function logActivity($activityType, $description, $metadata = [], $performedBy = null)
    {
        return $this->activities()->create([
            'activity_type' => $activityType,
            'description' => $description,
            'metadata' => $metadata,
            'performed_at' => now(),
            'performed_by' => $performedBy ?? auth()->id()
        ]);
    }

    public function getLifetimeValueAttribute()
    {
        return $this->total_revenue;
    }

    public function getAverageOrderValueAttribute()
    {
        return $this->average_shipment_value;
    }

    public function getFormattedAddressAttribute()
    {
        $parts = array_filter([$this->address, $this->city, $this->country?->name]);
        return implode(', ', $parts) ?: 'N/A';
    }
}