<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class Currency extends Model
{
    use HasFactory;

    protected $fillable = [
        'code',
        'name',
        'symbol',
        'symbol_position',
        'buying_exchange_rate',
        'selling_exchange_rate',
        'is_active'
    ];

    protected $casts = [
        'buying_exchange_rate' => 'decimal:6',
        'selling_exchange_rate' => 'decimal:6',
        'is_active' => 'boolean'
    ];

    public static function boot()
    {
        parent::boot();

        static::saved(function ($model) {
            Cache::tags(['currencies'])->flush();
        });

        static::deleted(function ($model) {
            Cache::tags(['currencies'])->flush();
        });
    }

    public function services()
    {
        return $this->hasMany(Service::class, 'default_currency_id');
    }

    public static function base(): self
    {
        return Cache::rememberForever('base_currency', function () {
            return static::where('code', config('app.currency', 'KES'))
                ->firstOrFail();
        });
    }

    public function isBaseCurrency(): bool
    {
        return $this->code === config('app.currency', 'KES');
    }

    public static function getExchangeRate(string $fromCurrency, string $toCurrency): float
    {
        $baseCurrency = config('app.currency', 'KES');
        $fromCurrency = strtoupper($fromCurrency);
        $toCurrency = strtoupper($toCurrency);

        if ($fromCurrency === $toCurrency) {
            return 1;
        }

        $cacheKey = "exchange_rate_{$fromCurrency}_{$toCurrency}";

        return Cache::tags(['currencies'])->remember($cacheKey, now()->addHour(), function() use ($fromCurrency, $toCurrency, $baseCurrency) {
            // If converting from base currency
            if ($fromCurrency === $baseCurrency) {
                $target = static::where('code', $toCurrency)->firstOrFail();
                return 1 / $target->selling_exchange_rate;
            }
            
            // If converting to base currency
            if ($toCurrency === $baseCurrency) {
                $source = static::where('code', $fromCurrency)->firstOrFail();
                return $source->selling_exchange_rate;
            }
            
            // Convert between two non-base currencies
            $source = static::where('code', $fromCurrency)->firstOrFail();
            $target = static::where('code', $toCurrency)->firstOrFail();
            
            return $source->selling_exchange_rate * (1 / $target->selling_exchange_rate);
        });
    }

    public static function formatAmount(float $amount, string $currencyCode, bool $includeSymbol = true): string
    {
        $currency = static::where('code', $currencyCode)->first();
        
        if (!$currency) {
            return number_format($amount, 2) . ' ' . $currencyCode;
        }

        $formatted = number_format($amount, 2, '.', ',');
        
        if ($includeSymbol && $currency->symbol) {
            return $currency->symbol_position === 'before' 
                ? $currency->symbol . $formatted
                : $formatted . $currency->symbol;
        }

        return $formatted . ' ' . $currency->code;
    }

    public static function activeCurrencies()
    {
        return Cache::tags(['currencies'])->remember('active_currencies', now()->addDay(), function() {
            return static::where('is_active', true)
                ->orderBy('code')
                ->get();
        });
    }
}