<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use App\Rules\ValidWeightForMode;

class UpdateShipmentRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function prepareForValidation(): void
    {
        $this->merge([
            'customer_phone' => $this->normalizePhone($this->customer_phone),
        ]);
    }

    protected function normalizePhone($phone): string
    {
        $phone = preg_replace('/\D/', '', $phone);
        
        if (Str::startsWith($phone, '+')) {
            return $phone;
        }
        
        if (Str::startsWith($phone, '0')) {
            return '+254' . substr($phone, 1);
        }
        
        if (Str::startsWith($phone, '254')) {
            return '+254' . substr($phone, 3);
        }
        
        return '+' . $phone;
    }

    public function rules(): array
    {
        return array_merge(
            $this->customerRules(),
            $this->shipmentRules(),
            $this->packageRules(),
            $this->serviceRules()
        );
    }

    protected function customerRules(): array
    {
        return [
            'customer_name' => 'required|string|max:255',
            'customer_phone' => 'required|string|max:20',
            'customer_email' => 'nullable|email|max:255',
            'customer_address' => 'required|string|max:255',
            'customer_national_id' => 'required|string|max:100',
        ];
    }

    protected function shipmentRules(): array
    {
        return [
            'origin_country_id' => 'required|exists:countries,id',
            'destination_country_id' => 'required|exists:countries,id',
            'mode' => 'required|in:standard_air,standard_sea,air_consolidation,sea_consolidation',
            'category_id' => 'required|exists:cargo_categories,id',
            'route_id' => 'required|exists:shipment_routes,id',
            'manual_rate' => 'nullable|numeric|min:0',
            'insurance_amount' => 'nullable|numeric|min:0',
            'insurance_currency' => 'required_with:insurance_amount|exists:currencies,code',
        ];
    }

    protected function packageRules(): array
    {
        return [
            'packages' => 'required|array|min:1|max:20',
            'packages.*.length' => 'required|numeric|min:0.1|max:500',
            'packages.*.width' => 'required|numeric|min:0.1|max:500',
            'packages.*.height' => 'required|numeric|min:0.1|max:500',
            'packages.*.weight' => [
                'required',
                'numeric',
                'min:0.01',
                'max:1000',
                new ValidWeightForMode($this->input('mode'))
            ],
            'packages.*.description' => 'nullable|string|max:255',
            'packages.*.origin_tracking' => [
                'nullable',
                'string',
                'max:255',
                Rule::unique('packages', 'origin_tracking_number')
                    ->ignore($this->route('shipment'))
            ],
            'packages.*.images' => 'nullable|array|max:3',
            'packages.*.images.*' => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
        ];
    }

    protected function serviceRules(): array
    {
        return [
            'services' => 'nullable|array',
            'services.*.service_id' => 'required|exists:services,id',
            'services.*.charge' => 'required|numeric|min:0',
            'services.*.currency' => 'required|string|exists:currencies,code',
        ];
    }
}