<?php

namespace App\Http\Controllers\Vendors;

use App\Http\Controllers\Controller;
use App\Models\SupplierPayment;
use App\Models\Vendors\ServiceProvider; // Correct namespace    
use Illuminate\Http\Request;
use Carbon\Carbon;

class SupplierPaymentController extends Controller
{
    /**
     * Display a listing of all supplier payments.
     */
    public function index()
    {
        $payments = SupplierPayment::with(['serviceProvider', 'createdBy'])
            ->orderBy('payment_date', 'desc')
            ->paginate(20);

        $totalPayments = SupplierPayment::sum('amount');
        $monthlyPayments = SupplierPayment::whereYear('payment_date', now()->year)
            ->whereMonth('payment_date', now()->month)
            ->sum('amount');

        return view('vendor.payments.index', compact('payments', 'totalPayments', 'monthlyPayments'));
    }

    /**
     * Display the specified payment.
     */
    public function show(SupplierPayment $supplierPayment)
    {
        $supplierPayment->load(['serviceProvider', 'createdBy']);
        return view('vendor.payments.show', compact('supplierPayment'));
    }

    /**
     * Show the form for editing the specified payment.
     */
    public function edit(SupplierPayment $supplierPayment)
    {
        $serviceProviders = ServiceProvider::where('status', 'active')->get();
        return view('vendor.payments.edit', compact('supplierPayment', 'serviceProviders'));
    }

    /**
     * Update the specified payment.
     */
    public function update(Request $request, SupplierPayment $supplierPayment)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0',
            'payment_date' => 'required|date',
            'payment_method' => 'required|in:bank_transfer,cash,cheque,online',
            'reference_number' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
        ]);

        try {
            // Calculate balance difference
            $amountDifference = $validated['amount'] - $supplierPayment->amount;

            $supplierPayment->update($validated);

            // Update service provider balance
            if ($amountDifference != 0) {
                $serviceProvider = $supplierPayment->serviceProvider;
                $serviceProvider->update([
                    'open_balance' => $serviceProvider->open_balance - $amountDifference
                ]);
            }

            return redirect()->route('supplier-payments.index')
                ->with('success', 'Payment updated successfully!');

        } catch (\Exception $e) {
            return redirect()->back()->withInput()->with('error', 'Error updating payment: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified payment.
     */
    public function destroy(SupplierPayment $supplierPayment)
    {
        DB::beginTransaction();
        try {
            // Restore service provider balance
            $serviceProvider = $supplierPayment->serviceProvider;
            $serviceProvider->update([
                'open_balance' => $serviceProvider->open_balance + $supplierPayment->amount
            ]);

            $supplierPayment->delete();
            DB::commit();

            return redirect()->route('supplier-payments.index')
                ->with('success', 'Payment deleted successfully!');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error deleting payment: ' . $e->getMessage());
        }
    }

    /**
     * Export payments to CSV or PDF
     */
    public function export(Request $request)
    {
        $validated = $request->validate([
            'format' => 'required|in:csv,pdf',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'service_provider_id' => 'nullable|exists:service_providers,id',
        ]);

        $payments = SupplierPayment::with('serviceProvider')
            ->when($validated['start_date'], function($query) use ($validated) {
                $query->where('payment_date', '>=', $validated['start_date']);
            })
            ->when($validated['end_date'], function($query) use ($validated) {
                $query->where('payment_date', '<=', $validated['end_date']);
            })
            ->when($validated['service_provider_id'], function($query) use ($validated) {
                $query->where('service_provider_id', $validated['service_provider_id']);
            })
            ->orderBy('payment_date', 'desc')
            ->get();

        if ($validated['format'] === 'csv') {
            return $this->exportToCsv($payments);
        }

        return $this->exportToPdf($payments);
    }

    private function exportToCsv($payments)
    {
        $fileName = 'supplier-payments-' . date('Y-m-d') . '.csv';
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $fileName . '"',
        ];

        $callback = function() use ($payments) {
            $file = fopen('php://output', 'w');
            
            // Add CSV headers
            fputcsv($file, [
                'Payment Date', 'Vendor', 'Vendor Code', 'Amount', 'Payment Method',
                'Reference Number', 'Notes'
            ]);

            // Add data rows
            foreach ($payments as $payment) {
                fputcsv($file, [
                    $payment->payment_date->format('Y-m-d'),
                    $payment->serviceProvider->name,
                    $payment->serviceProvider->vendor_code,
                    $payment->amount,
                    ucfirst(str_replace('_', ' ', $payment->payment_method)),
                    $payment->reference_number,
                    $payment->notes
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    private function exportToPdf($payments)
    {
        // You'll need to implement PDF generation using DomPDF or similar
        // This is a placeholder implementation
        return response()->json(['message' => 'PDF export not implemented yet']);
    }
}