<?php

namespace App\Http\Controllers\Vendors;

use App\Http\Controllers\Controller;
use App\Models\Vendors\ServiceProvider; // Correct namespace
use App\Models\Vendors\ServiceProviderRoute;
use App\Models\Shipment; // Assuming you have a Shipment model
use Illuminate\Http\Request;
use Carbon\Carbon;

class ServiceProviderReportController extends Controller
{
    /**
     * Display reports index page.
     */
    public function index()
    {
        return view('vendor.reports.index');
    }

    /**
     * Generate service provider performance report.
     */
    public function generate(Request $request)
    {
        $validated = $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'service_provider_id' => 'nullable|exists:service_providers,id',
            'report_type' => 'required|in:performance,payments,utilization,route_analysis'
        ]);

        $reportData = $this->generateReportData($validated);

        if ($request->wantsJson()) {
            return response()->json([
                'success' => true,
                'report' => $reportData
            ]);
        }

        return view('vendor.reports.generate', compact('reportData', 'validated'));
    }

    /**
     * Supplier payments report.
     */
    public function payments(Request $request)
    {
        $validated = $request->validate([
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'service_provider_id' => 'nullable|exists:service_providers,id',
        ]);

        $payments = SupplierPayment::with('serviceProvider')
            ->when($validated['start_date'] ?? false, function($query) use ($validated) {
                $query->where('payment_date', '>=', $validated['start_date']);
            })
            ->when($validated['end_date'] ?? false, function($query) use ($validated) {
                $query->where('payment_date', '<=', $validated['end_date']);
            })
            ->when($validated['service_provider_id'] ?? false, function($query) use ($validated) {
                $query->where('service_provider_id', $validated['service_provider_id']);
            })
            ->orderBy('payment_date', 'desc')
            ->paginate(20);

        $summary = [
            'total_payments' => $payments->sum('amount'),
            'payment_count' => $payments->total(),
            'average_payment' => $payments->average('amount'),
        ];

        return view('vendor.reports.payments', compact('payments', 'summary', 'validated'));
    }

    /**
     * Route performance report.
     */
    public function routePerformance(Request $request)
    {
        $validated = $request->validate([
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        // This would require integration with your Shipment model
        // Placeholder implementation
        $routePerformance = [];

        return view('vendor.reports.route-performance', compact('routePerformance', 'validated'));
    }

    /**
     * Generate comprehensive report data.
     */
    private function generateReportData($parameters)
    {
        $startDate = Carbon::parse($parameters['start_date']);
        $endDate = Carbon::parse($parameters['end_date']);

        $query = ServiceProvider::with(['routes', 'payments' => function($query) use ($startDate, $endDate) {
            $query->whereBetween('payment_date', [$startDate, $endDate]);
        }]);

        if ($parameters['service_provider_id']) {
            $query->where('id', $parameters['service_provider_id']);
        }

        $serviceProviders = $query->get();

        $reportData = [
            'period' => [
                'start_date' => $startDate->format('M d, Y'),
                'end_date' => $endDate->format('M d, Y'),
                'days' => $startDate->diffInDays($endDate),
            ],
            'service_providers' => [],
            'summary' => [
                'total_providers' => $serviceProviders->count(),
                'total_payments' => 0,
                'average_balance' => 0,
                'active_routes' => 0,
            ]
        ];

        foreach ($serviceProviders as $provider) {
            $providerData = [
                'id' => $provider->id,
                'name' => $provider->name,
                'vendor_code' => $provider->vendor_code,
                'type' => $provider->type,
                'status' => $provider->status,
                'open_balance' => $provider->open_balance,
                'total_payments' => $provider->payments->sum('amount'),
                'payment_count' => $provider->payments->count(),
                'route_count' => $provider->routes->where('is_active', true)->count(),
                'last_payment_date' => $provider->payments->max('payment_date'),
            ];

            $reportData['service_providers'][] = $providerData;
            $reportData['summary']['total_payments'] += $providerData['total_payments'];
            $reportData['summary']['active_routes'] += $providerData['route_count'];
        }

        if ($serviceProviders->count() > 0) {
            $reportData['summary']['average_balance'] = $serviceProviders->avg('open_balance');
        }

        return $reportData;
    }
}