<?php

namespace App\Http\Controllers;

use App\Services\ShipmentRateCalculator;
use App\Models\Country;
use App\Models\ShipmentRoute;
use App\Models\Currency;
use App\Models\CargoCategory;
use Illuminate\Http\Request;

class ShipmentFormController extends Controller
{
    protected $rateCalculator;
    
    public function __construct(ShipmentRateCalculator $rateCalculator)
    {
        $this->rateCalculator = $rateCalculator;
    }

    /**
     * Display the shipment form
     */
    public function showForm()
    {
        return view('shipments.form', [
            'countries' => Country::all(),
            'currencies' => Currency::all(),
            'cargoCategories' => CargoCategory::all(),
        ]);
    }

    /**
     * Calculate shipment charges
     */
    public function calculateChargeable(Request $request)
    {
        $data = $request->validate([
            'origin_country_id' => 'required|exists:countries,id',
            'destination_country_id' => 'required|exists:countries,id',
            'mode' => 'required|string',
            'weight' => 'required|numeric|min:0',
            'length' => 'nullable|numeric|min:0',
            'width' => 'nullable|numeric|min:0',
            'height' => 'nullable|numeric|min:0',
            'cargo_category_id' => 'nullable|exists:cargo_categories,id',
        ]);

        try {
            $result = $this->rateCalculator->calculateRate($data);
            
            return response()->json([
                'success' => true,
                'data' => $result,
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to calculate rates: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get route rates without weight calculations
     */
    public function getRouteRates(Request $request)
    {
        $data = $request->validate([
            'origin_country_id' => 'required|exists:countries,id',
            'destination_country_id' => 'required|exists:countries,id',
            'mode' => 'required|string',
            'cargo_category_id' => 'nullable|exists:cargo_categories,id',
        ]);

        try {
            // We'll use the same service but with empty weight/dimensions
            $result = $this->rateCalculator->calculateRate(array_merge($data, [
                'weight' => 0,
                'length' => 0,
                'width' => 0,
                'height' => 0,
            ]));

            return response()->json([
                'success' => true,
                'exists' => $result['route_found'],
                'volumetric_divisor' => $result['volumetric_divisor'],
                'currency' => $result['currency'],
                'applicable_rate' => $result['rate'],
                'rate_source' => $result['rate_source'],
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch route rates: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Find shipment route details
     */
    public function findRoute(Request $request)
    {
        $validated = $request->validate([
            'origin_country_id' => 'required|integer|exists:countries,id',
            'destination_country_id' => 'required|integer|exists:countries,id',
            'mode' => 'required|string',
            'cargo_category_id' => 'required|integer|exists:cargo_categories,id',
        ]);

        try {
            // Again using our service for consistency
            $result = $this->rateCalculator->calculateRate(array_merge($validated, [
                'weight' => 0,
                'length' => 0,
                'width' => 0,
                'height' => 0,
            ]));

            return response()->json([
                'success' => true,
                'rate_per_unit' => $result['rate'],
                'currency' => $result['currency'],
                'volumetric_divisor' => $result['volumetric_divisor'],
                'is_minimum_charge' => $result['is_minimum_charge'],
                'rate_source' => $result['rate_source'],
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Route not found: ' . $e->getMessage(),
                'volumetric_divisor' => 6000 // Default value
            ], 404);
        }
    }

    /**
     * Get breakdown of all available rates for a route
     */
    public function breakdown(Request $request)
    {
        $validated = $request->validate([
            'origin_country_id' => 'required|integer|exists:countries,id',
            'destination_country_id' => 'required|integer|exists:countries,id',
            'mode' => 'required|string',
            'cargo_category_id' => 'required|integer|exists:cargo_categories,id',
        ]);

        $routes = ShipmentRoute::with(['currency', 'cargoCategory'])
            ->where([
                'origin_country_id' => $validated['origin_country_id'],
                'destination_country_id' => $validated['destination_country_id'],
                'cargo_category_id' => $validated['cargo_category_id'],
            ])
            ->whereHas('shipmentMode', function($query) use ($validated) {
                $query->where('code', $validated['mode']);
            })
            ->orderBy('minimum_weight')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $routes->map(function($route) {
                return [
                    'id' => $route->id,
                    'rate_per_unit' => $route->rate_per_unit,
                    'flat_rate' => $route->flat_rate,
                    'minimum_weight' => $route->minimum_weight,
                    'currency' => $route->currency->code,
                    'volumetric_divisor' => $route->volumetric_divisor,
                    'created_at' => $route->created_at,
                    'updated_at' => $route->updated_at,
                ];
            })
        ]);
    }
}