<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Models\CustomerActivity;
use Illuminate\Http\Request;
use Carbon\Carbon;

class CustomerReportController extends Controller
{
    public function index()
    {
        return view('customers.reports.index');
    }

    public function revenueReport(Request $request)
    {
        $startDate = $request->get('start_date', now()->subYear()->format('Y-m-d'));
        $endDate = $request->get('end_date', now()->format('Y-m-d'));
        $customerType = $request->get('customer_type');

        $query = Customer::query();

        if ($customerType) {
            $query->where('customer_type', $customerType);
        }

        $customers = $query->withCount(['shipments as recent_shipments_count' => function($query) use ($startDate, $endDate) {
                $query->whereBetween('created_at', [$startDate, $endDate]);
            }])
            ->withSum(['shipments as recent_revenue' => function($query) use ($startDate, $endDate) {
                $query->whereBetween('created_at', [$startDate, $endDate]);
            }], 'customer_amount')
            ->orderBy('recent_revenue', 'desc')
            ->get();

        return view('customers.reports.revenue', compact('customers', 'startDate', 'endDate'));
    }

    public function customerActivity(Request $request)
    {
        $customerId = $request->get('customer_id');
        $days = $request->get('days', 30);

        $query = Customer::withCount(['activities as activities_count' => function($query) use ($days) {
                $query->where('performed_at', '>=', now()->subDays($days));
            }])
            ->with(['activities' => function($query) use ($days) {
                $query->where('performed_at', '>=', now()->subDays($days))
                      ->orderBy('performed_at', 'desc');
            }]);

        if ($customerId) {
            $query->where('id', $customerId);
        }

        $customers = $query->orderBy('activities_count', 'desc')->get();

        return view('customers.reports.activity', compact('customers', 'days'));
    }

    public function highValueCustomers()
    {
        $customers = Customer::where('total_revenue', '>=', 500)
            ->orderBy('total_revenue', 'desc')
            ->get();

        return view('customers.reports.high-value', compact('customers'));
    }

    public function customerLifetimeValue()
    {
        $customers = Customer::withCount('shipments')
            ->orderBy('total_revenue', 'desc')
            ->get();

        $stats = [
            'total_customers' => $customers->count(),
            'total_revenue' => $customers->sum('total_revenue'),
            'average_lifetime_value' => $customers->avg('total_revenue'),
            'top_customer' => $customers->first()
        ];

        return view('customers.reports.lifetime-value', compact('customers', 'stats'));
    }

    // Optional: Export methods
    public function exportRevenueReport(Request $request)
    {
        // Implement CSV/Excel export for revenue report
    }

    public function exportActivityReport(Request $request)
    {
        // Implement CSV/Excel export for activity report
    }
}