<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Models\Country;
use App\Models\CustomerActivity;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\DB;

class CustomerController extends Controller
{
    /**
     * Display a listing of customers with filtering and pagination
     */
    public function index(Request $request)
    {
        $query = Customer::query();

        // Apply filters
        if ($request->has('customer_type') && $request->customer_type) {
            $query->where('customer_type', $request->customer_type);
        }

        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'LIKE', "%{$search}%")
                  ->orWhere('phone', 'LIKE', "%{$search}%")
                  ->orWhere('email', 'LIKE', "%{$search}%")
                  ->orWhere('company_name', 'LIKE', "%{$search}%")
                  ->orWhere('id_number', 'LIKE', "%{$search}%");
            });
        }

        // Sort options
        $sort = $request->get('sort', 'total_revenue');
        $direction = $request->get('direction', 'desc');
        
        $validSorts = ['name', 'phone', 'total_revenue', 'total_shipments', 'created_at', 'last_shipment_date'];
        if (in_array($sort, $validSorts)) {
            $query->orderBy($sort, $direction);
        } else {
            $query->orderBy('total_revenue', 'desc');
        }

        $customers = $query->paginate(20)->withQueryString();

        return view('customers.index', compact('customers'));
    }

    /**
     * Show the form for creating a new customer
     */
    public function create()
    {
        $countries = Country::orderBy('name')->get();
        return view('customers.create', compact('countries'));
    }

    /**
     * Store a newly created customer
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20|unique:customers',
            'email' => 'nullable|email|max:255',
            'id_number' => 'nullable|string|max:50',
            'address' => 'nullable|string|max:500',
            'city' => 'nullable|string|max:100',
            'country_id' => 'nullable|exists:countries,id',
            'customer_type' => 'required|in:individual,business,corporate',
            'company_name' => 'nullable|string|max:255',
            'tax_number' => 'nullable|string|max:50',
            'preferred_communication' => 'required|in:whatsapp,email,sms',
            'status' => 'required|in:active,inactive,blocked',
            'notes' => 'nullable|string'
        ]);

        DB::transaction(function () use ($validated) {
            $customer = Customer::create($validated);
            
            // Log the customer creation activity
            $customer->logActivity(
                'customer_created',
                "Customer {$customer->name} was created",
                ['phone' => $customer->phone, 'type' => $customer->customer_type]
            );
        });

        return redirect()->route('customers.index')
                        ->with('success', 'Customer created successfully!');
    }

    /**
     * Display the specified customer
     */
    public function show(Customer $customer)
    {
        $customer->load(['shipments' => function($query) {
            $query->orderBy('created_at', 'desc')->limit(10);
        }, 'activities' => function($query) {
            $query->orderBy('performed_at', 'desc')->limit(10);
        }]);

        return view('customers.show', compact('customer'));
    }

    /**
     * Show the form for editing the specified customer
     */
    public function edit(Customer $customer)
    {
        $countries = Country::orderBy('name')->get();
        return view('customers.edit', compact('customer', 'countries'));
    }

    /**
     * Update the specified customer
     */
    public function update(Request $request, Customer $customer)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20|unique:customers,phone,' . $customer->id,
            'email' => 'nullable|email|max:255',
            'id_number' => 'nullable|string|max:50',
            'address' => 'nullable|string|max:500',
            'city' => 'nullable|string|max:100',
            'country_id' => 'nullable|exists:countries,id',
            'customer_type' => 'required|in:individual,business,corporate',
            'company_name' => 'nullable|string|max:255',
            'tax_number' => 'nullable|string|max:50',
            'preferred_communication' => 'required|in:whatsapp,email,sms',
            'status' => 'required|in:active,inactive,blocked',
            'notes' => 'nullable|string'
        ]);

        DB::transaction(function () use ($customer, $validated) {
            $oldData = $customer->getOriginal();
            $customer->update($validated);
            
            // Log the customer update activity
            $changes = [];
            foreach ($validated as $field => $value) {
                if ($oldData[$field] != $value) {
                    $changes[$field] = [
                        'from' => $oldData[$field],
                        'to' => $value
                    ];
                }
            }
            
            if (!empty($changes)) {
                $customer->logActivity(
                    'customer_updated',
                    "Customer {$customer->name} was updated",
                    ['changes' => $changes]
                );
            }
        });

        return redirect()->route('customers.index')
                        ->with('success', 'Customer updated successfully!');
    }

    /**
     * Remove the specified customer
     */
    public function destroy(Customer $customer)
    {
        DB::transaction(function () use ($customer) {
            // Log the deletion activity before deleting
            $customer->logActivity(
                'customer_deleted',
                "Customer {$customer->name} was deleted",
                ['phone' => $customer->phone, 'total_revenue' => $customer->total_revenue]
            );
            
            $customer->delete();
        });

        return redirect()->route('customers.index')
                        ->with('success', 'Customer deleted successfully!');
    }

    /**
     * API: Search customer by phone number
     */
    public function searchByPhone(Request $request): JsonResponse
    {
        try {
            $phone = $request->get('phone');
            
            Log::info('Customer search initiated', ['phone' => $phone, 'ip' => $request->ip()]);
            
            if (!$phone) {
                return response()->json([
                    'success' => false,
                    'message' => 'Phone number is required'
                ], 400);
            }

            // Validate phone format
            if (strlen($phone) < 3) {
                return response()->json([
                    'success' => false,
                    'message' => 'Please enter a valid phone number (at least 3 digits)'
                ], 400);
            }

            // Clean phone number for search
            $cleanPhone = preg_replace('/\D+/', '', $phone);
            
            $customer = Customer::where(function($query) use ($phone, $cleanPhone) {
                    $query->where('phone', 'LIKE', '%' . $phone . '%')
                          ->orWhereRaw('REPLACE(REPLACE(phone, " ", ""), "-", "") LIKE ?', ['%' . $cleanPhone . '%']);
                })
                ->with(['country'])
                ->first();

            Log::info('Customer search completed', [
                'phone' => $phone, 
                'found' => !is_null($customer),
                'customer_id' => $customer?->id
            ]);

            if ($customer) {
                // Log the search activity
                $customer->logActivity(
                    'customer_searched',
                    "Customer was searched via shipment form",
                    ['search_phone' => $phone, 'ip_address' => $request->ip()]
                );
            }

            return response()->json([
                'success' => true,
                'customer' => $customer,
                'message' => $customer ? 'Customer found successfully!' : 'No customer found with this phone number'
            ]);

        } catch (\Exception $e) {
            Log::error('Customer search failed', [
                'phone' => $request->get('phone'),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Server error while searching for customer'
            ], 500);
        }
    }
    /**
     * API: Store a new customer via AJAX
     */
    public function storeApi(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'phone' => 'required|string|max:20|unique:customers,phone',
                'id_number' => 'nullable|string|max:50',
                'email' => 'nullable|email|max:255',
                'address' => 'nullable|string|max:500',
                'city' => 'nullable|string|max:100',
                'country_id' => 'nullable|exists:countries,id',
                'customer_type' => 'required|in:individual,business,corporate',
                'company_name' => 'nullable|string|max:255',
                'preferred_communication' => 'required|in:whatsapp,email,sms'
            ]);

            $customer = DB::transaction(function () use ($validated) {
                $customer = Customer::create($validated);
                
                // Log the customer creation activity
                $customer->logActivity(
                    'customer_created',
                    "Customer {$customer->name} was created via shipment form",
                    ['phone' => $customer->phone, 'type' => $customer->customer_type, 'source' => 'shipment_form']
                );
                
                return $customer;
            });

            return response()->json([
                'success' => true,
                'customer' => $customer->load('country'),
                'message' => 'Customer created successfully'
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create customer: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * API: Get customer statistics
     */
    public function getCustomerStats(Customer $customer): JsonResponse
    {
        $customer->loadCount(['shipments', 'activities']);
        
        return response()->json([
            'success' => true,
            'stats' => [
                'total_revenue' => (float) $customer->total_revenue,
                'total_shipments' => $customer->total_shipments,
                'average_shipment_value' => (float) $customer->average_shipment_value,
                'first_shipment_date' => $customer->first_shipment_date?->toDateString(),
                'last_shipment_date' => $customer->last_shipment_date?->toDateString(),
                'activities_count' => $customer->activities_count,
                'customer_since' => $customer->created_at->toDateString(),
                'lifetime_value' => (float) $customer->total_revenue,
            ]
        ]);
    }

    /**
     * API: Get top customers
     */
    public function getTopCustomers(Request $request): JsonResponse
    {
        $limit = $request->get('limit', 10);
        $period = $request->get('period', 'all'); // all, month, year
        
        $query = Customer::query();
        
        if ($period === 'month') {
            $query->where('last_shipment_date', '>=', now()->subMonth());
        } elseif ($period === 'year') {
            $query->where('last_shipment_date', '>=', now()->subYear());
        }
        
        $customers = $query->orderBy('total_revenue', 'desc')
            ->limit($limit)
            ->get(['id', 'name', 'phone', 'total_revenue', 'total_shipments', 'average_shipment_value']);

        return response()->json([
            'success' => true,
            'period' => $period,
            'customers' => $customers
        ]);
    }

    /**
     * API: Update customer status
     */
    public function updateStatus(Request $request, Customer $customer): JsonResponse
    {
        $request->validate([
            'status' => 'required|in:active,inactive,blocked'
        ]);

        $oldStatus = $customer->status;
        $newStatus = $request->status;

        DB::transaction(function () use ($customer, $oldStatus, $newStatus) {
            $customer->update(['status' => $newStatus]);
            
            $customer->logActivity(
                'status_changed',
                "Customer status changed from {$oldStatus} to {$newStatus}",
                ['old_status' => $oldStatus, 'new_status' => $newStatus]
            );
        });

        return response()->json([
            'success' => true,
            'message' => 'Customer status updated successfully',
            'new_status' => $newStatus
        ]);
    }

    /**
     * API: Get customer activities
     */
    public function getCustomerActivities(Customer $customer, Request $request): JsonResponse
    {
        $limit = $request->get('limit', 20);
        $offset = $request->get('offset', 0);
        
        $activities = $customer->activities()
            ->with('performer')
            ->orderBy('performed_at', 'desc')
            ->offset($offset)
            ->limit($limit)
            ->get();

        $total = $customer->activities()->count();

        return response()->json([
            'success' => true,
            'activities' => $activities,
            'total' => $total,
            'has_more' => ($offset + $limit) < $total
        ]);
    }

    /**
     * API: Bulk update customer status
     */
    public function bulkUpdateStatus(Request $request): JsonResponse
    {
        $request->validate([
            'customer_ids' => 'required|array',
            'customer_ids.*' => 'exists:customers,id',
            'status' => 'required|in:active,inactive,blocked'
        ]);

        $updatedCount = DB::transaction(function () use ($request) {
            $customers = Customer::whereIn('id', $request->customer_ids)->get();
            $newStatus = $request->status;
            
            foreach ($customers as $customer) {
                $oldStatus = $customer->status;
                $customer->update(['status' => $newStatus]);
                
                $customer->logActivity(
                    'status_changed',
                    "Customer status changed from {$oldStatus} to {$newStatus} (bulk update)",
                    ['old_status' => $oldStatus, 'new_status' => $newStatus, 'bulk_update' => true]
                );
            }
            
            return $customers->count();
        });

        return response()->json([
            'success' => true,
            'message' => "Status updated for {$updatedCount} customers",
            'updated_count' => $updatedCount
        ]);
    }

    /**
     * Export customers to CSV
     */
    public function export(Request $request)
    {
        $query = Customer::query();

        // Apply the same filters as index
        if ($request->has('customer_type') && $request->customer_type) {
            $query->where('customer_type', $request->customer_type);
        }

        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        $customers = $query->orderBy('total_revenue', 'desc')->get();

        $fileName = 'customers-' . date('Y-m-d') . '.csv';
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$fileName}\"",
        ];

        $callback = function() use ($customers) {
            $file = fopen('php://output', 'w');
            
            // Add CSV headers
            fputcsv($file, [
                'Name', 'Phone', 'Email', 'Customer Type', 'Company Name',
                'Total Revenue', 'Total Shipments', 'Average Shipment Value',
                'First Shipment', 'Last Shipment', 'Status', 'Created Date'
            ]);

            // Add data rows
            foreach ($customers as $customer) {
                fputcsv($file, [
                    $customer->name,
                    $customer->phone,
                    $customer->email ?? 'N/A',
                    ucfirst($customer->customer_type),
                    $customer->company_name ?? 'N/A',
                    $customer->total_revenue,
                    $customer->total_shipments,
                    $customer->average_shipment_value,
                    $customer->first_shipment_date?->format('Y-m-d') ?? 'Never',
                    $customer->last_shipment_date?->format('Y-m-d') ?? 'Never',
                    ucfirst($customer->status),
                    $customer->created_at->format('Y-m-d')
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * Get customer dashboard statistics
     */
    public function getDashboardStats(): JsonResponse
    {
        $stats = [
            'total_customers' => Customer::count(),
            'active_customers' => Customer::where('status', 'active')->count(),
            'total_revenue' => (float) Customer::sum('total_revenue'),
            'average_revenue_per_customer' => (float) Customer::avg('total_revenue'),
            'customers_by_type' => Customer::groupBy('customer_type')
                ->selectRaw('customer_type, COUNT(*) as count')
                ->pluck('count', 'customer_type'),
            'recent_customers' => Customer::orderBy('created_at', 'desc')
                ->limit(5)
                ->get(['id', 'name', 'phone', 'created_at']),
            'top_customers' => Customer::orderBy('total_revenue', 'desc')
                ->limit(5)
                ->get(['id', 'name', 'phone', 'total_revenue'])
        ];

        return response()->json([
            'success' => true,
            'stats' => $stats
        ]);
    }
}