<?php

namespace App\Http\Controllers;

use App\Models\Currency;
use Illuminate\Http\Request;

class CurrencyController extends Controller
{
    public function index()
    {
        // Changed from get() to paginate() to fix the hasPages() error
        $currencies = Currency::query()
            ->when(true, function ($query) {
                // Only try to count services if the column exists
                if (\Schema::hasColumn('services', 'default_currency_id')) {
                    return $query->withCount('services');
                }
                return $query;
            })
            ->paginate(10); // 10 items per page

        return view('currencies.index', compact('currencies'));
    }

    public function create()
    {
        return view('currencies.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'code' => 'required|string|max:3|unique:currencies|regex:/^[A-Z]{3}$/',
            'name' => 'required|string|max:255',
            'symbol' => 'nullable|string|max:5',
            'buying_exchange_rate' => 'required|numeric|min:0.0001',
            'selling_exchange_rate' => 'required|numeric|min:0.0001|gte:buying_exchange_rate',
        ], [
            'selling_exchange_rate.gte' => 'The selling rate must be greater than or equal to the buying rate.',
            'code.regex' => 'The currency code must be exactly 3 uppercase letters.'
        ]);

        Currency::create($validated);

        return redirect()->route('currencies.index')
            ->with('success', 'Currency created successfully.');
    }

    public function edit(Currency $currency)
    {
        return view('currencies.edit', compact('currency'));
    }

    public function update(Request $request, Currency $currency)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'symbol' => 'nullable|string|max:5',
            'buying_exchange_rate' => 'required|numeric|min:0.0001',
            'selling_exchange_rate' => 'required|numeric|min:0.0001|gte:buying_exchange_rate',
        ], [
            'selling_exchange_rate.gte' => 'The selling rate must be greater than or equal to the buying rate.'
        ]);

        $currency->update($validated);

        return redirect()->route('currencies.index')
            ->with('success', 'Currency updated successfully.');
    }

    public function destroy(Currency $currency)
    {
        if ($currency->code === 'KES') {
            return redirect()->route('currencies.index')
                ->with('error', 'Cannot delete the base currency (KES).');
        }

        // Check if services relationship exists and has records
        if (\Schema::hasColumn('services', 'default_currency_id') && $currency->services()->exists()) {
            return redirect()->route('currencies.index')
                ->with('error', 'Cannot delete currency - it is being used by services.');
        }

        $currency->delete();

        return redirect()->route('currencies.index')
            ->with('success', 'Currency deleted successfully.');
    }


    // Add these methods to your CurrencyController

/**
 * API: Get exchange rate between two currencies
 */
public function getExchangeRate($from, $to)
{
    $baseCurrency = Currency::base();
    
    // If converting from base currency
    if (strtoupper($from) === $baseCurrency->code) {
        $target = Currency::where('code', strtoupper($to))->first();
        if (!$target) {
            return response()->json(['error' => 'Target currency not found'], 404);
        }
        return response()->json([
            'rate' => 1 / $target->selling_exchange_rate,
            'inverse_rate' => $target->selling_exchange_rate
        ]);
    }
    
    // If converting to base currency
    if (strtoupper($to) === $baseCurrency->code) {
        $source = Currency::where('code', strtoupper($from))->first();
        if (!$source) {
            return response()->json(['error' => 'Source currency not found'], 404);
        }
        return response()->json([
            'rate' => $source->selling_exchange_rate,
            'inverse_rate' => 1 / $source->selling_exchange_rate
        ]);
    }
    
    // Convert between two non-base currencies
    $source = Currency::where('code', strtoupper($from))->first();
    $target = Currency::where('code', strtoupper($to))->first();
    
    if (!$source || !$target) {
        return response()->json(['error' => 'Currency not found'], 404);
    }
    
    // Convert source to base, then base to target
    $rate = $source->selling_exchange_rate * (1 / $target->selling_exchange_rate);
    
    return response()->json([
        'rate' => $rate,
        'inverse_rate' => 1 / $rate
    ]);
}

/**
 * API: Get all currencies for dropdown
 */
public function getCurrencies()
{
    $currencies = Currency::select('code', 'name', 'symbol')
        ->orderBy('code')
        ->get()
        ->map(function($currency) {
            return [
                'code' => $currency->code,
                'name' => $currency->name,
                'symbol' => $currency->symbol,
                'display' => "{$currency->code} - {$currency->name}"
            ];
        });
    
    return response()->json($currencies);
}
}