<?php

use Illuminate\Support\Facades\Route;
use Illuminate\Http\Request;

// Controllers
use App\Http\Controllers\PosController;
use App\Http\Controllers\Auth\AuthenticatedSessionController;
use App\Http\Controllers\Api\PosProductController;
use App\Http\Controllers\Api\ProductApiController;
use App\Http\Controllers\Api\PaymentApiController;
use App\Http\Controllers\CustomerController;
use App\Http\Controllers\SaleController;
use App\Http\Controllers\MpesaTransactionController;
use App\Http\Controllers\HoldSaleController;
use App\Http\Controllers\DashboardController;
use App\Http\Controllers\CategoryController;
use App\Http\Controllers\SettingsController;
use App\Http\Controllers\StoreInformationController;
use App\Http\Controllers\PaymentController;
use App\Http\Controllers\ReceiptPDFController;
use App\Http\Controllers\ETIMSController;
use App\Http\Controllers\VatReportController;
use App\Http\Controllers\UnifiedNumberController;

// New API Controllers
use App\Http\Controllers\Api\PaymentMethodController;
use App\Http\Controllers\Api\PaymentProcessingController;
use App\Http\Controllers\Api\PaymentCallbackController;
use App\Http\Controllers\Api\ReceiptController;
use App\Http\Controllers\Api\StatsApiController;
use App\Http\Controllers\Api\HoldSaleApiController;
use App\Http\Controllers\Api\PosApiController;
use App\Http\Controllers\Api\DashboardApiController;
use App\Http\Controllers\Api\SettingsApiController;

// -----------------------------
// Debug route (for testing)
// -----------------------------
Route::get('/debug/test', function() {
    return response()->json([
        'success' => true,
        'message' => 'API is accessible',
        'timestamp' => now(),
        'request' => [
            'url' => request()->fullUrl(),
            'method' => request()->method(),
            'user' => auth()->check() ? auth()->user()->name : 'Not authenticated'
        ]
    ]);
});

// -----------------------------
// Public API routes (guest)
// -----------------------------
Route::middleware('guest:sanctum')->group(function () {
    Route::post('/login', [AuthenticatedSessionController::class, 'store'])
        ->name('api.login');
});

// -----------------------------
// Protected API routes (authenticated)
// -----------------------------
Route::middleware(['auth:sanctum', 'verified'])->group(function () {

    // API Logout
    Route::post('/logout', [AuthenticatedSessionController::class, 'destroy'])
        ->name('api.logout');

    // Test API endpoint
    Route::get('/pos/test', function() {
        return response()->json([
            'success' => true,
            'message' => 'POS API is working',
            'timestamp' => now()->toISOString(),
            'user' => auth()->user()->name
        ]);
    });

    // -----------------------------
    // Store Information API Routes
    // -----------------------------
    Route::prefix('store')->name('store.')->group(function () {
        Route::get('/info', [StoreInformationController::class, 'index'])->name('info');
        Route::post('/info', [StoreInformationController::class, 'store'])->name('store');
        Route::put('/info/{storeInformation}', [StoreInformationController::class, 'update'])->name('update');
        Route::post('/clear-cache', [StoreInformationController::class, 'clearCache'])->name('clear-cache');
        Route::get('/pos-info', [StoreInformationController::class, 'posInfo'])->name('pos-info');
        Route::get('/receipt-settings', [StoreInformationController::class, 'receiptSettings'])->name('receipt-settings');
        Route::get('/health', [StoreInformationController::class, 'health'])->name('health');
        Route::get('/config', [StoreInformationController::class, 'config'])->name('config');
    });

    // -----------------------------
    // POS API Routes
    // -----------------------------
    Route::prefix('pos')->name('pos.')->group(function () {
        Route::post('/process-sale', [PosController::class, 'processSaleFromPOS'])->name('process-sale');
        Route::post('/checkout', [PosApiController::class, 'checkout'])->name('checkout');
        Route::post('/test-payment', function(Request $request) {
            return response()->json([
                'success' => true,
                'message' => 'Payment test successful',
                'test_data' => $request->all(),
                'invoice_no' => 'TEST-' . date('Ymd') . '-' . rand(1000, 9999)
            ]);
        });
        Route::get('/today-stats', [PosController::class, 'getTodayStats'])->name('today-stats');
        Route::get('/dashboard-stats', [PosController::class, 'getDashboardStats'])->name('dashboard-stats');
        Route::post('/scan-barcode', [PosController::class, 'scanBarcode'])->name('scan-barcode');
        Route::get('/search-products', [PosController::class, 'searchProducts'])->name('search-products');
    });

    // -----------------------------
    // Payment Methods API
    // -----------------------------
    Route::prefix('payment-methods')->name('payment-methods.')->group(function () {
        Route::get('/', [PaymentMethodController::class, 'index'])->name('index');
        Route::get('/active', [PaymentMethodController::class, 'active'])->name('active');
        Route::get('/types', [PaymentMethodController::class, 'types'])->name('types');
        Route::get('/{id}', [PaymentMethodController::class, 'show'])->name('show');
    });

    // -----------------------------
    // Payment Processing API
    // -----------------------------
    Route::prefix('payments')->name('payments.')->group(function () {
        // Main payment processing
        Route::post('/process', [PaymentController::class, 'process'])->name('process');
        Route::post('/verify', [PaymentController::class, 'verify'])->name('verify');
        Route::post('/refund', [PaymentController::class, 'refund'])->name('refund');
        
        // Quick payment methods
        Route::post('/cash', [PaymentController::class, 'processCash'])->name('cash');
        Route::post('/mpesa', [PaymentController::class, 'processMpesa'])->name('mpesa');
        Route::post('/card', [PaymentController::class, 'processCard'])->name('card');
        Route::post('/split', [PaymentController::class, 'processSplit'])->name('split');
        
        // M-Pesa specific
        Route::post('/mpesa/stk-push', [PaymentController::class, 'stkPush'])->name('mpesa.stk-push');
        Route::post('/mpesa/manual', [PaymentController::class, 'manualMpesa'])->name('mpesa.manual');
        Route::post('/mpesa/check-status', [PaymentController::class, 'checkMpesaStatus'])->name('mpesa.check-status');
        
        // Payment methods list
        Route::get('/methods', [PaymentController::class, 'getPaymentMethods'])->name('methods');
        
        // Status check
        Route::get('/status/{reference}', [PaymentProcessingController::class, 'status'])->name('status');
    });

    // -----------------------------
    // Sale API Routes
    // -----------------------------
    Route::prefix('sales')->name('sales.')->group(function () {
        Route::post('/process', [SaleController::class, 'processSale'])->name('process');
        Route::post('/checkout', [SaleController::class, 'checkout'])->name('checkout');
        Route::get('/{id}/receipt', [SaleController::class, 'apiGetReceipt'])->name('receipt');
        Route::get('/{id}/payment', [SaleController::class, 'getSalePayment'])->name('payment');
        Route::get('/recent', [SaleController::class, 'apiGetRecentSales'])->name('recent');
        Route::get('/search', [SaleController::class, 'apiSearchSales'])->name('search');
        Route::post('/{id}/refund', [SaleController::class, 'apiRefund'])->name('refund');
        Route::post('/{id}/send-receipt', [SaleController::class, 'apiSendReceipt'])->name('send-receipt');
    });

    // -----------------------------
    // Receipt API Routes
    // -----------------------------
    Route::prefix('receipts')->name('receipts.')->group(function () {
        Route::post('/generate-pdf', [ReceiptPDFController::class, 'generatePDF'])->name('generate-pdf');
        Route::post('/generate-from-data', [ReceiptPDFController::class, 'generateFromData'])->name('generate-from-data');
        Route::post('/download-link', [ReceiptPDFController::class, 'getDownloadLink'])->name('download-link');
        Route::get('/{id}', [ReceiptController::class, 'show'])->name('show');
        Route::post('/{id}/print', [ReceiptController::class, 'print'])->name('print');
        Route::post('/{id}/email', [ReceiptController::class, 'email'])->name('email');
        Route::post('/{id}/sms', [ReceiptController::class, 'sms'])->name('sms');
    });

    // -----------------------------
    // Customer API Routes - FIXED VERSION
    // -----------------------------
    Route::prefix('customers')->name('customers.')->group(function () {
        // List customers
        Route::get('/', [CustomerController::class, 'apiGetCustomers'])->name('index');
        
        // Get customer statistics
        Route::get('/stats', [CustomerController::class, 'apiGetStats'])->name('stats');
        
        // Alternative endpoint for statistics
        Route::get('/statistics', [CustomerController::class, 'apiGetStats'])->name('statistics');
        
        // Search customers
        Route::get('/search', [CustomerController::class, 'apiSearchCustomers'])->name('search');
        
        // Quick add customer
        Route::post('/quick-add', [CustomerController::class, 'apiQuickAdd'])->name('quick-add');
        
        // Get single customer
        Route::get('/{id}', [CustomerController::class, 'apiGetCustomer'])->name('show');
        
        // Get customer stats by ID
        Route::get('/{id}/stats', [CustomerController::class, 'apiGetCustomerStats'])->name('customer-stats');
        
        // Store new customer
        Route::post('/', [CustomerController::class, 'apiStore'])->name('store');
        
        // Credit info
        Route::get('/{id}/credit-info', [CustomerController::class, 'apiGetCreditInfo'])->name('credit-info');
        
        // Update balance
        Route::post('/{id}/update-balance', [CustomerController::class, 'updateCreditBalance'])->name('update-balance');
        
        // Purchase history
        Route::get('/{id}/history', [CustomerController::class, 'getPurchaseHistory'])->name('history');
        
        // VAT operations
        Route::post('/{customer}/apply-exemption', [CustomerController::class, 'applyVatExemption'])->name('apply-exemption');
        Route::post('/{customer}/apply-zero-rated', [CustomerController::class, 'applyZeroRated'])->name('apply-zero-rated');
        Route::post('/{customer}/remove-vat-status', [CustomerController::class, 'removeVatStatus'])->name('remove-vat-status');
        Route::get('/{customer}/vat-report', [CustomerController::class, 'vatReport'])->name('vat-report');
    });

    // -----------------------------
    // Product API Routes
    // -----------------------------
    Route::prefix('products')->name('products.')->group(function () {
        Route::get('/search', [ProductApiController::class, 'search'])->name('search');
        Route::get('/scan', [ProductApiController::class, 'scan'])->name('scan');
        Route::get('/barcode/{barcode}', [ProductApiController::class, 'getByBarcode'])->name('barcode');
        Route::get('/{id}', [ProductApiController::class, 'show'])->name('show');
        Route::post('/{id}/adjust-stock', [ProductApiController::class, 'adjustStock'])->name('adjust-stock');
    });

    // -----------------------------
    // Hold Sale API Routes
    // -----------------------------
    Route::prefix('hold-sales')->name('hold-sales.')->group(function () {
        Route::get('/', [HoldSaleController::class, 'index'])->name('index');
        Route::post('/', [HoldSaleController::class, 'store'])->name('store');
        Route::get('/{id}/load', [HoldSaleController::class, 'load'])->name('load');
        Route::delete('/{id}', [HoldSaleController::class, 'destroy'])->name('destroy');
        Route::get('/stats', [HoldSaleController::class, 'stats'])->name('stats');
        Route::get('/search', [HoldSaleController::class, 'search'])->name('search');
    });

    // -----------------------------
    // M-Pesa API Routes
    // -----------------------------
    Route::prefix('mpesa')->name('mpesa.')->group(function () {
        Route::get('/transactions', [MpesaTransactionController::class, 'index'])->name('transactions.index');
        Route::post('/check-payment', [MpesaTransactionController::class, 'checkPendingPayment'])->name('check-payment');
        Route::post('/transactions/{id}/match', [MpesaTransactionController::class, 'match'])->name('transactions.match');
        Route::post('/stk-push', [MpesaTransactionController::class, 'stkPush'])->name('stk-push');
        Route::get('/transactions/search', [MpesaTransactionController::class, 'search'])->name('transactions.search');
        Route::get('/{id}', [MpesaTransactionController::class, 'show'])->name('show');
        Route::get('/stats', [MpesaTransactionController::class, 'stats'])->name('stats');
    });

    // -----------------------------
    // Payment API Routes (Legacy - for backward compatibility)
    // -----------------------------
    Route::prefix('payment')->name('payment.')->group(function () {
        Route::post('/process', [PaymentApiController::class, 'processPayment'])->name('process');
        Route::post('/verify', [PaymentApiController::class, 'verify'])->name('verify');
    });

    // -----------------------------
    // Dashboard API Routes
    // -----------------------------
    Route::prefix('dashboard')->name('dashboard.')->group(function () {
        Route::get('/stats', [DashboardController::class, 'getStats'])->name('stats');
        Route::get('/payment-stats', [PaymentProcessingController::class, 'dashboardStats'])->name('payment-stats');
        Route::get('/recent-sales', [DashboardController::class, 'getRecentSales'])->name('recent-sales');
        Route::get('/recent-purchases', [DashboardController::class, 'getRecentPurchases'])->name('recent-purchases');
        Route::get('/low-stock', [DashboardController::class, 'getLowStock'])->name('low-stock');
        Route::get('/top-products', [DashboardController::class, 'getTopProducts'])->name('top-products');
    });

    // -----------------------------
    // Settings API Routes
    // -----------------------------
    Route::prefix('settings')->name('settings.')->group(function () {
        Route::get('/store', [SettingsController::class, 'apiGetSettings'])->name('store');
        Route::get('/payment-settings', [PaymentProcessingController::class, 'getSettings'])->name('payment-settings');
        Route::get('/printer-settings', [SettingsApiController::class, 'getPrinterSettings'])->name('printer-settings');
    });

    // -----------------------------
    // Print API Routes
    // -----------------------------
    Route::prefix('print')->name('print.')->group(function () {
        Route::post('/receipt', [ReceiptController::class, 'printReceipt'])->name('receipt');
        Route::post('/test', [ReceiptController::class, 'testPrint'])->name('test');
    });

    // -----------------------------
    // Quick Operations
    // -----------------------------
    Route::post('/quick-customer', [CustomerController::class, 'quickCreate'])->name('quick-customer');

    // -----------------------------
    // Categories API
    // -----------------------------
    Route::prefix('categories')->name('categories.')->group(function () {
        Route::get('/', [CategoryController::class, 'apiGetCategories'])->name('index');
        Route::get('/tree', [CategoryController::class, 'apiGetCategoryTree'])->name('tree');
        Route::get('/{id}/products', [CategoryController::class, 'apiGetCategoryProducts'])->name('products');
    });

    // -----------------------------
    // System APIs
    // -----------------------------
    Route::post('/log-error', function(Request $request) {
        \Log::error('Client-side error: ' . json_encode($request->all()));
        return response()->json(['success' => true]);
    });

    Route::post('/analytics/track', function(Request $request) {
        \Log::info('Analytics: ' . json_encode($request->all()));
        return response()->json(['success' => true]);
    });

    // -----------------------------
    // Inventory APIs
    // -----------------------------
    Route::prefix('inventory')->name('inventory.')->group(function () {
        Route::post('/update-stock', function(Request $request) {
            try {
                $data = $request->validate([
                    'product_id' => 'required|exists:products,id',
                    'quantity' => 'required|integer',
                    'type' => 'required|in:sale,restock,adjustment',
                    'reason' => 'nullable|string'
                ]);
                
                // Update stock logic here
                
                return response()->json([
                    'success' => true,
                    'message' => 'Stock updated successfully'
                ]);
            } catch (\Exception $e) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to update stock'
                ], 500);
            }
        })->name('update-stock');
    });

    // -----------------------------
    // eTIMS API Routes
    // -----------------------------
    Route::prefix('etims')->name('etims.')->group(function () {
        Route::get('/config', [ETIMSController::class, 'getConfig'])->name('config');
        Route::get('/credentials/active', [ETIMSController::class, 'getActiveCredentials'])->name('credentials.active');
        Route::post('/submit', [ETIMSController::class, 'submitReceipt'])->name('submit');
        Route::get('/status/{id}', [ETIMSController::class, 'getStatus'])->name('status');
        Route::post('/test-connection', [ETIMSController::class, 'testConnection'])->name('test-connection');
    });

    // -----------------------------
    // VAT Reports API
    // -----------------------------
    Route::prefix('vat-reports')->name('vat-reports.')->group(function () {
        Route::get('/summary', [VatReportController::class, 'summary'])->name('summary');
        Route::get('/detailed', [VatReportController::class, 'detailed'])->name('detailed');
        Route::get('/expiring-certificates', [VatReportController::class, 'expiringCertificates'])->name('expiring-certificates');
    });
});

// -----------------------------
// Public Callback Routes (No Auth)
// -----------------------------

// M-Pesa Callback
Route::prefix('mpesa')->group(function () {
    Route::post('/callback', [MpesaTransactionController::class, 'storeCallback'])
        ->withoutMiddleware(['web', 'csrf'])
        ->name('mpesa.callback.public');
});

// Payment Provider Callbacks
Route::post('/payment/callback/{provider}', [PaymentCallbackController::class, 'handle'])
    ->withoutMiddleware(['web', 'csrf'])
    ->name('payment.callback.public');

// -----------------------------
// Emergency fallback routes
// -----------------------------
Route::get('/customer-statistics', function() {
    return response()->json([
        'success' => true,
        'total_customers' => 150,
        'active_today' => 25,
        'total_credit' => 500000,
        'avg_sale' => 4500,
        'emergency' => true,
        'message' => 'Using fallback endpoint'
    ]);
});

Route::get('/customers/stats', function() {
    return response()->json([
        'success' => true,
        'total_customers' => 150,
        'active_today' => 25,
        'total_credit' => 500000,
        'avg_sale' => 4500,
        'emergency' => true,
        'message' => 'Using fallback stats endpoint'
    ]);
});

// Test route
Route::get('/test', function () {
    return response()->json([
        'success' => true,
        'message' => 'API is working',
        'timestamp' => now(),
    ]);
});



// In your API routes 
Route::get('/api/sales/latest', function() {
    $sale = \App\Models\Sale::latest()->first();
    
    if (!$sale) {
        return response()->json([
            'success' => false,
            'message' => 'No sales found'
        ]);
    }
    
    return response()->json([
        'success' => true,
        'sale' => [
            'id' => $sale->id,
            'invoice_no' => $sale->invoice_no,
            'receipt_no' => $sale->receipt_no,
            'grand_total' => $sale->grand_total,
            'date' => $sale->created_at->format('Y-m-d H:i:s')
        ]
    ]);
});

// -----------------------------
// Fallback
// -----------------------------
Route::fallback(function () {
    return response()->json([
        'error' => 'Route not found',
        'message' => 'The requested API endpoint does not exist',
        'url' => request()->fullUrl(),
        'method' => request()->method()
    ], 404);
});


/////

// -----------------------------
// POS Health Routes (CSRF exempt)
// -----------------------------
Route::prefix('pos')->group(function () {
    // Allow GET for health check (no CSRF needed)
    Route::get('/health', [\App\Http\Controllers\POS\POSHealthController::class, 'report'])
        ->name('api.pos.health.get')
        ->middleware('auth:sanctum'); // Keep auth but remove CSRF
    
    // Allow POST for health check with CSRF token
    Route::post('/health', [\App\Http\Controllers\POS\POSHealthController::class, 'report'])
        ->name('api.pos.health.post')
        ->middleware('auth:sanctum'); // CSRF will be handled by token in header
    
    // Simple health check endpoint (public or minimal)
    Route::get('/health-check', [\App\Http\Controllers\POS\POSHealthController::class, 'health'])
        ->name('api.pos.health-check');
    
    // Error logging endpoint
    Route::post('/log-error', [\App\Http\Controllers\POS\POSHealthController::class, 'logError'])
        ->name('api.pos.log-error')
        ->middleware('auth:sanctum');
});

// Add missing receipt route
Route::get('/pos/receipt/{identifier}', [\App\Http\Controllers\PaymentController::class, 'generateReceipt'])
    ->name('pos.receipt');
    Route::get('/unified-number', [\App\Http\Controllers\UnifiedNumberController::class, 'getNextNumber']);

    // In routes/api.php
Route::middleware(['auth:sanctum'])->group(function () {
    // Debug endpoint for unified numbers
    Route::get('/debug/unified/{identifier?}', function ($identifier = null) {
        try {
            $salesQuery = Sale::query();
            
            if ($identifier) {
                $salesQuery->where(function($q) use ($identifier) {
                    $q->where('unified_number', $identifier)
                      ->orWhere('invoice_no', $identifier)
                      ->orWhere('receipt_no', $identifier)
                      ->orWhere('id', $identifier);
                });
            }
            
            $sales = $salesQuery->orderBy('created_at', 'desc')
                ->limit(10)
                ->get();
            
            $todaySales = Sale::whereDate('created_at', today())->count();
            
            return response()->json([
                'success' => true,
                'search_identifier' => $identifier,
                'sales_found' => $sales->count(),
                'sales' => $sales->map(function($sale) {
                    return [
                        'id' => $sale->id,
                        'unified_number' => $sale->unified_number,
                        'invoice_no' => $sale->invoice_no,
                        'receipt_no' => $sale->receipt_no,
                        'created_at' => $sale->created_at->format('Y-m-d H:i:s'),
                        'grand_total' => $sale->grand_total,
                        'customer' => $sale->customer ? $sale->customer->name : null,
                        'payment_method' => $sale->payment_method
                    ];
                }),
                'today_sales_count' => $todaySales,
                'total_sales_count' => Sale::count(),
                'database' => config('database.default'),
                'timestamp' => now()->toDateTimeString()
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ], 500);
        }
    });



    // POS-specific API routes (no version prefix for simplicity)
Route::prefix('pos-api')->name('pos-api.')->middleware(['auth:sanctum', 'verified'])->group(function () {
    // Products for POS
    Route::get('/products', [\App\Http\Controllers\Api\PosProductController::class, 'index'])->name('products.index');
    Route::get('/products/search', [\App\Http\Controllers\Api\PosProductController::class, 'search'])->name('products.search');
    Route::get('/products/barcode/{barcode}', [\App\Http\Controllers\Api\PosProductController::class, 'barcode'])->name('products.barcode');
    Route::get('/categories', [\App\Http\Controllers\Api\PosProductController::class, 'categories'])->name('categories');
    
    // Quick test endpoint
    Route::get('/test', function() {
        return response()->json([
            'success' => true,
            'message' => 'POS API is working',
            'timestamp' => now(),
            'user' => auth()->user()->name ?? 'Guest'
        ]);
    });
});

    
});
