@extends('layouts.app')

@section('style')
    <style>
        .unit-badge {
            font-size: 0.85em;
            padding: 0.25em 0.6em;
        }
        .unit-type {
            font-size: 0.75em;
            padding: 0.2em 0.5em;
        }
        .action-buttons .btn {
            padding: 0.25rem 0.5rem;
            font-size: 0.875rem;
        }
        .conversion-rate {
            font-family: monospace;
            background: #f8f9fa;
            padding: 0.2rem 0.5rem;
            border-radius: 4px;
            font-size: 0.9em;
        }
    </style>
@endsection

@section('content')
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">Units</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item active" aria-current="page">Units Management</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <a href="{{ route('units.create') }}" class="btn btn-primary">
                    <i class="bx bx-plus"></i> Add New Unit
                </a>
            </div>
        </div>
        <!--end breadcrumb-->

        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-body">
                        <!-- Statistics -->
                        <div class="row mb-4">
                            <div class="col-md-3 col-sm-6">
                                <div class="card bg-primary bg-gradient text-white">
                                    <div class="card-body">
                                        <div class="d-flex align-items-center">
                                            <div class="flex-grow-1">
                                                <h5 class="mb-0">{{ $stats['total'] }}</h5>
                                                <p class="mb-0">Total Units</p>
                                            </div>
                                            <div class="flex-shrink-0">
                                                <i class="bx bx-ruler fs-1"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3 col-sm-6">
                                <div class="card bg-success bg-gradient text-white">
                                    <div class="card-body">
                                        <div class="d-flex align-items-center">
                                            <div class="flex-grow-1">
                                                <h5 class="mb-0">{{ $stats['active'] }}</h5>
                                                <p class="mb-0">Active Units</p>
                                            </div>
                                            <div class="flex-shrink-0">
                                                <i class="bx bx-check-circle fs-1"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3 col-sm-6">
                                <div class="card bg-info bg-gradient text-white">
                                    <div class="card-body">
                                        <div class="d-flex align-items-center">
                                            <div class="flex-grow-1">
                                                <h5 class="mb-0">{{ $stats['base_units'] }}</h5>
                                                <p class="mb-0">Base Units</p>
                                            </div>
                                            <div class="flex-shrink-0">
                                                <i class="bx bx-cube-alt fs-1"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3 col-sm-6">
                                <div class="card bg-warning bg-gradient text-white">
                                    <div class="card-body">
                                        <div class="d-flex align-items-center">
                                            <div class="flex-grow-1">
                                                <h5 class="mb-0">{{ $stats['derived_units'] }}</h5>
                                                <p class="mb-0">Derived Units</p>
                                            </div>
                                            <div class="flex-shrink-0">
                                                <i class="bx bx-layer fs-1"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Search and Filter -->
                        <div class="row mb-4">
                            <div class="col-md-8">
                                <form method="GET" action="{{ route('units.index') }}" class="row g-3">
                                    <div class="col-md-4">
                                        <input type="text" name="search" class="form-control" 
                                               placeholder="Search units..." value="{{ request('search') }}">
                                    </div>
                                    <div class="col-md-3">
                                        <select name="status" class="form-select">
                                            <option value="">All Status</option>
                                            <option value="active" {{ request('status') == 'active' ? 'selected' : '' }}>Active</option>
                                            <option value="inactive" {{ request('status') == 'inactive' ? 'selected' : '' }}>Inactive</option>
                                        </select>
                                    </div>
                                    <div class="col-md-3">
                                        <select name="type" class="form-select">
                                            <option value="">All Types</option>
                                            <option value="base" {{ request('type') == 'base' ? 'selected' : '' }}>Base Units</option>
                                            <option value="derived" {{ request('type') == 'derived' ? 'selected' : '' }}>Derived Units</option>
                                        </select>
                                    </div>
                                    <div class="col-md-2">
                                        <button type="submit" class="btn btn-primary w-100">
                                            <i class="bx bx-search"></i> Filter
                                        </button>
                                    </div>
                                </form>
                            </div>
                            <div class="col-md-4 text-end">
                                <div class="btn-group" role="group">
                                    <button type="button" class="btn btn-outline-secondary" onclick="exportUnits()">
                                        <i class="bx bx-export"></i> Export
                                    </button>
                                    <button type="button" class="btn btn-outline-secondary" onclick="bulkUpdateModal()">
                                        <i class="bx bx-edit"></i> Bulk Edit
                                    </button>
                                    <button type="button" class="btn btn-outline-danger" onclick="bulkDeleteModal()">
                                        <i class="bx bx-trash"></i> Bulk Delete
                                    </button>
                                </div>
                            </div>
                        </div>

                        <!-- Units Table -->
                        <div class="table-responsive">
                            <table class="table table-hover table-striped">
                                <thead class="table-light">
                                    <tr>
                                        <th width="50">
                                            <input type="checkbox" id="selectAll">
                                        </th>
                                        <th>Unit Name</th>
                                        <th>Abbreviation</th>
                                        <th>Type</th>
                                        <th>Conversion Rate</th>
                                        <th>Base Unit</th>
                                        <th>Status</th>
                                        <th>Products</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @forelse($units as $unit)
                                        <tr>
                                            <td>
                                                <input type="checkbox" class="unit-checkbox" value="{{ $unit->id }}">
                                            </td>
                                            <td>
                                                <strong>{{ $unit->name }}</strong>
                                                @if($unit->description)
                                                    <br><small class="text-muted">{{ Str::limit($unit->description, 50) }}</small>
                                                @endif
                                            </td>
                                            <td>
                                                <span class="badge bg-primary unit-badge">{{ $unit->abbreviation }}</span>
                                            </td>
                                            <td>
                                                @if($unit->is_base_unit)
                                                    <span class="badge bg-info unit-type">Base Unit</span>
                                                @else
                                                    <span class="badge bg-secondary unit-type">Derived Unit</span>
                                                @endif
                                            </td>
                                            <td>
                                                <span class="conversion-rate">{{ number_format($unit->conversion_rate, 4) }}</span>
                                            </td>
                                            <td>
                                                @if($unit->baseUnit)
                                                    <span class="badge bg-light text-dark">{{ $unit->baseUnit->abbreviation }}</span>
                                                @else
                                                    <span class="text-muted">-</span>
                                                @endif
                                            </td>
                                            <td>
                                                <span class="badge bg-{{ $unit->status == 'active' ? 'success' : 'danger' }}">
                                                    {{ ucfirst($unit->status) }}
                                                </span>
                                            </td>
                                            <td>
                                                <span class="badge bg-secondary">{{ $unit->products_count ?? 0 }}</span>
                                            </td>
                                            <td>
                                                <div class="action-buttons">
                                                    <a href="{{ route('units.show', $unit) }}" class="btn btn-sm btn-info" 
                                                       title="View">
                                                        <i class="bx bx-show"></i>
                                                    </a>
                                                    <a href="{{ route('units.edit', $unit) }}" class="btn btn-sm btn-warning" 
                                                       title="Edit">
                                                        <i class="bx bx-edit"></i>
                                                    </a>
                                                    <button type="button" class="btn btn-sm btn-danger" 
                                                            onclick="deleteUnit({{ $unit->id }}, '{{ $unit->name }}')"
                                                            title="Delete">
                                                        <i class="bx bx-trash"></i>
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                    @empty
                                        <tr>
                                            <td colspan="9" class="text-center text-muted py-4">
                                                <i class="bx bx-package fs-1"></i>
                                                <p class="mb-0">No units found</p>
                                                <a href="{{ route('units.create') }}" class="btn btn-primary mt-2">
                                                    <i class="bx bx-plus"></i> Add Your First Unit
                                                </a>
                                            </td>
                                        </tr>
                                    @endforelse
                                </tbody>
                            </table>
                        </div>

                        <!-- Pagination -->
                        <div class="row mt-3">
                            <div class="col-md-6">
                                <p class="text-muted">
                                    Showing {{ $units->firstItem() }} to {{ $units->lastItem() }} of {{ $units->total() }} entries
                                </p>
                            </div>
                            <div class="col-md-6">
                                <div class="float-end">
                                    {{ $units->withQueryString()->links() }}
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Bulk Update Modal -->
<div class="modal fade" id="bulkUpdateModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Bulk Update Units</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <form id="bulkUpdateForm">
                    @csrf
                    <input type="hidden" name="unit_ids" id="bulkUpdateIds">
                    
                    <div class="mb-3">
                        <label class="form-label">Status</label>
                        <select name="data[status]" class="form-select">
                            <option value="">No Change</option>
                            <option value="active">Active</option>
                            <option value="inactive">Inactive</option>
                        </select>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="submitBulkUpdate()">Update Units</button>
            </div>
        </div>
    </div>
</div>

<!-- Bulk Delete Modal -->
<div class="modal fade" id="bulkDeleteModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Confirm Bulk Delete</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to delete the selected units?</p>
                <div class="alert alert-warning">
                    <i class="bx bx-error"></i> This action cannot be undone. Units that are in use cannot be deleted.
                </div>
                <ul id="bulkDeleteList" class="list-group"></ul>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-danger" onclick="submitBulkDelete()">Delete Selected</button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script>
    // Select/Deselect all
    $('#selectAll').change(function() {
        $('.unit-checkbox').prop('checked', this.checked);
    });

    // Get selected unit IDs
    function getSelectedUnitIds() {
        return $('.unit-checkbox:checked').map(function() {
            return $(this).val();
        }).get();
    }

    // Get selected unit names
    function getSelectedUnitNames() {
        let names = [];
        $('.unit-checkbox:checked').each(function() {
            let unitName = $(this).closest('tr').find('td:nth-child(2) strong').text();
            names.push(unitName);
        });
        return names;
    }

    // Bulk update modal
    function bulkUpdateModal() {
        let selectedIds = getSelectedUnitIds();
        if (selectedIds.length === 0) {
            Swal.fire('Warning', 'Please select at least one unit.', 'warning');
            return;
        }

        $('#bulkUpdateIds').val(JSON.stringify(selectedIds));
        $('#bulkUpdateModal').modal('show');
    }

    // Submit bulk update
    function submitBulkUpdate() {
        let formData = $('#bulkUpdateForm').serialize();
        
        $.ajax({
            url: '{{ route("units.bulk-update") }}',
            type: 'POST',
            data: formData,
            success: function(response) {
                Swal.fire('Success', response.message, 'success');
                $('#bulkUpdateModal').modal('hide');
                location.reload();
            },
            error: function(xhr) {
                Swal.fire('Error', xhr.responseJSON?.message || 'Failed to update units', 'error');
            }
        });
    }

    // Bulk delete modal
    function bulkDeleteModal() {
        let selectedIds = getSelectedUnitIds();
        if (selectedIds.length === 0) {
            Swal.fire('Warning', 'Please select at least one unit.', 'warning');
            return;
        }

        let unitNames = getSelectedUnitNames();
        let listHtml = '';
        unitNames.forEach(function(name) {
            listHtml += `<li class="list-group-item">${name}</li>`;
        });

        $('#bulkDeleteList').html(listHtml);
        $('#bulkDeleteIds').val(JSON.stringify(selectedIds));
        $('#bulkDeleteModal').modal('show');
    }

    // Submit bulk delete
    function submitBulkDelete() {
        let selectedIds = getSelectedUnitIds();
        
        $.ajax({
            url: '{{ route("units.bulk-delete") }}',
            type: 'POST',
            data: {
                _token: '{{ csrf_token() }}',
                unit_ids: selectedIds
            },
            success: function(response) {
                Swal.fire('Success', response.message, 'success');
                if (response.errors && response.errors.length > 0) {
                    let errorHtml = '<ul class="text-start">';
                    response.errors.forEach(function(error) {
                        errorHtml += `<li>${error}</li>`;
                    });
                    errorHtml += '</ul>';
                    Swal.fire({
                        title: 'Some units could not be deleted',
                        html: errorHtml,
                        icon: 'warning'
                    });
                }
                $('#bulkDeleteModal').modal('hide');
                location.reload();
            },
            error: function(xhr) {
                Swal.fire('Error', xhr.responseJSON?.message || 'Failed to delete units', 'error');
            }
        });
    }

    // Delete single unit
    function deleteUnit(id, name) {
        Swal.fire({
            title: 'Delete Unit?',
            html: `Are you sure you want to delete <strong>${name}</strong>?`,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#d33',
            cancelButtonColor: '#3085d6',
            confirmButtonText: 'Yes, delete it!'
        }).then((result) => {
            if (result.isConfirmed) {
                $.ajax({
                    url: `/units/${id}`,
                    type: 'DELETE',
                    data: {
                        _token: '{{ csrf_token() }}'
                    },
                    success: function(response) {
                        Swal.fire('Deleted!', response.message || 'Unit deleted successfully.', 'success');
                        location.reload();
                    },
                    error: function(xhr) {
                        Swal.fire('Error!', xhr.responseJSON?.message || 'Failed to delete unit.', 'error');
                    }
                });
            }
        });
    }

    // Export units
    function exportUnits() {
        let params = new URLSearchParams(window.location.search);
        window.location.href = '{{ route("units.export") }}?' + params.toString();
    }
</script>
@endsection