@extends('layouts.app')

@section('style')
    <style>
        .import-card {
            border: 2px dashed #dee2e6;
            border-radius: 10px;
            padding: 40px 20px;
            text-align: center;
            transition: all 0.3s;
        }
        .import-card:hover {
            border-color: #667eea;
            background-color: #f8f9fa;
        }
        .import-card.dragover {
            border-color: #28a745;
            background-color: #e8f5e9;
        }
        .template-download {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 10px;
            margin-top: 20px;
        }
    </style>
@endsection

@section('content')
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">Suppliers</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item"><a href="{{ route('suppliers.index') }}">Suppliers</a></li>
                        <li class="breadcrumb-item active" aria-current="page">Import Suppliers</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <a href="{{ route('suppliers.index') }}" class="btn btn-outline-secondary">
                    <i class="bx bx-arrow-back"></i> Back to Suppliers
                </a>
            </div>
        </div>
        <!--end breadcrumb-->

        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title mb-4">Import Suppliers from CSV</h5>
                        
                        @if(session('success'))
                            <div class="alert alert-success border-0 bg-success alert-dismissible fade show py-2">
                                <div class="d-flex align-items-center">
                                    <div class="font-35 text-white">
                                        <i class="bx bxs-check-circle"></i>
                                    </div>
                                    <div class="ms-3">
                                        <h6 class="mb-0 text-white">Success</h6>
                                        <div class="text-white">{{ session('success') }}</div>
                                    </div>
                                </div>
                                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                            </div>
                        @endif

                        @if(session('error'))
                            <div class="alert alert-danger border-0 bg-danger alert-dismissible fade show py-2">
                                <div class="d-flex align-items-center">
                                    <div class="font-35 text-white">
                                        <i class="bx bxs-message-square-x"></i>
                                    </div>
                                    <div class="ms-3">
                                        <h6 class="mb-0 text-white">Error</h6>
                                        <div class="text-white">{{ session('error') }}</div>
                                    </div>
                                </div>
                                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                            </div>
                        @endif

                        @if(session('import_errors'))
                            <div class="alert alert-warning border-0 bg-warning alert-dismissible fade show py-2">
                                <div class="d-flex align-items-center">
                                    <div class="font-35 text-dark">
                                        <i class="bx bxs-error"></i>
                                    </div>
                                    <div class="ms-3">
                                        <h6 class="mb-0 text-dark">Import Warnings</h6>
                                        <ul class="text-dark mb-0">
                                            @foreach(session('import_errors') as $error)
                                                <li>{{ $error }}</li>
                                            @endforeach
                                        </ul>
                                    </div>
                                </div>
                                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                            </div>
                        @endif

                        <div class="row">
                            <!-- Left Column: Upload Form -->
                            <div class="col-md-6">
                                <form method="POST" action="{{ route('suppliers.process-import') }}" enctype="multipart/form-data" id="importForm">
                                    @csrf
                                    
                                    <div class="import-card" id="dropZone">
                                        <div class="mb-3">
                                            <i class="bx bx-cloud-upload bx-lg text-muted"></i>
                                        </div>
                                        <h5 class="mb-3">Upload CSV File</h5>
                                        <p class="text-muted mb-4">
                                            Drag and drop your CSV file here or click to browse
                                        </p>
                                        <input type="file" name="file" id="fileInput" accept=".csv,.txt" class="d-none" required>
                                        <button type="button" class="btn btn-primary mb-3" onclick="document.getElementById('fileInput').click()">
                                            <i class="bx bx-folder-open"></i> Browse Files
                                        </button>
                                        <div id="fileName" class="text-muted small"></div>
                                    </div>
                                    
                                    <div class="mt-3">
                                        <button type="submit" class="btn btn-success w-100" id="submitBtn" disabled>
                                            <i class="bx bx-import"></i> Import Suppliers
                                        </button>
                                    </div>
                                </form>
                            </div>
                            
                            <!-- Right Column: Instructions -->
                            <div class="col-md-6">
                                <div class="template-download">
                                    <h6 class="mb-3"><i class="bx bx-download"></i> Download Template</h6>
                                    <p class="text-muted mb-3">
                                        Download our CSV template to ensure proper formatting:
                                    </p>
                                    <a href="{{ asset('templates/suppliers-template.csv') }}" class="btn btn-outline-primary w-100 mb-3">
                                        <i class="bx bx-download"></i> Download Template CSV
                                    </a>
                                    
                                    <h6 class="mt-4 mb-3"><i class="bx bx-info-circle"></i> CSV Format Requirements</h6>
                                    <ul class="text-muted">
                                        <li>File must be in CSV format</li>
                                        <li>Maximum file size: 2MB</li>
                                        <li>Required columns: Name</li>
                                        <li>Optional columns: Email, Phone, Address, Contact Person, Website, Tax ID, Payment Terms, Notes, Status</li>
                                        <li>Status can be "active" or "inactive" (default: active)</li>
                                    </ul>
                                    
                                    <h6 class="mt-4 mb-3"><i class="bx bx-table"></i> Sample Data</h6>
                                    <div class="table-responsive">
                                        <table class="table table-sm table-bordered">
                                            <thead>
                                                <tr>
                                                    <th>Name</th>
                                                    <th>Email</th>
                                                    <th>Phone</th>
                                                    <th>Status</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <tr>
                                                    <td>ABC Suppliers</td>
                                                    <td>contact@abcsuppliers.com</td>
                                                    <td>+1 (555) 123-4567</td>
                                                    <td>active</td>
                                                </tr>
                                                <tr>
                                                    <td>XYZ Wholesale</td>
                                                    <td>info@xyzwholesale.com</td>
                                                    <td>+1 (555) 987-6543</td>
                                                    <td>active</td>
                                                </tr>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script>
    $(document).ready(function() {
        const dropZone = document.getElementById('dropZone');
        const fileInput = document.getElementById('fileInput');
        const fileName = document.getElementById('fileName');
        const submitBtn = document.getElementById('submitBtn');
        const importForm = document.getElementById('importForm');
        
        // Prevent default drag behaviors
        ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
            dropZone.addEventListener(eventName, preventDefaults, false);
            document.body.addEventListener(eventName, preventDefaults, false);
        });
        
        // Highlight drop zone when item is dragged over it
        ['dragenter', 'dragover'].forEach(eventName => {
            dropZone.addEventListener(eventName, highlight, false);
        });
        
        ['dragleave', 'drop'].forEach(eventName => {
            dropZone.addEventListener(eventName, unhighlight, false);
        });
        
        // Handle dropped files
        dropZone.addEventListener('drop', handleDrop, false);
        
        // Handle file input change
        fileInput.addEventListener('change', handleFileSelect, false);
        
        function preventDefaults(e) {
            e.preventDefault();
            e.stopPropagation();
        }
        
        function highlight() {
            dropZone.classList.add('dragover');
        }
        
        function unhighlight() {
            dropZone.classList.remove('dragover');
        }
        
        function handleDrop(e) {
            const dt = e.dataTransfer;
            const files = dt.files;
            
            if (files.length > 0) {
                fileInput.files = files;
                handleFileSelect();
            }
        }
        
        function handleFileSelect() {
            if (fileInput.files.length > 0) {
                const file = fileInput.files[0];
                fileName.textContent = `Selected file: ${file.name} (${formatFileSize(file.size)})`;
                submitBtn.disabled = false;
                
                // Validate file type
                const validTypes = ['text/csv', 'application/vnd.ms-excel', 'text/plain'];
                if (!validTypes.includes(file.type) && !file.name.endsWith('.csv')) {
                    alert('Please select a CSV file.');
                    fileInput.value = '';
                    fileName.textContent = '';
                    submitBtn.disabled = true;
                    return;
                }
                
                // Validate file size (2MB)
                if (file.size > 2 * 1024 * 1024) {
                    alert('File size must be less than 2MB.');
                    fileInput.value = '';
                    fileName.textContent = '';
                    submitBtn.disabled = true;
                }
            }
        }
        
        function formatFileSize(bytes) {
            if (bytes === 0) return '0 Bytes';
            const k = 1024;
            const sizes = ['Bytes', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        }
        
        // Form submission confirmation
        importForm.addEventListener('submit', function(e) {
            if (!fileInput.files.length) {
                e.preventDefault();
                alert('Please select a file to import.');
                return;
            }
            
            if (!confirm('Are you sure you want to import suppliers from this file?')) {
                e.preventDefault();
            }
        });
    });
</script>
@endsection