@extends('layouts.app')

@section('title', 'Store Information')

@section('content')
<!--start page wrapper -->
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">Settings</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('settings.index') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item active" aria-current="page">Store Information</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <div class="btn-group" role="group">
                    <button type="button" class="btn btn-outline-secondary" onclick="loadFromConfig()">
                        <i class="bx bx-import mr-1"></i>Load Defaults
                    </button>
                    <button type="button" class="btn btn-outline-info" onclick="viewConfiguration()">
                        <i class="bx bx-cog mr-1"></i>View Config
                    </button>
                    <button type="button" class="btn btn-primary" onclick="saveStoreInfo()" id="saveBtn">
                        <i class="bx bx-save mr-1"></i>Save Changes
                    </button>
                </div>
            </div>
        </div>
        <!--end breadcrumb-->

        @if(session('success'))
        <div class="alert alert-success border-0 alert-dismissible fade show" role="alert">
            <i class="bx bx-check-circle me-2"></i>
            {{ session('success') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        @endif

        @if(session('error'))
        <div class="alert alert-danger border-0 alert-dismissible fade show" role="alert">
            <i class="bx bx-error-circle me-2"></i>
            {{ session('error') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        @endif

        <div class="row">
            <!-- Store Information Form -->
            <div class="col-lg-8">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">
                            <i class="bx bx-store mr-2"></i>Store Details
                        </h5>
                        <hr>
                        
                        <form id="storeInfoForm" method="POST" action="{{ route('settings.store-information.store') }}" enctype="multipart/form-data">
                            @csrf
                            
                            <!-- Store Name & Logo -->
                            <div class="row mb-3">
                                <div class="col-md-8">
                                    <label for="store_name" class="form-label">Store Name <span class="text-danger">*</span></label>
                                    <input type="text" 
                                        class="form-control @error('store_name') is-invalid @enderror" 
                                        id="store_name" 
                                        name="store_name" 
                                        value="{{ old('store_name', $store->store_name ?? '') }}"
                                        required
                                        placeholder="Enter store name"
                                        oninput="updatePreview()">
                                    @error('store_name')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                                <div class="col-md-4">
                                    <label for="logo" class="form-label">Store Logo</label>
                                    <div class="d-flex align-items-center">
                                        <div class="flex-grow-1">
                                            <input type="file" 
                                                class="form-control @error('logo') is-invalid @enderror" 
                                                id="logo" 
                                                name="logo"
                                                accept="image/*"
                                                onchange="previewLogo(this)">
                                            @error('logo')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>
                                        <div class="ms-2">
                                            <button type="button" class="btn btn-outline-secondary" onclick="clearLogo()">
                                                <i class="bx bx-trash"></i>
                                            </button>
                                        </div>
                                    </div>
                                    <div class="form-text">Upload store logo (Max: 2MB)</div>
                                    
                                    <!-- Hidden field for logo URL -->
                                    <input type="hidden" id="logo_url" name="logo_url" value="{{ old('logo_url', $store->logo ?? '') }}">
                                </div>
                            </div>

                            <!-- Contact Information -->
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label for="telephone" class="form-label">Telephone <span class="text-danger">*</span></label>
                                    <input type="text" 
                                           class="form-control @error('telephone') is-invalid @enderror" 
                                           id="telephone" 
                                           name="telephone"
                                           value="{{ old('telephone', $store->telephone ?? '') }}"
                                           required
                                           placeholder="0700 000 000"
                                           oninput="updatePreview()">
                                    @error('telephone')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                                <div class="col-md-6">
                                    <label for="whatsapp_number" class="form-label">WhatsApp Number</label>
                                    <input type="text" 
                                           class="form-control @error('whatsapp_number') is-invalid @enderror" 
                                           id="whatsapp_number" 
                                           name="whatsapp_number"
                                           value="{{ old('whatsapp_number', $store->whatsapp_number ?? '') }}"
                                           placeholder="0700 000 000"
                                           oninput="updatePreview()">
                                    @error('whatsapp_number')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <!-- Email & Timezone -->
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <label for="email" class="form-label">Email Address</label>
                                    <input type="email" 
                                           class="form-control @error('email') is-invalid @enderror" 
                                           id="email" 
                                           name="email"
                                           value="{{ old('email', $store->email ?? '') }}"
                                           placeholder="store@example.com"
                                           oninput="updatePreview()">
                                    @error('email')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                                <div class="col-md-6">
                                    <label for="timezone" class="form-label">Timezone</label>
                                    <select class="form-select select2 @error('timezone') is-invalid @enderror" 
                                            id="timezone" 
                                            name="timezone"
                                            onchange="updatePreview()">
                                        <option value="">Select Timezone</option>
                                        <option value="Africa/Nairobi" {{ old('timezone', $store->timezone ?? '') == 'Africa/Nairobi' ? 'selected' : '' }}>Africa/Nairobi (EAT)</option>
                                        <option value="Africa/Dar_es_Salaam" {{ old('timezone', $store->timezone ?? '') == 'Africa/Dar_es_Salaam' ? 'selected' : '' }}>Africa/Dar_es_Salaam (EAT)</option>
                                        <option value="Africa/Kampala" {{ old('timezone', $store->timezone ?? '') == 'Africa/Kampala' ? 'selected' : '' }}>Africa/Kampala (EAT)</option>
                                        <option value="UTC" {{ old('timezone', $store->timezone ?? '') == 'UTC' ? 'selected' : '' }}>UTC</option>
                                    </select>
                                    @error('timezone')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <!-- Address Information -->
                            <div class="row mb-3">
                                <div class="col-12">
                                    <label for="address" class="form-label">Address <span class="text-danger">*</span></label>
                                    <textarea class="form-control @error('address') is-invalid @enderror" 
                                              id="address" 
                                              name="address" 
                                              rows="2"
                                              required
                                              placeholder="Enter store physical address"
                                              oninput="updatePreview()">{{ old('address', $store->address ?? '') }}</textarea>
                                    @error('address')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <div class="row mb-3">
                                <div class="col-md-4">
                                    <label for="city" class="form-label">City</label>
                                    <input type="text" 
                                           class="form-control @error('city') is-invalid @enderror" 
                                           id="city" 
                                           name="city"
                                           value="{{ old('city', $store->city ?? '') }}"
                                           placeholder="e.g., Nairobi"
                                           oninput="updatePreview()">
                                    @error('city')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                                <div class="col-md-4">
                                    <label for="country" class="form-label">Country</label>
                                    <input type="text" 
                                           class="form-control @error('country') is-invalid @enderror" 
                                           id="country" 
                                           name="country"
                                           value="{{ old('country', $store->country ?? '') }}"
                                           placeholder="e.g., Kenya"
                                           oninput="updatePreview()">
                                    @error('country')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                                <div class="col-md-4">
                                    <label for="currency" class="form-label">Currency</label>
                                    <select class="form-select @error('currency') is-invalid @enderror" 
                                            id="currency" 
                                            name="currency"
                                            onchange="updatePreview()">
                                        <option value="">Select Currency</option>
                                        <option value="KES" {{ old('currency', $store->currency ?? '') == 'KES' ? 'selected' : '' }}>KES - Kenyan Shilling</option>
                                        <option value="USD" {{ old('currency', $store->currency ?? '') == 'USD' ? 'selected' : '' }}>USD - US Dollar</option>
                                        <option value="EUR" {{ old('currency', $store->currency ?? '') == 'EUR' ? 'selected' : '' }}>EUR - Euro</option>
                                        <option value="GBP" {{ old('currency', $store->currency ?? '') == 'GBP' ? 'selected' : '' }}>GBP - British Pound</option>
                                    </select>
                                    @error('currency')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

<!-- Business Registration -->
                        <div class="row mb-3">
                            <div class="col-md-4">
                                <label for="company_pin" class="form-label">Company PIN</label>
                                <input type="text" 
                                    class="form-control @error('company_pin') is-invalid @enderror" 
                                    id="company_pin" 
                                    name="company_pin"
                                    value="{{ old('company_pin', $store->company_pin ?? '') }}"
                                    placeholder="P051234567K"
                                    oninput="updatePreview()">
                                @error('company_pin')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                            <div class="col-md-4">
                                <label for="vat_number" class="form-label">VAT Number</label>
                                <input type="text" 
                                    class="form-control @error('vat_number') is-invalid @enderror" 
                                    id="vat_number" 
                                    name="vat_number"
                                    value="{{ old('vat_number', $store->vat_number ?? '') }}"
                                    placeholder="00012345A"
                                    oninput="updatePreview()">
                                @error('vat_number')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                            <div class="col-md-4">
                                <label for="branch_code" class="form-label">Branch Code</label>
                                <input type="text" 
                                    class="form-control @error('branch_code') is-invalid @enderror" 
                                    id="branch_code" 
                                    name="branch_code"
                                    value="{{ old('branch_code', $store->branch_code ?? '') }}"
                                    placeholder="001"
                                    oninput="updatePreview()">
                                @error('branch_code')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                                </div>

                                <!-- Add this Business Registration field -->
                                <div class="row mb-3">
                                    <div class="col-md-12">
                                        <label for="business_registration" class="form-label">Business Registration Number</label>
                                        <input type="text" 
                                            class="form-control @error('business_registration') is-invalid @enderror" 
                                            id="business_registration" 
                                            name="business_registration"
                                            value="{{ old('business_registration', $store->business_registration ?? '') }}"
                                            placeholder="Business registration number">
                                        @error('business_registration')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>

                <!-- Additional Business Info -->
                <div class="row mb-3">
                    <div class="col-md-6">
                        <label for="kra_pin" class="form-label">KRA PIN</label>
                        <input type="text" 
                            class="form-control @error('kra_pin') is-invalid @enderror" 
                            id="kra_pin" 
                            name="kra_pin"
                            value="{{ old('kra_pin', $store->kra_pin ?? '') }}"
                            placeholder="KRA personal identification number">
                        @error('kra_pin')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    <div class="col-md-6">
                        <label for="permit_number" class="form-label">Permit Number</label>
                        <input type="text" 
                            class="form-control @error('permit_number') is-invalid @enderror" 
                            id="permit_number" 
                            name="permit_number"
                            value="{{ old('permit_number', $store->permit_number ?? '') }}"
                            placeholder="Business permit number">
                        @error('permit_number')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                </div>



                            <!-- Business Hours -->
                            <div class="row mb-3">
                                <div class="col-12">
                                    <label class="form-label">Business Hours</label>
                                    <div class="row g-2">
                                        <div class="col-md-6">
                                            <div class="input-group">
                                                <span class="input-group-text">Opening</span>
                                                @php
                                                    // Safely get business hours open time
                                                    $openTime = '08:00';
                                                    if (isset($store->business_hours)) {
                                                        if (is_string($store->business_hours)) {
                                                            $hours = json_decode($store->business_hours, true);
                                                            $openTime = $hours['open'] ?? '08:00';
                                                        } elseif (is_array($store->business_hours)) {
                                                            $openTime = $store->business_hours['open'] ?? '08:00';
                                                        } elseif (is_object($store->business_hours)) {
                                                            $openTime = $store->business_hours->open ?? '08:00';
                                                        }
                                                    }
                                                @endphp
                                                <input type="time" 
                                                    class="form-control" 
                                                    id="business_hours_open" 
                                                    name="business_hours_open"
                                                    value="{{ $openTime }}"
                                                    onchange="updateBusinessHours()">
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="input-group">
                                                <span class="input-group-text">Closing</span>
                                                @php
                                                    // Safely get business hours close time
                                                    $closeTime = '20:00';
                                                    if (isset($store->business_hours)) {
                                                        if (is_string($store->business_hours)) {
                                                            $hours = json_decode($store->business_hours, true);
                                                            $closeTime = $hours['close'] ?? '20:00';
                                                        } elseif (is_array($store->business_hours)) {
                                                            $closeTime = $store->business_hours['close'] ?? '20:00';
                                                        } elseif (is_object($store->business_hours)) {
                                                            $closeTime = $store->business_hours->close ?? '20:00';
                                                        }
                                                    }
                                                @endphp
                                                <input type="time" 
                                                    class="form-control" 
                                                    id="business_hours_close" 
                                                    name="business_hours_close"
                                                    value="{{ $closeTime }}"
                                                    onchange="updateBusinessHours()">
                                            </div>
                                        </div>
                                    </div>
                                    <div class="form-text">Set your store operating hours (24-hour format)</div>
                                </div>
                            </div>

                            <!-- Hidden field for business hours JSON -->
                            <input type="hidden" id="business_hours" name="business_hours" 
                                   value="@php
                                        // Safely encode business hours
                                        $businessHours = ['open' => '08:00', 'close' => '20:00'];
                                        if (isset($store->business_hours)) {
                                            if (is_string($store->business_hours)) {
                                                $decoded = json_decode($store->business_hours, true);
                                                if ($decoded) {
                                                    $businessHours = $decoded;
                                                }
                                            } elseif (is_array($store->business_hours) || is_object($store->business_hours)) {
                                                $businessHours = (array)$store->business_hours;
                                            }
                                        }
                                        echo json_encode($businessHours);
                                   @endphp">
                        </form>
                    </div>
                </div>
            </div>

            <!-- Preview & Information -->
            <div class="col-lg-4">
                
                <!-- Preview Card -->
                <div class="card mb-3">
                    <div class="card-body">
                        <h5 class="card-title">
                            <i class="bx bx-show mr-2"></i>Live Preview
                        </h5>
                        <hr>
                        
                        <div class="text-center mb-3">
                            <div id="logoPreview" class="store-logo-preview mb-3">
                                @php
                                    $logoUrl = $store->logo ?? '';
                                    $hasLogo = !empty($logoUrl);
                                    $logoPath = '';
                                    
                                    if ($hasLogo) {
                                        // Check if it's a URL or a local file
                                        if (filter_var($logoUrl, FILTER_VALIDATE_URL)) {
                                            // It's a full URL
                                            $logoPath = $logoUrl;
                                        } else {
                                            // It's a local file path - prepend storage path
                                            $logoPath = asset('storage/' . $logoUrl);
                                        }
                                    }
                                @endphp
                                
                                @if($hasLogo)
                                    <img src="{{ $logoPath }}" alt="Store Logo" class="img-fluid rounded" 
                                        style="max-height: 60px;" 
                                        onerror="this.onerror=null; this.style.display='none'; document.getElementById('logo-placeholder').style.display='block';">
                                @endif
                                <div id="logo-placeholder" class="logo-placeholder" style="{{ $hasLogo ? 'display: none;' : '' }}">
                                    <i class="bx bx-store text-muted" style="font-size: 3rem;"></i>
                                </div>
                            </div>
                            <h4 id="previewStoreName" class="mb-1">{{ $store->store_name ?? 'Store Name' }}</h4>
                            <p class="text-muted mb-2" id="previewAddress">
                                @php
                                    $addressParts = [];
                                    if (!empty($store->address)) $addressParts[] = $store->address;
                                    if (!empty($store->city)) $addressParts[] = $store->city;
                                    if (!empty($store->country)) $addressParts[] = $store->country;
                                    echo implode(', ', $addressParts) ?: 'Store Address';
                                @endphp
                            </p>
                        </div>
                        
                        <!-- ... rest of your preview code ... -->
                    </div>
                </div>

                <!-- System Status -->
                <div class="card mb-3">
                    <div class="card-body">
                        <h5 class="card-title">
                            <i class="bx bx-data mr-2"></i>System Status
                        </h5>
                        <hr>
                        
                        <div class="mb-3">
                            <div class="d-flex justify-content-between align-items-center mb-2">
                                <span>Data Source:</span>
                                <span class="badge bg-{{ isset($store->is_fallback) && $store->is_fallback ? 'warning' : 'success' }}">
                                    {{ isset($store->is_fallback) && $store->is_fallback ? 'Config Fallback' : 'Database' }}
                                </span>
                            </div>
                            
                            <div class="d-flex justify-content-between align-items-center mb-2">
                                <span>Cache Status:</span>
                                <span class="badge bg-info">Active</span>
                            </div>
                            
                            <div class="d-flex justify-content-between align-items-center">
                                <span>Last Updated:</span>
                                <span id="lastUpdated">
                                    @php
                                        if (isset($store->updated_at) && $store->updated_at) {
                                            echo $store->updated_at->format('Y-m-d H:i');
                                        } elseif (isset($store->created_at) && $store->created_at) {
                                            echo $store->created_at->format('Y-m-d H:i');
                                        } else {
                                            echo 'Never';
                                        }
                                    @endphp
                                </span>
                            </div>
                        </div>
                        
                        <div class="d-grid gap-2">
                            <button type="button" class="btn btn-outline-danger" onclick="clearCache()">
                                <i class="bx bx-trash mr-1"></i>Clear Cache
                            </button>
                            <button type="button" class="btn btn-outline-info" onclick="viewHealthStatus()">
                                <i class="bx bx-heart mr-1"></i>Health Check
                            </button>
                        </div>
                    </div>
                </div>

                <!-- Quick Actions -->
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">
                            <i class="bx bx-bolt mr-2"></i>Quick Actions
                        </h5>
                        <hr>
                        
                        <div class="d-grid gap-2">
                            <button type="button" class="btn btn-outline-primary" onclick="testReceiptSettings()">
                                <i class="bx bx-printer mr-1"></i>Test Receipt
                            </button>
                            <button type="button" class="btn btn-outline-secondary" onclick="resetForm()">
                                <i class="bx bx-reset mr-1"></i>Reset Form
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Information Alert -->
        <div class="row mt-3">
            <div class="col-12">
                <div class="alert alert-light border">
                    <div class="d-flex">
                        <div class="me-3">
                            <i class="bx bx-info-circle" style="font-size: 1.5rem;"></i>
                        </div>
                        <div>
                            <h6 class="alert-heading mb-2">Store Information Usage</h6>
                            <p class="mb-0 small">
                                This information is used in:
                                <ul class="mb-0 small">
                                    <li>Receipts and invoices</li>
                                    <li>POS system display</li>
                                    <li>Customer communications</li>
                                    <li>Tax reporting (VAT, KRA)</li>
                                    <li>eTIMS integration</li>
                                </ul>
                            </p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
<!--end page wrapper -->

<!-- Modals -->
<div class="modal fade" id="configModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="bx bx-cog mr-2"></i>Configuration Settings</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="alert alert-light border mb-3">
                    <i class="bx bx-info-circle mr-2"></i>
                    This shows the configuration from <code>config/pos.php</code>
                </div>
                <pre id="configContent" class="bg-light p-3 rounded" style="max-height: 400px; overflow-y: auto; font-size: 12px;"></pre>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                <button type="button" class="btn btn-primary" onclick="copyConfig()">
                    <i class="bx bx-copy mr-1"></i>Copy
                </button>
            </div>
        </div>
    </div>
</div>

<div class="modal fade" id="healthModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="bx bx-heart mr-2"></i>System Health Status</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div id="healthStatus" class="text-center">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    <p class="mt-2">Checking system health...</p>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                <button type="button" class="btn btn-primary" onclick="refreshHealth()">
                    <i class="bx bx-refresh mr-1"></i>Refresh
                </button>
            </div>
        </div>
    </div>
</div>

<div class="modal fade" id="receiptModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="bx bx-receipt mr-2"></i>Receipt Preview</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="alert alert-warning border-0 mb-3">
                    <i class="bx bx-error-circle mr-2"></i>
                    This is a test receipt preview. Actual receipts will include sale data.
                </div>
                <iframe id="receiptFrame" style="width: 100%; height: 500px; border: 1px solid #ddd;" 
                        title="Receipt Preview"></iframe>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                <button type="button" class="btn btn-primary" onclick="printReceipt()">
                    <i class="bx bx-printer mr-1"></i>Print
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script src="{{ asset('assets/plugins/select2/js/select2.min.js') }}"></script>
<script>
// Global variables
let lastSaveTime = null;

// Initialize Select2
$(document).ready(function() {
    $('.select2').select2({
        theme: 'bootstrap4',
        width: '100%'
    });
    
    // Update preview on input
    $('input, select, textarea').on('change input', function() {
        updatePreview();
    });
    
    // Initial preview
    updatePreview();
    
    // Auto-save warning
    $(window).on('beforeunload', function() {
        const hasUnsavedChanges = checkFormChanges();
        if (hasUnsavedChanges) {
            return 'You have unsaved changes. Are you sure you want to leave?';
        }
    });
});

// Preview logo before upload
function previewLogo(input) {
    const preview = $('#logoPreview');
    const placeholder = preview.find('.logo-placeholder');
    let img = preview.find('img');
    
    if (input.files && input.files[0]) {
        const reader = new FileReader();
        
        reader.onload = function(e) {
            if (img.length === 0) {
                preview.prepend(`<img src="${e.target.result}" alt="Store Logo" class="img-fluid rounded" style="max-height: 60px;">`);
                img = preview.find('img');
            } else {
                img.attr('src', e.target.result).show();
            }
            placeholder.hide();
            
            // Update logo URL field with temporary base64 data
            $('#logo_url').val(e.target.result);
        }
        
        reader.readAsDataURL(input.files[0]);
    }
}

// Clear logo selection
function clearLogo() {
    $('#logo').val('');
    $('#logo_url').val('');
    
    const preview = $('#logoPreview');
    const placeholder = preview.find('.logo-placeholder');
    const img = preview.find('img');
    
    if (img.length > 0) img.hide();
    placeholder.show();
    updatePreview();
}

// Update business hours JSON
function updateBusinessHours() {
    const openTime = $('#business_hours_open').val() || '08:00';
    const closeTime = $('#business_hours_close').val() || '20:00';
    const businessHours = {
        open: openTime,
        close: closeTime
    };
    $('#business_hours').val(JSON.stringify(businessHours));
    updatePreview();
}

// Update preview
function updatePreview() {
    // Store name
    const storeName = $('#store_name').val();
    if (storeName) {
        $('#previewStoreName').text(storeName);
    }
    
    // Address
    const address = $('#address').val();
    const city = $('#city').val();
    const country = $('#country').val();
    let addressText = address || 'Store Address';
    if (city) addressText += ', ' + city;
    if (country) addressText += ', ' + country;
    $('#previewAddress').text(addressText);
    
    // Phone numbers
    const phone = $('#telephone').val();
    $('#previewPhone').text(phone || '0700 000 000');
    
    const whatsapp = $('#whatsapp_number').val();
    if (whatsapp) {
        $('#previewWhatsApp').text(whatsapp).closest('div').show();
    } else {
        $('#previewWhatsApp').closest('div').hide();
    }
    
    // Email
    const email = $('#email').val();
    if (email) {
        $('#previewEmail').text(email).closest('div').show();
    } else {
        $('#previewEmail').closest('div').hide();
    }
    
    // PIN
    const pin = $('#company_pin').val();
    $('#previewPin').text(pin ? 'PIN: ' + pin : 'PIN: Not set');
    
    // Business hours
    const openTime = $('#business_hours_open').val() || '08:00';
    const closeTime = $('#business_hours_close').val() || '20:00';
    $('#previewBusinessHours').text(`${openTime} - ${closeTime}`);
    
    // Currency
    const currency = $('#currency').val();
    $('#previewCurrency').text(currency || 'KES');
    
    // Logo preview from URL
    const logoUrl = $('#logo_url').val();
    updateLogoPreview(logoUrl);
}

// Update logo preview
function updateLogoPreview(logoUrl) {
    const logoPreview = $('#logoPreview');
    const placeholder = logoPreview.find('.logo-placeholder');
    let img = logoPreview.find('img');
    
    if (logoUrl) {
        if (img.length === 0) {
            logoPreview.prepend(`<img src="${logoUrl}" alt="Store Logo" class="img-fluid rounded" style="max-height: 60px;">`);
            img = logoPreview.find('img');
        } else {
            img.attr('src', logoUrl).show();
        }
        placeholder.hide();
        img.on('error', function() {
            $(this).hide();
            placeholder.show();
        });
    } else {
        if (img.length > 0) img.hide();
        placeholder.show();
    }
}

// Save store information
function saveStoreInfo() {
    const form = $('#storeInfoForm');
    const formData = new FormData(form[0]);
    
    // Update business hours
    updateBusinessHours();
    const businessHours = $('#business_hours').val();
    formData.set('business_hours', businessHours);
    
    // Get logo preview (base64 data)
    const logoPreview = $('#logoPreview img');
    if (logoPreview.length > 0 && logoPreview.is(':visible')) {
        const logoUrl = logoPreview.attr('src');
        // Check if it's a base64 string
        if (logoUrl && logoUrl.startsWith('data:image/')) {
            formData.set('logo_url', logoUrl);
        }
    }
    
    // Validate required fields
    let isValid = true;
    const requiredFields = ['store_name', 'telephone', 'address'];
    
    requiredFields.forEach(field => {
        const input = $(`#${field}`);
        if (!input.val().trim()) {
            input.addClass('is-invalid');
            isValid = false;
        } else {
            input.removeClass('is-invalid');
        }
    });
    
    if (!isValid) {
        showNotification('Please fill in all required fields!', 'error');
        return;
    }
    
    // Validate email
    const email = $('#email').val();
    if (email && !isValidEmail(email)) {
        $('#email').addClass('is-invalid');
        showNotification('Please enter a valid email address!', 'error');
        return;
    }
    
    // Validate file size if uploaded
    const logoFile = $('#logo')[0].files[0];
    if (logoFile && logoFile.size > 2 * 1024 * 1024) { // 2MB
        showNotification('Logo file size must be less than 2MB!', 'error');
        return;
    }
    
    // Disable button and show loading
    const saveBtn = $('#saveBtn');
    const originalText = saveBtn.html();
    saveBtn.prop('disabled', true);
    saveBtn.html('<i class="bx bx-loader-circle bx-spin mr-1"></i>Saving...');
    
    // Get CSRF token
    const csrfToken = $('meta[name="csrf-token"]').attr('content');
    
    // Submit form via AJAX for file upload
    $.ajax({
        url: form.attr('action'),
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        headers: {
            'X-CSRF-TOKEN': csrfToken,
            'X-Requested-With': 'XMLHttpRequest'
        },
        success: function(response) {
            if (response.redirect) {
                // Handle redirect from controller
                window.location.href = response.redirect;
                return;
            }
            
            if (response.success) {
                showNotification(response.message || 'Store information saved successfully!', 'success');
                lastSaveTime = new Date();
                
                // Refresh page to show new logo
                setTimeout(() => window.location.reload(), 1500);
            } else {
                showNotification(response.message || 'Failed to save store information!', 'error');
            }
        },
        error: function(xhr) {
            let message = 'Failed to save store information!';
            
            if (xhr.responseJSON) {
                if (xhr.responseJSON.message) {
                    message = xhr.responseJSON.message;
                }
                
                if (xhr.responseJSON.errors) {
                    // Show validation errors
                    const errors = xhr.responseJSON.errors;
                    message = Object.values(errors)[0][0];
                    
                    // Mark invalid fields
                    $.each(errors, function(field, messages) {
                        const input = $(`#${field}`);
                        input.addClass('is-invalid');
                        const errorDiv = input.siblings('.invalid-feedback');
                        if (errorDiv.length) {
                            errorDiv.text(messages[0]);
                        } else {
                            input.after(`<div class="invalid-feedback">${messages[0]}</div>`);
                        }
                    });
                }
            }
            
            showNotification(message, 'error');
        },
        complete: function() {
            // Re-enable button
            saveBtn.prop('disabled', false);
            saveBtn.html(originalText);
        }
    });
}

// Load defaults from config
function loadFromConfig() {
    Swal.fire({
        title: 'Load Defaults?',
        text: 'This will replace your current values with configuration defaults.',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#3085d6',
        cancelButtonColor: '#d33',
        confirmButtonText: 'Yes, load defaults!'
    }).then((result) => {
        if (result.isConfirmed) {
            showLoading('Loading defaults...');
            
            fetch('{{ route("settings.store-information.config") }}')
                .then(response => response.json())
                .then(data => {
                    hideLoading();
                    
                    if (data.success) {
                        const config = data.config.store;
                        
                        // Fill form with config values
                        $('#store_name').val(config.name || '');
                        $('#telephone').val(config.phone || '');
                        $('#whatsapp_number').val(config.whatsapp || '');
                        $('#email').val(config.email || '');
                        $('#address').val(config.address || '');
                        $('#city').val(config.city || '');
                        $('#country').val(config.country || '');
                        $('#currency').val(config.currency || '');
                        $('#timezone').val(config.timezone || '').trigger('change');
                        $('#company_pin').val(config.pin || '');
                        $('#vat_number').val(config.vat_number || '');
                        $('#branch_code').val(config.branch_code || '');
                        $('#kra_pin').val(config.kra_pin || '');
                        $('#permit_number').val(config.permit_number || '');
                        
                        // Business hours
                        if (config.business_hours) {
                            $('#business_hours_open').val(config.business_hours.open || '08:00');
                            $('#business_hours_close').val(config.business_hours.close || '20:00');
                        }
                        
                        updatePreview();
                        updateBusinessHours();
                        showNotification('Defaults loaded!', 'success');
                    } else {
                        showNotification('Failed to load config!', 'error');
                    }
                })
                .catch(error => {
                    hideLoading();
                    showNotification('Error loading defaults!', 'error');
                });
        }
    });
}

// Clear cache
function clearCache() {
    Swal.fire({
        title: 'Clear Cache?',
        text: 'This will reload data from the database.',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#d33',
        cancelButtonColor: '#3085d6',
        confirmButtonText: 'Yes, clear cache!'
    }).then((result) => {
        if (result.isConfirmed) {
            showLoading('Clearing cache...');
            
            fetch('{{ route("settings.store-information.clear-cache") }}', {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': '{{ csrf_token() }}',
                    'Accept': 'application/json'
                }
            })
            .then(response => response.json())
            .then(data => {
                hideLoading();
                
                if (data.success) {
                    showNotification('Cache cleared!', 'success');
                    $('#lastUpdated').text(new Date().toLocaleString());
                    
                    // Refresh page
                    setTimeout(() => window.location.reload(), 1000);
                } else {
                    showNotification('Failed to clear cache!', 'error');
                }
            })
            .catch(error => {
                hideLoading();
                showNotification('Error clearing cache!', 'error');
            });
        }
    });
}

// View configuration
function viewConfiguration() {
    showLoading('Loading configuration...');
    
    fetch('{{ route("settings.store-information.config") }}')
        .then(response => response.json())
        .then(data => {
            hideLoading();
            
            if (data.success) {
                const config = JSON.stringify(data.config, null, 2);
                $('#configContent').text(config);
                new bootstrap.Modal('#configModal').show();
            } else {
                showNotification('Failed to load config!', 'error');
            }
        })
        .catch(error => {
            hideLoading();
            showNotification('Error loading config!', 'error');
        });
}

// Copy configuration
function copyConfig() {
    const configText = $('#configContent').text();
    navigator.clipboard.writeText(configText).then(() => {
        showNotification('Copied to clipboard!', 'success');
    }).catch(err => {
        showNotification('Failed to copy!', 'error');
    });
}

// View health status
function viewHealthStatus() {
    new bootstrap.Modal('#healthModal').show();
    refreshHealth();
}

function refreshHealth() {
    $('#healthStatus').html(`
        <div class="text-center">
            <div class="spinner-border text-primary" role="status">
                <span class="visually-hidden">Loading...</span>
            </div>
            <p class="mt-2">Checking system health...</p>
        </div>
    `);
    
    fetch('{{ route("settings.store-information.health") }}')
        .then(response => response.json())
        .then(data => {
            let html = '';
            if (data.success) {
                const health = data.health;
                
                html = `
                    <div class="text-start">
                        <div class="mb-2">
                            <strong>Database:</strong>
                            <span class="badge bg-${health.database === 'has_data' ? 'success' : 'danger'} ms-2">
                                ${health.database === 'has_data' ? 'Connected' : 'No Data'}
                            </span>
                        </div>
                        <div class="mb-2">
                            <strong>Using Fallback:</strong>
                            <span class="badge bg-${health.using_fallback ? 'warning' : 'success'} ms-2">
                                ${health.using_fallback ? 'Yes' : 'No'}
                            </span>
                        </div>
                        <div class="mb-2">
                            <strong>Cache Status:</strong>
                            <span class="badge bg-${health.cache.general === 'cached' ? 'success' : 'secondary'} ms-2">
                                ${health.cache.general}
                            </span>
                        </div>
                        <div class="mb-3">
                            <strong>Last Updated:</strong>
                            <span>${health.last_updated || 'Never'}</span>
                        </div>
                        ${health.using_fallback ? `
                        <div class="alert alert-warning">
                            <i class="bx bx-error-circle mr-2"></i>
                            Currently using fallback configuration. Save store information to database.
                        </div>
                        ` : ''}
                    </div>
                `;
            } else {
                html = '<div class="alert alert-danger">Failed to load health status</div>';
            }
            
            $('#healthStatus').html(html);
        })
        .catch(error => {
            $('#healthStatus').html('<div class="alert alert-danger">Error loading health status</div>');
        });
}

// Test receipt settings
function testReceiptSettings() {
    showLoading('Opening test receipt...');
    
    // Directly open the test receipt route
    const testUrl = '{{ route("settings.receipt.test") }}';
    
    // Open in new window
    const newWindow = window.open(testUrl, '_blank', 'width=800,height=600');
    
    hideLoading();
    
    if (!newWindow) {
        // If popup blocked, show message
        showNotification('Please allow popups to view test receipt', 'warning');
        
        // Alternative: open in iframe modal
        const receiptFrame = $('#receiptFrame');
        receiptFrame.attr('src', testUrl);
        new bootstrap.Modal('#receiptModal').show();
    }
}

// Print receipt
function printReceipt() {
    const receiptFrame = $('#receiptFrame')[0];
    if (receiptFrame && receiptFrame.contentWindow) {
        receiptFrame.contentWindow.print();
    }
}

// Reset form
function resetForm() {
    Swal.fire({
        title: 'Reset Form?',
        text: 'This will reset all changes to original values.',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#3085d6',
        cancelButtonColor: '#d33',
        confirmButtonText: 'Yes, reset!'
    }).then((result) => {
        if (result.isConfirmed) {
            window.location.reload();
        }
    });
}

// Check form changes
function checkFormChanges() {
    const form = $('#storeInfoForm');
    let hasChanges = false;
    
    form.find('input, select, textarea').each(function() {
        const defaultValue = $(this).data('default') || '';
        const currentValue = $(this).val() || '';
        if (defaultValue !== currentValue) {
            hasChanges = true;
            return false; // Break loop
        }
    });
    
    return hasChanges;
}

// Utility functions
function isValidEmail(email) {
    const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return re.test(email);
}

function isValidUrl(string) {
    try {
        new URL(string);
        return true;
    } catch (_) {
        return false;
    }
}

function showNotification(message, type = 'info') {
    const Toast = Swal.mixin({
        toast: true,
        position: 'top-end',
        showConfirmButton: false,
        timer: 3000,
        timerProgressBar: true
    });
    
    Toast.fire({
        icon: type,
        title: message
    });
}

function showLoading(message) {
    Swal.fire({
        title: message,
        allowOutsideClick: false,
        showConfirmButton: false,
        willOpen: () => {
            Swal.showLoading();
        }
    });
}

function hideLoading() {
    Swal.close();
}

// Keyboard shortcuts
$(document).on('keydown', function(e) {
    // Ctrl/Cmd + S to save
    if ((e.ctrlKey || e.metaKey) && e.key === 's') {
        e.preventDefault();
        saveStoreInfo();
    }
    
    // Escape to reset form
    if (e.key === 'Escape') {
        resetForm();
    }
});
</script>

<style>
.store-logo-preview {
    height: 60px;
    display: flex;
    align-items: center;
    justify-content: center;
    border: 2px dashed #dee2e6;
    border-radius: 8px;
    padding: 10px;
    background: #f8f9fa;
}

.logo-placeholder {
    color: #6c757d;
}

.store-contact-preview .bx {
    min-width: 20px;
    text-align: center;
}

#previewStoreName, #previewAddress, #previewPhone, 
#previewEmail, #previewBusinessHours, #previewCurrency {
    transition: all 0.3s ease;
}

.badge {
    font-size: 0.75em;
    padding: 0.35em 0.65em;
}

.alert-light {
    background-color: #f8f9fa;
    border-color: #e9ecef;
}

/* Responsive adjustments */
@media (max-width: 768px) {
    .btn-group {
        flex-wrap: wrap;
    }
    
    .btn-group .btn {
        margin-bottom: 5px;
    }
}
</style>
@endsection