@extends('layouts.app')

@section('title', 'Receipt Settings')


@section('content')
<!--start page wrapper -->
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">Settings</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('settings.index') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item active" aria-current="page">Receipt Settings</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <div class="btn-group" role="group">
                    <button type="button" class="btn btn-outline-primary" data-bs-toggle="modal" data-bs-target="#createModal">
                        <i class="bx bx-plus mr-1"></i>New Setting
                    </button>
                    <button type="button" class="btn btn-outline-success" onclick="testReceipt()">
                        <i class="bx bx-printer mr-1"></i>Test Receipt
                    </button>
                    <button type="button" class="btn btn-outline-info" onclick="viewDefaultConfig()">
                        <i class="bx bx-cog mr-1"></i>View Config
                    </button>
                </div>
            </div>
        </div>
        <!--end breadcrumb-->

        @if(session('success'))
        <div class="alert alert-success border-0 alert-dismissible fade show" role="alert">
            <i class="bx bx-check-circle me-2"></i>
            {{ session('success') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        @endif

        @if(session('error'))
        <div class="alert alert-danger border-0 alert-dismissible fade show" role="alert">
            <i class="bx bx-error-circle me-2"></i>
            {{ session('error') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        @endif

        <!-- Settings List -->
        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title mb-3">
                            <i class="bx bx-receipt mr-2"></i>Receipt Settings
                        </h5>
                        
                        @if($settings->isEmpty())
                        <div class="alert alert-info border-0">
                            <i class="bx bx-info-circle mr-2"></i>
                            No receipt settings found. <a href="#" data-bs-toggle="modal" data-bs-target="#createModal">Create your first receipt setting</a>.
                        </div>
                        @else
                        <div class="table-responsive">
                            <table class="table table-hover table-striped">
                                <thead>
                                    <tr>
                                        <th>Name</th>
                                        <th>Paper Size</th>
                                        <th>Default</th>
                                        <th>Status</th>
                                        <th>Created</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach($settings as $setting)
                                    <tr>
                                        <td>
                                            <strong>{{ $setting->name }}</strong>
                                            @if($setting->is_default)
                                            <span class="badge bg-primary ms-2">Default</span>
                                            @endif
                                        </td>
                                        <td>{{ $setting->paper_size }}</td>
                                        <td>
                                            @if($setting->is_default)
                                            <span class="badge bg-success">Default</span>
                                            @else
                                            <button type="button" class="btn btn-sm btn-outline-secondary" onclick="setAsDefault({{ $setting->id }})">
                                                Set as Default
                                            </button>
                                            @endif
                                        </td>
                                        <td>
                                            @if($setting->is_active)
                                            <span class="badge bg-success">Active</span>
                                            @else
                                            <span class="badge bg-danger">Inactive</span>
                                            @endif
                                        </td>
                                        <td>{{ $setting->created_at->format('Y-m-d') }}</td>
                                        <td>
                                            <div class="btn-group" role="group">
                                                <button type="button" class="btn btn-sm btn-outline-info" onclick="editSetting({{ $setting->id }})" title="Edit">
                                                    <i class="bx bx-edit"></i>
                                                </button>
                                                <button type="button" class="btn btn-sm btn-outline-primary" onclick="testSetting({{ $setting->id }})" title="Test">
                                                    <i class="bx bx-printer"></i>
                                                </button>
                                                <button type="button" class="btn btn-sm btn-outline-danger" onclick="deleteSetting({{ $setting->id }})" title="Delete">
                                                    <i class="bx bx-trash"></i>
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                        @endif
                    </div>
                </div>
            </div>
        </div>

        <!-- Preview Card -->
        <div class="row mt-4">
            <div class="col-lg-6">
                <div class="card">
                    <div class="card-header bg-transparent">
                        <h5 class="mb-0"><i class="bx bx-show mr-2"></i>Receipt Preview</h5>
                    </div>
                    <div class="card-body">
                        <div class="receipt-preview">
                            <div class="receipt-header text-center mb-3">
                                @if($defaultSetting && $defaultSetting->print_store_logo && $defaultSetting->logo_path)
                                <img src="{{ $defaultSetting->logo_path }}" alt="Store Logo" class="mb-2" style="max-height: 60px;">
                                @endif
                                @if($defaultSetting && $defaultSetting->print_store_name)
                                <h4 class="mb-1">{{ config('app.name') }}</h4>
                                @endif
                                @if($defaultSetting && $defaultSetting->print_store_address)
                                <p class="small text-muted mb-1">123 Main Street, Nairobi</p>
                                @endif
                                @if($defaultSetting && $defaultSetting->print_store_phone)
                                <p class="small text-muted mb-1">Phone: 0700 000 000</p>
                                @endif
                                @if($defaultSetting && $defaultSetting->print_vat_number)
                                <p class="small text-muted mb-1">VAT: VAT-001</p>
                                @endif
                                @if($defaultSetting && $defaultSetting->print_branch_code)
                                <p class="small text-muted mb-1">Branch: 001</p>
                                @endif
                            </div>
                            
                            <div class="receipt-body">
                                <div class="border-top border-bottom py-2">
                                    @if($defaultSetting && $defaultSetting->print_cashier_name)
                                    <div class="d-flex justify-content-between small">
                                        <span>Cashier:</span>
                                        <span>{{ auth()->user()->name }}</span>
                                    </div>
                                    @endif
                                    @if($defaultSetting && $defaultSetting->print_date_time)
                                    <div class="d-flex justify-content-between small">
                                        <span>Date:</span>
                                        <span>{{ now()->format('Y-m-d H:i') }}</span>
                                    </div>
                                    @endif
                                    @if($defaultSetting && $defaultSetting->print_transaction_number)
                                    <div class="d-flex justify-content-between small">
                                        <span>Transaction:</span>
                                        <span>TEST-001</span>
                                    </div>
                                    @endif
                                </div>
                                
                                <div class="my-3">
                                    <div class="d-flex justify-content-between border-bottom py-2">
                                        <span>Test Product x 2</span>
                                        <span>KSh 1,000</span>
                                    </div>
                                    <div class="d-flex justify-content-between border-bottom py-2">
                                        <span>Another Product x 1</span>
                                        <span>KSh 500</span>
                                    </div>
                                </div>
                                
                                @if($defaultSetting && $defaultSetting->print_tax_breakdown)
                                <div class="border-top border-bottom py-2">
                                    <div class="d-flex justify-content-between small">
                                        <span>Subtotal:</span>
                                        <span>KSh 1,500</span>
                                    </div>
                                    <div class="d-flex justify-content-between small">
                                        <span>Tax (16%):</span>
                                        <span>KSh 240</span>
                                    </div>
                                </div>
                                @endif
                                
                                <div class="d-flex justify-content-between fw-bold py-2">
                                    <span>Total:</span>
                                    <span>KSh 1,740</span>
                                </div>
                                
                                @if($defaultSetting && $defaultSetting->print_payment_method)
                                <div class="d-flex justify-content-between small">
                                    <span>Payment:</span>
                                    <span>Cash</span>
                                </div>
                                @endif
                                
                                @if($defaultSetting && $defaultSetting->print_change_amount)
                                <div class="d-flex justify-content-between small">
                                    <span>Change:</span>
                                    <span>KSh 260</span>
                                </div>
                                @endif
                                
                                @if($defaultSetting && $defaultSetting->print_qr_code)
                                <div class="text-center my-3">
                                    <div class="d-inline-block border p-2">
                                        <div class="text-muted small">QR Code</div>
                                    </div>
                                </div>
                                @endif
                            </div>
                            
                            @if($defaultSetting && $defaultSetting->print_footer)
                            <div class="receipt-footer text-center mt-3">
                                <p class="small text-muted mb-0">{{ $defaultSetting->footer_text ?? 'Thank you for shopping with us!' }}</p>
                            </div>
                            @endif
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="col-lg-6">
                <div class="card">
                    <div class="card-header bg-transparent">
                        <h5 class="mb-0"><i class="bx bx-info-circle mr-2"></i>Receipt Information</h5>
                    </div>
                    <div class="card-body">
                        <div class="mb-3">
                            <strong class="d-block mb-1">Current Default Setting:</strong>
                            <span class="text-muted">{{ $defaultSetting ? $defaultSetting->name : 'No default setting' }}</span>
                        </div>
                        
                        <div class="mb-3">
                            <strong class="d-block mb-1">Paper Size:</strong>
                            <span class="text-muted">{{ $defaultSetting ? $defaultSetting->paper_size : '80mm' }}</span>
                        </div>
                        
                        <div class="mb-3">
                            <strong class="d-block mb-1">Characters Per Line:</strong>
                            <span class="text-muted">{{ $defaultSetting ? $defaultSetting->chars_per_line : '42' }}</span>
                        </div>
                        
                        <div class="mb-3">
                            <strong class="d-block mb-1">Copies:</strong>
                            <span class="text-muted">{{ $defaultSetting ? $defaultSetting->copies : '1' }}</span>
                        </div>
                        
                        <div class="alert alert-light border">
                            <i class="bx bx-info-circle mr-2"></i>
                            <strong>Receipt Settings Guide:</strong>
                            <ul class="mb-0 mt-2 ps-3">
                                <li>Create multiple settings for different printers</li>
                                <li>Test settings before making them default</li>
                                <li>80mm paper works best with thermal printers</li>
                                <li>A4/A5 paper for invoice-style receipts</li>
                            </ul>
                        </div>
                        
                        <div class="d-grid gap-2">
                            <button type="button" class="btn btn-primary" onclick="loadReceiptSettings()">
                                <i class="bx bx-refresh mr-1"></i>Refresh Settings
                            </button>
                            <button type="button" class="btn btn-outline-danger" onclick="clearReceiptCache()">
                                <i class="bx bx-trash mr-1"></i>Clear Cache
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
<!--end page wrapper -->

<!-- Create/Edit Modal -->
<div class="modal fade" id="receiptModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="modalTitle"><i class="bx bx-receipt mr-2"></i>Receipt Setting</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <form id="receiptForm">
                    @csrf
                    <input type="hidden" id="setting_id" name="id">
                    
                    <!-- Basic Information -->
                    <div class="row mb-3">
                        <div class="col-md-8">
                            <label for="name" class="form-label">Setting Name <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" id="name" name="name" required>
                        </div>
                        <div class="col-md-4">
                            <label for="paper_size" class="form-label">Paper Size <span class="text-danger">*</span></label>
                            <select class="form-select" id="paper_size" name="paper_size" required>
                                <option value="80mm">80mm (Thermal)</option>
                                <option value="58mm">58mm (Thermal)</option>
                                <option value="A4">A4 (Invoice)</option>
                                <option value="A5">A5 (Invoice)</option>
                            </select>
                        </div>
                    </div>
                    
                    <!-- Printer Settings -->
                    <div class="row mb-3">
                        <div class="col-md-4">
                            <label for="chars_per_line" class="form-label">Characters Per Line</label>
                            <input type="number" class="form-control" id="chars_per_line" name="chars_per_line" value="42" min="32" max="80">
                        </div>
                        <div class="col-md-4">
                            <label for="encoding" class="form-label">Encoding</label>
                            <select class="form-select" id="encoding" name="encoding">
                                <option value="CP850">CP850 (Default)</option>
                                <option value="CP437">CP437</option>
                                <option value="UTF-8">UTF-8</option>
                            </select>
                        </div>
                        <div class="col-md-4">
                            <label for="copies" class="form-label">Copies</label>
                            <input type="number" class="form-control" id="copies" name="copies" value="1" min="1" max="3">
                        </div>
                    </div>
                    
                    <!-- Printer Features -->
                    <div class="row mb-3">
                        <div class="col-md-4">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="auto_cut" name="auto_cut" checked>
                                <label class="form-check-label" for="auto_cut">Auto Cut Paper</label>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="auto_open_drawer" name="auto_open_drawer">
                                <label class="form-check-label" for="auto_open_drawer">Auto Open Cash Drawer</label>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="is_default" name="is_default">
                                <label class="form-check-label" for="is_default">Set as Default</label>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Header Settings -->
                    <div class="card mb-3">
                        <div class="card-header bg-light">
                            <h6 class="mb-0"><i class="bx bx-header mr-2"></i>Header Settings</h6>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="print_store_logo" name="print_store_logo">
                                        <label class="form-check-label" for="print_store_logo">Print Store Logo</label>
                                    </div>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="print_store_name" name="print_store_name" checked>
                                        <label class="form-check-label" for="print_store_name">Print Store Name</label>
                                    </div>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="print_store_address" name="print_store_address" checked>
                                        <label class="form-check-label" for="print_store_address">Print Store Address</label>
                                    </div>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="print_store_phone" name="print_store_phone" checked>
                                        <label class="form-check-label" for="print_store_phone">Print Store Phone</label>
                                    </div>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="print_vat_number" name="print_vat_number" checked>
                                        <label class="form-check-label" for="print_vat_number">Print VAT Number</label>
                                    </div>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="print_branch_code" name="print_branch_code" checked>
                                        <label class="form-check-label" for="print_branch_code">Print Branch Code</label>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Body Settings -->
                    <div class="card mb-3">
                        <div class="card-header bg-light">
                            <h6 class="mb-0"><i class="bx bx-body mr-2"></i>Body Settings</h6>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="print_cashier_name" name="print_cashier_name" checked>
                                        <label class="form-check-label" for="print_cashier_name">Print Cashier Name</label>
                                    </div>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="print_date_time" name="print_date_time" checked>
                                        <label class="form-check-label" for="print_date_time">Print Date & Time</label>
                                    </div>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="print_transaction_number" name="print_transaction_number" checked>
                                        <label class="form-check-label" for="print_transaction_number">Print Transaction Number</label>
                                    </div>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="print_customer_info" name="print_customer_info">
                                        <label class="form-check-label" for="print_customer_info">Print Customer Info</label>
                                    </div>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="print_item_barcode" name="print_item_barcode">
                                        <label class="form-check-label" for="print_item_barcode">Print Item Barcodes</label>
                                    </div>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="print_qr_code" name="print_qr_code" checked>
                                        <label class="form-check-label" for="print_qr_code">Print QR Code</label>
                                    </div>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="qr_code_type" class="form-label">QR Code Type</label>
                                    <select class="form-select" id="qr_code_type" name="qr_code_type">
                                        <option value="store_info">Store Information</option>
                                        <option value="transaction_id">Transaction ID</option>
                                        <option value="payment_link">Payment Link</option>
                                    </select>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="print_tax_breakdown" name="print_tax_breakdown" checked>
                                        <label class="form-check-label" for="print_tax_breakdown">Print Tax Breakdown</label>
                                    </div>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="print_discounts" name="print_discounts" checked>
                                        <label class="form-check-label" for="print_discounts">Print Discounts</label>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Footer Settings -->
                    <div class="card mb-3">
                        <div class="card-header bg-light">
                            <h6 class="mb-0"><i class="bx bx-footer mr-2"></i>Footer Settings</h6>
                        </div>
                        <div class="card-body">
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="print_payment_method" name="print_payment_method" checked>
                                        <label class="form-check-label" for="print_payment_method">Print Payment Method</label>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="print_change_amount" name="print_change_amount" checked>
                                        <label class="form-check-label" for="print_change_amount">Print Change Amount</label>
                                    </div>
                                </div>
                            </div>
                            <div class="row">
                                <div class="col-12">
                                    <div class="form-check mb-2">
                                        <input class="form-check-input" type="checkbox" id="print_footer" name="print_footer" checked>
                                        <label class="form-check-label" for="print_footer">Print Footer</label>
                                    </div>
                                    <label for="footer_text" class="form-label">Footer Text</label>
                                    <textarea class="form-control" id="footer_text" name="footer_text" rows="2" placeholder="Thank you for shopping with us!"></textarea>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Status -->
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="is_active" name="is_active" checked>
                                <label class="form-check-label" for="is_active">Active</label>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="saveReceiptSetting()">Save Setting</button>
            </div>
        </div>
    </div>
</div>

<!-- Test Receipt Modal -->
<div class="modal fade" id="testReceiptModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="bx bx-printer mr-2"></i>Test Receipt</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="alert alert-info border-0 mb-3">
                    <i class="bx bx-info-circle mr-2"></i>
                    This will generate a test receipt with sample data. Make sure your printer is connected.
                </div>
                <iframe id="testReceiptFrame" style="width: 100%; height: 500px; border: 1px solid #ddd;" title="Test Receipt"></iframe>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                <button type="button" class="btn btn-primary" onclick="printTestReceipt()">
                    <i class="bx bx-printer mr-1"></i>Print
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Config Modal -->
<div class="modal fade" id="configModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="bx bx-cog mr-2"></i>Configuration Settings</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <pre id="configContent" class="bg-light p-3 rounded" style="max-height: 400px; overflow-y: auto; font-size: 12px;"></pre>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                <button type="button" class="btn btn-primary" onclick="copyConfig()">
                    <i class="bx bx-copy mr-1"></i>Copy
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script>
$(document).ready(function() {
    // Initialize modal for create
    $('#createModal').on('click', function() {
        $('#modalTitle').html('<i class="bx bx-plus mr-2"></i>Create Receipt Setting');
        $('#receiptForm')[0].reset();
        $('#setting_id').val('');
        new bootstrap.Modal('#receiptModal').show();
    });
});

// Edit receipt setting
function editSetting(id) {
    showLoading('Loading setting...');
    
    fetch(`/settings/receipt/${id}/edit`)
        .then(response => response.json())
        .then(data => {
            hideLoading();
            
            if (data.success) {
                const setting = data.setting;
                
                // Fill form with setting data
                $('#modalTitle').html('<i class="bx bx-edit mr-2"></i>Edit Receipt Setting');
                $('#setting_id').val(id);
                $('#name').val(setting.name);
                $('#paper_size').val(setting.paper_size);
                $('#chars_per_line').val(setting.chars_per_line);
                $('#encoding').val(setting.encoding);
                $('#copies').val(setting.copies);
                $('#auto_cut').prop('checked', setting.auto_cut);
                $('#auto_open_drawer').prop('checked', setting.auto_open_drawer);
                $('#is_default').prop('checked', setting.is_default);
                $('#print_store_logo').prop('checked', setting.print_store_logo);
                $('#print_store_name').prop('checked', setting.print_store_name);
                $('#print_store_address').prop('checked', setting.print_store_address);
                $('#print_store_phone').prop('checked', setting.print_store_phone);
                $('#print_vat_number').prop('checked', setting.print_vat_number);
                $('#print_branch_code').prop('checked', setting.print_branch_code);
                $('#print_cashier_name').prop('checked', setting.print_cashier_name);
                $('#print_date_time').prop('checked', setting.print_date_time);
                $('#print_transaction_number').prop('checked', setting.print_transaction_number);
                $('#print_customer_info').prop('checked', setting.print_customer_info);
                $('#print_item_barcode').prop('checked', setting.print_item_barcode);
                $('#print_qr_code').prop('checked', setting.print_qr_code);
                $('#qr_code_type').val(setting.qr_code_type);
                $('#print_tax_breakdown').prop('checked', setting.print_tax_breakdown);
                $('#print_discounts').prop('checked', setting.print_discounts);
                $('#print_payment_method').prop('checked', setting.print_payment_method);
                $('#print_change_amount').prop('checked', setting.print_change_amount);
                $('#print_footer').prop('checked', setting.print_footer);
                $('#footer_text').val(setting.footer_text);
                $('#is_active').prop('checked', setting.is_active);
                
                new bootstrap.Modal('#receiptModal').show();
            } else {
                showNotification('Failed to load setting!', 'error');
            }
        })
        .catch(error => {
            hideLoading();
            showNotification('Error loading setting!', 'error');
        });
}

// Save receipt setting
function saveReceiptSetting() {
    const form = $('#receiptForm');
    const formData = new FormData(form[0]);
    
    const id = $('#setting_id').val();
    const url = id ? `/settings/receipt/${id}` : '/settings/receipt';
    const method = id ? 'PUT' : 'POST';
    
    showLoading('Saving setting...');
    
    fetch(url, {
        method: method,
        body: formData,
        headers: {
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        }
    })
    .then(response => response.json())
    .then(data => {
        hideLoading();
        
        if (data.success) {
            showNotification(data.message || 'Setting saved successfully!', 'success');
            bootstrap.Modal.getInstance('#receiptModal').hide();
            setTimeout(() => window.location.reload(), 1000);
        } else {
            showNotification(data.message || 'Failed to save setting!', 'error');
        }
    })
    .catch(error => {
        hideLoading();
        showNotification('Error saving setting!', 'error');
    });
}

// Delete receipt setting
function deleteSetting(id) {
    if (!confirm('Are you sure you want to delete this receipt setting?')) {
        return;
    }
    
    showLoading('Deleting...');
    
    fetch(`/settings/receipt/${id}`, {
        method: 'DELETE',
        headers: {
            'X-CSRF-TOKEN': '{{ csrf_token() }}',
            'Accept': 'application/json'
        }
    })
    .then(response => response.json())
    .then(data => {
        hideLoading();
        
        if (data.success) {
            showNotification('Setting deleted successfully!', 'success');
            setTimeout(() => window.location.reload(), 1000);
        } else {
            showNotification(data.message || 'Failed to delete setting!', 'error');
        }
    })
    .catch(error => {
        hideLoading();
        showNotification('Error deleting setting!', 'error');
    });
}

// Set as default
function setAsDefault(id) {
    showLoading('Setting as default...');
    
    fetch(`/settings/receipt/${id}/set-default`, {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': '{{ csrf_token() }}',
            'Accept': 'application/json'
        }
    })
    .then(response => response.json())
    .then(data => {
        hideLoading();
        
        if (data.success) {
            showNotification('Setting set as default!', 'success');
            setTimeout(() => window.location.reload(), 1000);
        } else {
            showNotification('Failed to set as default!', 'error');
        }
    })
    .catch(error => {
        hideLoading();
        showNotification('Error setting default!', 'error');
    });
}

// Test receipt
function testReceipt() {
    const receiptFrame = $('#testReceiptFrame');
    receiptFrame.attr('src', '{{ route("settings.receipt.test") }}');
    new bootstrap.Modal('#testReceiptModal').show();
}

// Test specific setting
function testSetting(id) {
    const receiptFrame = $('#testReceiptFrame');
    receiptFrame.attr('src', '{{ route("settings.receipt.test") }}?setting_id=' + id);
    new bootstrap.Modal('#testReceiptModal').show();
}

// Print test receipt
function printTestReceipt() {
    const receiptFrame = $('#testReceiptFrame')[0];
    receiptFrame.contentWindow.print();
}

// View default config
function viewDefaultConfig() {
    showLoading('Loading configuration...');
    
    fetch('{{ route("settings.receipt.api-config") }}')
        .then(response => response.json())
        .then(data => {
            hideLoading();
            
            if (data.success) {
                $('#configContent').text(JSON.stringify(data.settings, null, 2));
                new bootstrap.Modal('#configModal').show();
            } else {
                showNotification('Failed to load config!', 'error');
            }
        })
        .catch(error => {
            hideLoading();
            showNotification('Error loading config!', 'error');
        });
}

// Copy config
function copyConfig() {
    const configText = $('#configContent').text();
    navigator.clipboard.writeText(configText).then(() => {
        showNotification('Copied to clipboard!', 'success');
    }).catch(err => {
        showNotification('Failed to copy!', 'error');
    });
}

// Load receipt settings
function loadReceiptSettings() {
    showLoading('Refreshing settings...');
    
    setTimeout(() => {
        hideLoading();
        window.location.reload();
    }, 1000);
}

// Clear receipt cache
function clearReceiptCache() {
    showLoading('Clearing cache...');
    
    fetch('{{ route("settings.receipt.clear-cache") }}', {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': '{{ csrf_token() }}',
            'Accept': 'application/json'
        }
    })
    .then(response => response.json())
    .then(data => {
        hideLoading();
        
        if (data.success) {
            showNotification('Cache cleared!', 'success');
        } else {
            showNotification('Failed to clear cache!', 'error');
        }
    })
    .catch(error => {
        hideLoading();
        showNotification('Error clearing cache!', 'error');
    });
}

// Utility functions
function showNotification(message, type = 'info') {
    const Toast = Swal.mixin({
        toast: true,
        position: 'top-end',
        showConfirmButton: false,
        timer: 3000,
        timerProgressBar: true
    });
    
    Toast.fire({
        icon: type,
        title: message
    });
}

function showLoading(message) {
    Swal.fire({
        title: message,
        allowOutsideClick: false,
        showConfirmButton: false,
        willOpen: () => {
            Swal.showLoading();
        }
    });
}

function hideLoading() {
    Swal.close();
}
</script>

<style>
.receipt-preview {
    border: 1px solid #dee2e6;
    border-radius: 8px;
    padding: 20px;
    background: white;
    font-family: 'Courier New', monospace;
    font-size: 14px;
    max-width: 400px;
    margin: 0 auto;
}

.receipt-header {
    border-bottom: 2px dashed #dee2e6;
    padding-bottom: 15px;
}

.receipt-footer {
    border-top: 2px dashed #dee2e6;
    padding-top: 15px;
}

.badge {
    font-size: 0.75em;
    padding: 0.35em 0.65em;
}

.card-header.bg-light {
    background-color: #f8f9fa !important;
    border-bottom: 1px solid #dee2e6;
}

@media (max-width: 768px) {
    .receipt-preview {
        font-size: 12px;
        padding: 10px;
    }
}
</style>
@endsection