@extends('layouts.app')

@section('title', 'System Logs')


@section('content')
<!--start page wrapper -->
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">Settings</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('settings.index') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item active" aria-current="page">System Logs</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <button type="button" class="btn btn-outline-danger me-2" onclick="clearLogs()">
                    <i class="bx bx-trash mr-1"></i>Clear All
                </button>
                <button type="button" class="btn btn-primary" onclick="refreshLogs()">
                    <i class="bx bx-refresh mr-1"></i>Refresh
                </button>
            </div>
        </div>
        <!--end breadcrumb-->
        
        <!-- Log Statistics -->
        <div class="row row-cols-1 row-cols-md-2 row-cols-lg-4 g-3 mb-4">
            <div class="col">
                <div class="card radius-10 border-start border-0 border-4 border-info">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-0 text-secondary">Log Files</p>
                                <h4 class="my-1" id="logFilesCount">0</h4>
                            </div>
                            <div class="widgets-icons-2 rounded-circle bg-gradient-scooter text-white ms-auto">
                                <i class='bx bx-file'></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="card radius-10 border-start border-0 border-4 border-warning">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-0 text-secondary">Errors Today</p>
                                <h4 class="my-1" id="errorsToday">0</h4>
                            </div>
                            <div class="widgets-icons-2 rounded-circle bg-gradient-burning text-white ms-auto">
                                <i class='bx bx-error-circle'></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="card radius-10 border-start border-0 border-4 border-success">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-0 text-secondary">Activities Today</p>
                                <h4 class="my-1" id="activitiesToday">0</h4>
                            </div>
                            <div class="widgets-icons-2 rounded-circle bg-gradient-ohhappiness text-white ms-auto">
                                <i class='bx bx-user-check'></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="card radius-10 border-start border-0 border-4 border-danger">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-0 text-secondary">Log Size</p>
                                <h4 class="my-1" id="logSize">0 MB</h4>
                            </div>
                            <div class="widgets-icons-2 rounded-circle bg-gradient-bloody text-white ms-auto">
                                <i class='bx bx-hdd'></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-body">
                        <!-- Logs Tabs -->
                        <ul class="nav nav-tabs nav-primary mb-3" role="tablist">
                            <li class="nav-item" role="presentation">
                                <a class="nav-link active" data-bs-toggle="tab" href="#laravel" role="tab" aria-selected="true">
                                    <i class="bx bx-bug mr-1"></i>Laravel Logs
                                </a>
                            </li>
                            <li class="nav-item" role="presentation">
                                <a class="nav-link" data-bs-toggle="tab" href="#activity" role="tab" aria-selected="false">
                                    <i class="bx bx-user-pin mr-1"></i>Activity Logs
                                </a>
                            </li>
                            <li class="nav-item" role="presentation">
                                <a class="nav-link" data-bs-toggle="tab" href="#error" role="tab" aria-selected="false">
                                    <i class="bx bx-error-alt mr-1"></i>Error Logs
                                </a>
                            </li>
                            <li class="nav-item" role="presentation">
                                <a class="nav-link" data-bs-toggle="tab" href="#access" role="tab" aria-selected="false">
                                    <i class="bx bx-log-in mr-1"></i>Access Logs
                                </a>
                            </li>
                        </ul>
                        
                        <div class="tab-content">
                            <!-- Laravel Logs -->
                            <div class="tab-pane fade show active" id="laravel" role="tabpanel">
                                <div class="table-responsive">
                                    <table id="laravelTable" class="table table-striped table-bordered" style="width:100%">
                                        <thead>
                                            <tr>
                                                <th width="120">Time</th>
                                                <th width="100">Level</th>
                                                <th>Message</th>
                                                <th width="100">Context</th>
                                                <th width="80">Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody id="laravelLogs">
                                            <!-- Logs will be loaded here -->
                                        </tbody>
                                    </table>
                                </div>
                            </div>

                            <!-- Activity Logs -->
                            <div class="tab-pane fade" id="activity" role="tabpanel">
                                <div class="table-responsive">
                                    <table id="activityTable" class="table table-striped table-bordered" style="width:100%">
                                        <thead>
                                            <tr>
                                                <th width="120">Time</th>
                                                <th>User</th>
                                                <th>Activity</th>
                                                <th width="120">IP Address</th>
                                                <th width="100">Browser</th>
                                            </tr>
                                        </thead>
                                        <tbody id="activityLogs">
                                            <!-- Activity logs will be loaded here -->
                                        </tbody>
                                    </table>
                                </div>
                            </div>

                            <!-- Error Logs -->
                            <div class="tab-pane fade" id="error" role="tabpanel">
                                <div class="table-responsive">
                                    <table id="errorTable" class="table table-striped table-bordered" style="width:100%">
                                        <thead>
                                            <tr>
                                                <th width="120">Time</th>
                                                <th width="100">Type</th>
                                                <th>Error</th>
                                                <th>File</th>
                                                <th width="80">Line</th>
                                            </tr>
                                        </thead>
                                        <tbody id="errorLogs">
                                            <!-- Error logs will be loaded here -->
                                        </tbody>
                                    </table>
                                </div>
                            </div>

                            <!-- Access Logs -->
                            <div class="tab-pane fade" id="access" role="tabpanel">
                                <div class="table-responsive">
                                    <table id="accessTable" class="table table-striped table-bordered" style="width:100%">
                                        <thead>
                                            <tr>
                                                <th width="120">Time</th>
                                                <th>IP Address</th>
                                                <th width="80">Method</th>
                                                <th>URL</th>
                                                <th width="80">Status</th>
                                                <th width="100">Duration</th>
                                            </tr>
                                        </thead>
                                        <tbody id="accessLogs">
                                            <!-- Access logs will be loaded here -->
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Log Settings -->
        <div class="row mt-4">
            <div class="col-lg-8">
                <div class="card">
                    <div class="card-header bg-transparent">
                        <h5 class="mb-0"><i class="bx bx-cog mr-2"></i>Log Settings</h5>
                    </div>
                    <div class="card-body">
                        <div class="row g-3">
                            <div class="col-md-4">
                                <label class="form-label">Log Retention</label>
                                <select class="form-select" id="retentionPeriod">
                                    <option value="7">7 days</option>
                                    <option value="30" selected>30 days</option>
                                    <option value="90">90 days</option>
                                    <option value="365">1 year</option>
                                    <option value="0">Forever</option>
                                </select>
                            </div>
                            <div class="col-md-4">
                                <label class="form-label">Log Level</label>
                                <select class="form-select" id="logLevel">
                                    <option value="debug">Debug</option>
                                    <option value="info">Info</option>
                                    <option value="warning" selected>Warning</option>
                                    <option value="error">Error</option>
                                </select>
                            </div>
                            <div class="col-md-4">
                                <label class="form-label">Log Channel</label>
                                <select class="form-select" id="logChannel">
                                    <option value="single">Single File</option>
                                    <option value="daily" selected>Daily Files</option>
                                    <option value="slack">Slack</option>
                                </select>
                            </div>
                        </div>
                        <div class="d-flex gap-2 mt-4">
                            <button type="button" class="btn btn-primary" onclick="saveLogSettings()">
                                <i class="bx bx-save mr-1"></i>Save Settings
                            </button>
                            <button type="button" class="btn btn-outline-secondary" onclick="downloadLogs()">
                                <i class="bx bx-download mr-1"></i>Download Logs
                            </button>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="col-lg-4">
                <div class="card">
                    <div class="card-header bg-transparent">
                        <h5 class="mb-0"><i class="bx bx-info-circle mr-2"></i>Log Information</h5>
                    </div>
                    <div class="card-body">
                        <div class="mb-3">
                            <strong class="d-block mb-1">Log Directory:</strong>
                            <code class="small">storage/logs/</code>
                        </div>
                        <div class="mb-3">
                            <strong class="d-block mb-1">Current Log File:</strong>
                            <code class="small">laravel-{{ date('Y-m-d') }}.log</code>
                        </div>
                        <div class="mb-3">
                            <strong class="d-block mb-1">Last Modified:</strong>
                            <span class="text-muted" id="lastModified">Just now</span>
                        </div>
                        <div class="alert alert-light border">
                            <i class="bx bx-info-circle mr-2"></i>
                            <small>Logs are automatically rotated daily. Older logs are archived.</small>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
<!--end page wrapper -->

<!-- Log Detail Modal -->
<div class="modal fade" id="logDetailModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="bx bx-file mr-2"></i>Log Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <pre id="logDetailContent" class="bg-light p-3 rounded" style="max-height: 400px; overflow-y: auto; font-size: 12px;"></pre>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                <button type="button" class="btn btn-primary" onclick="copyLogDetail()">
                    <i class="bx bx-copy mr-1"></i>Copy
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script src="{{ asset('assets/plugins/datatable/js/jquery.dataTables.min.js') }}"></script>
<script src="{{ asset('assets/plugins/datatable/js/dataTables.bootstrap5.min.js') }}"></script>

<script>
// Initialize DataTables
let laravelTable, activityTable, errorTable, accessTable;

$(document).ready(function() {
    loadLogs();
    initializeTables();
});

// Initialize DataTables
function initializeTables() {
    const tableConfig = {
        lengthMenu: [[10, 25, 50, -1], [10, 25, 50, "All"]],
        pageLength: 10,
        ordering: true,
        order: [[0, 'desc']],
        language: {
            search: "Search:",
            lengthMenu: "Show _MENU_ entries",
            info: "Showing _START_ to _END_ of _TOTAL_ entries",
            paginate: {
                previous: "<i class='bx bx-chevron-left'>",
                next: "<i class='bx bx-chevron-right'>"
            }
        }
    };
    
    laravelTable = $('#laravelTable').DataTable(tableConfig);
    activityTable = $('#activityTable').DataTable(tableConfig);
    errorTable = $('#errorTable').DataTable(tableConfig);
    accessTable = $('#accessTable').DataTable(tableConfig);
}

// Load logs
function loadLogs() {
    showLoading('Loading logs...');
    
    // Simulate API call
    setTimeout(() => {
        // Laravel logs
        const laravelLogs = [
            { time: new Date().toLocaleTimeString(), level: 'ERROR', message: 'Failed to connect to database', context: 'DB', stack: 'Connection refused' },
            { time: new Date(Date.now() - 300000).toLocaleTimeString(), level: 'WARNING', message: 'Cache cleared', context: 'Cache', stack: 'Manual clear' },
            { time: new Date(Date.now() - 600000).toLocaleTimeString(), level: 'INFO', message: 'User logged in', context: 'Auth', stack: 'User ID: 1' },
            { time: new Date(Date.now() - 900000).toLocaleTimeString(), level: 'DEBUG', message: 'Route accessed', context: 'Routing', stack: '/api/store/info' }
        ];
        
        populateLaravelLogs(laravelLogs);
        
        // Update stats
        updateLogStats();
        hideLoading();
    }, 1000);
}

// Populate laravel logs
function populateLaravelLogs(logs) {
    laravelTable.clear();
    
    logs.forEach(log => {
        const row = laravelTable.row.add([
            log.time,
            getLevelBadge(log.level),
            `<span class="text-truncate" title="${log.message}">${log.message}</span>`,
            `<span class="badge bg-secondary">${log.context}</span>`,
            `<button type="button" class="btn btn-sm btn-outline-info" onclick="viewLogDetail('${log.message}', '${log.stack}')">
                <i class="bx bx-show"></i>
            </button>`
        ]).draw(false);
    });
}

// Get level badge
function getLevelBadge(level) {
    const colors = {
        'ERROR': 'danger',
        'WARNING': 'warning',
        'INFO': 'info',
        'DEBUG': 'secondary'
    };
    
    return `<span class="badge bg-${colors[level] || 'secondary'}">${level}</span>`;
}

// Update log statistics
function updateLogStats() {
    $('#logFilesCount').text('4');
    $('#errorsToday').text('1');
    $('#activitiesToday').text('2');
    $('#logSize').text('2.4 MB');
    $('#lastModified').text('Just now');
}

// View log detail
function viewLogDetail(message, stack) {
    const detail = `Message: ${message}\n\nStack Trace:\n${stack}`;
    $('#logDetailContent').text(detail);
    new bootstrap.Modal('#logDetailModal').show();
}

// Copy log detail
function copyLogDetail() {
    const logDetail = $('#logDetailContent').text();
    navigator.clipboard.writeText(logDetail).then(() => {
        showNotification('Copied to clipboard!', 'success');
    }).catch(err => {
        showNotification('Failed to copy!', 'error');
    });
}

// Clear logs
function clearLogs() {
    Swal.fire({
        title: 'Clear All Logs?',
        text: 'This action cannot be undone!',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#d33',
        cancelButtonColor: '#3085d6',
        confirmButtonText: 'Yes, clear all!'
    }).then((result) => {
        if (result.isConfirmed) {
            showLoading('Clearing logs...');
            
            setTimeout(() => {
                // Clear all tables
                laravelTable.clear().draw();
                activityTable.clear().draw();
                errorTable.clear().draw();
                accessTable.clear().draw();
                
                // Reset stats
                $('#logFilesCount').text('0');
                $('#errorsToday').text('0');
                $('#activitiesToday').text('0');
                $('#logSize').text('0 MB');
                
                hideLoading();
                showNotification('All logs cleared!', 'success');
            }, 1500);
        }
    });
}

// Refresh logs
function refreshLogs() {
    showLoading('Refreshing...');
    
    setTimeout(() => {
        loadLogs();
        showNotification('Logs refreshed!', 'info');
    }, 1000);
}

// Save log settings
function saveLogSettings() {
    const retention = $('#retentionPeriod').val();
    const level = $('#logLevel').val();
    const channel = $('#logChannel').val();
    
    showLoading('Saving settings...');
    
    setTimeout(() => {
        hideLoading();
        showNotification('Settings saved!', 'success');
        console.log('Log settings:', { retention, level, channel });
    }, 1000);
}

// Download logs
function downloadLogs() {
    showLoading('Preparing download...');
    
    setTimeout(() => {
        hideLoading();
        
        // Create download
        const data = "Log data would be here in real implementation";
        const blob = new Blob([data], { type: 'text/plain' });
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `system-logs-${new Date().toISOString().split('T')[0]}.txt`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        window.URL.revokeObjectURL(url);
        
        showNotification('Download started!', 'success');
    }, 1500);
}

// Utility functions
function showLoading(message) {
    Swal.fire({
        title: message,
        allowOutsideClick: false,
        showConfirmButton: false,
        willOpen: () => {
            Swal.showLoading();
        }
    });
}

function hideLoading() {
    Swal.close();
}

function showNotification(message, type = 'info') {
    const Toast = Swal.mixin({
        toast: true,
        position: 'top-end',
        showConfirmButton: false,
        timer: 3000,
        timerProgressBar: true
    });
    
    Toast.fire({
        icon: type,
        title: message
    });
}
</script>

<style>
.widgets-icons-2 {
    width: 56px;
    height: 56px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.bg-gradient-scooter {
    background: linear-gradient(45deg, #17ead9, #6078ea) !important;
}

.bg-gradient-burning {
    background: linear-gradient(45deg, #ff416c, #ff4b2b) !important;
}

.bg-gradient-ohhappiness {
    background: linear-gradient(45deg, #00b09b, #96c93d) !important;
}

.bg-gradient-bloody {
    background: linear-gradient(45deg, #f54ea2, #ff7676) !important;
}

.table td {
    vertical-align: middle;
}

.text-truncate {
    max-width: 300px;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}
</style>
@endsection