<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Receipt - {{ $data['invoice_no'] ?? 'POS Receipt' }}</title>
    <style>
        /* Thermal printer friendly styling */
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Courier New', monospace;
            font-size: 12px;
            line-height: 1.2;
            max-width: 320px;
            margin: 0 auto;
            padding: 5px;
            color: #000;
        }
        
        /* Thermal printer specific */
        @media print {
            @page {
                margin: 0;
                size: 80mm auto;
            }
            
            body {
                width: 80mm;
                padding: 0;
            }
            
            .no-print {
                display: none !important;
            }
            
            .page-break {
                page-break-before: always;
            }
        }
        
        /* Common styles */
        .text-center { text-align: center; }
        .text-right { text-align: right; }
        .text-left { text-align: left; }
        .bold { font-weight: bold; }
        .underline { text-decoration: underline; }
        
        /* Layout */
        .receipt {
            width: 100%;
        }
        
        .header {
            text-align: center;
            padding-bottom: 5px;
            border-bottom: 1px dashed #000;
            margin-bottom: 5px;
        }
        
        .store-name {
            font-size: 14px;
            font-weight: bold;
            text-transform: uppercase;
            margin: 3px 0;
        }
        
        .store-details {
            font-size: 10px;
            margin: 2px 0;
        }
        
        .section {
            margin: 5px 0;
            padding: 3px 0;
        }
        
        .section-title {
            font-weight: bold;
            border-bottom: 1px solid #000;
            margin-bottom: 3px;
            padding-bottom: 2px;
        }
        
        .item-row {
            display: flex;
            justify-content: space-between;
            margin: 1px 0;
            padding: 1px 0;
        }
        
        .item-name {
            flex: 3;
            text-align: left;
        }
        
        .item-qty {
            flex: 1;
            text-align: center;
        }
        
        .item-price {
            flex: 2;
            text-align: right;
        }
        
        .item-total {
            flex: 2;
            text-align: right;
            font-weight: bold;
        }
        
        .total-row {
            display: flex;
            justify-content: space-between;
            margin: 2px 0;
            padding: 2px 0;
        }
        
        .total-label {
            flex: 2;
        }
        
        .total-value {
            flex: 1;
            text-align: right;
            font-weight: bold;
        }
        
        .grand-total {
            border-top: 2px solid #000;
            border-bottom: 2px solid #000;
            margin: 5px 0;
            padding: 5px 0;
            font-size: 14px;
        }
        
        .payment-method {
            text-align: center;
            margin: 5px 0;
            padding: 3px;
            border: 1px dashed #000;
        }
        
        .footer {
            text-align: center;
            font-size: 10px;
            margin-top: 10px;
            padding-top: 5px;
            border-top: 1px dashed #000;
        }
        
        .vat-breakdown {
            font-size: 10px;
            margin: 3px 0;
        }
        
        .vat-line {
            display: flex;
            justify-content: space-between;
            margin: 1px 0;
        }
        
        /* Responsive for screen */
        @media screen {
            body {
                background: #f5f5f5;
                padding: 20px;
            }
            
            .receipt {
                background: white;
                padding: 20px;
                box-shadow: 0 0 10px rgba(0,0,0,0.1);
                border-radius: 5px;
            }
        }
        
        /* Barcode style */
        .barcode {
            text-align: center;
            margin: 10px 0;
            font-family: 'Libre Barcode 128', cursive;
            font-size: 36px;
        }
        
        /* QR Code container */
        .qrcode {
            text-align: center;
            margin: 10px 0;
        }
        
        .qrcode img {
            width: 100px;
            height: 100px;
        }
        
        /* Customer info warning */
        .customer-warning {
            font-style: italic;
            color: #666;
        }
    </style>
    
    <!-- Barcode font -->
    <link href="https://fonts.googleapis.com/css2?family=Libre+Barcode+128&display=swap" rel="stylesheet">
    
    <!-- QR Code library -->
    <script src="https://cdn.rawgit.com/davidshimjs/qrcodejs/gh-pages/qrcode.min.js"></script>
    
    <script>
        @if($data['autoprint'] ?? false)
        // Auto-print after short delay
        window.onload = function() {
            setTimeout(function() {
                window.print();
                
                // Close window after printing if not in iframe
                setTimeout(function() {
                    if (window.self !== window.top) {
                        window.top.postMessage('receipt-printed', '*');
                    } else {
                        // Optionally close the window
                        // window.close();
                    }
                }, 1000);
            }, 500);
        };
        @endif
        
        // Generate QR code if needed
        document.addEventListener('DOMContentLoaded', function() {
            // Generate invoice QR code
            const invoiceNo = '{{ $data["invoice_no"] ?? "" }}';
            if (invoiceNo && typeof QRCode !== 'undefined') {
                const qrContainer = document.getElementById('qrcode');
                if (qrContainer) {
                    new QRCode(qrContainer, {
                        text: invoiceNo,
                        width: 100,
                        height: 100,
                        colorDark : "#000000",
                        colorLight : "#ffffff",
                        correctLevel : QRCode.CorrectLevel.H
                    });
                }
            }
            
            // Add barcode
            const barcodeContainer = document.getElementById('barcode');
            if (barcodeContainer && invoiceNo) {
                barcodeContainer.textContent = `*${invoiceNo}*`;
            }
        });
    </script>
</head>
<body>
    @php
        // Safe data extraction with defaults
        $store = $data['store'] ?? [
            'name' => config('app.name', 'Kenyan Supermarket'),
            'address' => config('app.address', 'Nairobi CBD'),
            'phone' => config('app.phone', '0700 000 000'),
            'pin' => config('app.pin', 'P051234567N'),
            'vat_number' => config('app.vat_number', 'VAT001234567'),
        ];
        
        $customer = $data['customer'] ?? null;
        $items = $data['items'] ?? [];
        $totals = $data['totals'] ?? [];
        $payment = $data['payment'] ?? [];
        $vatInfo = $data['vat_info'] ?? [];
        $notes = $data['notes'] ?? [];
        
        // Safe customer data
        $customerName = $customer['name'] ?? null;
        $customerPhone = $customer['phone'] ?? null;
        $customerEmail = $customer['email'] ?? null;
        $customerPin = $customer['pin'] ?? null;
        $customerVatNumber = $customer['vat_number'] ?? null;
        $customerVatStatus = $customer['vat_status'] ?? 'vatable';
        
        // Safe totals
        $subtotal = $totals['subtotal'] ?? 0;
        $discount = $totals['discount'] ?? 0;
        $vatAmount = $totals['vat_amount'] ?? 0;
        $grandTotal = $totals['grand_total'] ?? 0;
        $taxableAmount = $totals['taxable_amount'] ?? 0;
        $nonTaxableAmount = $totals['non_taxable_amount'] ?? 0;
        $exemptedAmount = $totals['exempted_amount'] ?? 0;
        $zeroRatedAmount = $totals['zero_rated_amount'] ?? 0;
        
        // Safe payment data
        $paymentMethod = $payment['method'] ?? 'cash';
        $amountPaid = $payment['amount_paid'] ?? $grandTotal;
        $cashReceived = $payment['cash_received'] ?? null;
        $change = $payment['change'] ?? null;
        $transactionId = $payment['transaction_id'] ?? null;
        $mpesaDetails = $payment['mpesa_details'] ?? null;
        
        // Safe vat info
        $isVatApplicable = $vatInfo['is_vat_applicable'] ?? true;
        $isZeroRated = $vatInfo['is_zero_rated'] ?? false;
        $isExempted = $vatInfo['is_exempted'] ?? false;
        $vatBreakdown = $vatInfo['breakdown'] ?? [];
        
        // Default notes if none provided
        if (empty($notes)) {
            $notes = [
                'Thank you for your business!',
                'Keep this receipt for warranty claims',
                'VAT Invoice - Valid for Tax Purposes',
                'Items sold are not returnable unless defective',
                'Please check goods before leaving the counter',
                'Valid for 7 days from date of purchase'
            ];
        }
    @endphp
    
    <div class="receipt">
        <!-- Store Header -->
        <div class="header">
            <div class="store-name">{{ strtoupper($store['name']) }}</div>
            <div class="store-details">{{ $store['address'] ?? '' }}</div>
            <div class="store-details">Tel: {{ $store['phone'] ?? '' }}</div>
            @if(!empty($store['pin']))
            <div class="store-details">PIN: {{ $store['pin'] }}</div>
            @endif
            @if(!empty($store['vat_number']))
            <div class="store-details">VAT: {{ $store['vat_number'] }}</div>
            @endif
        </div>
        
        <!-- Transaction Info -->
        <div class="section">
            <div class="item-row">
                <span>Invoice No:</span>
                <span class="bold">{{ $data['invoice_no'] ?? '' }}</span>
            </div>
            <div class="item-row">
                <span>Receipt No:</span>
                <span class="bold">{{ $data['receipt_no'] ?? '' }}</span>
            </div>
            <div class="item-row">
                <span>Date:</span>
                <span>{{ $data['date'] ?? '' }} {{ $data['time'] ?? '' }}</span>
            </div>
            <div class="item-row">
                <span>Cashier:</span>
                <span>{{ $data['cashier'] ?? 'Cashier' }}</span>
            </div>
            
            {{-- Customer Information --}}
            @if($customerName)
            <div class="item-row">
                <span>Customer:</span>
                <span>{{ $customerName }}</span>
            </div>
            @endif
            
            @if($customerPhone)
            <div class="item-row">
                <span>Phone:</span>
                <span>{{ $customerPhone }}</span>
            </div>
            @endif
            
            @if($customerVatNumber)
            <div class="item-row">
                <span>VAT No:</span>
                <span>{{ $customerVatNumber }}</span>
            </div>
            @endif
            
            @if($customerVatStatus && in_array($customerVatStatus, ['exempted', 'zero_rated']))
            <div class="item-row">
                <span>VAT Status:</span>
                <span class="bold">{{ 
                    $customerVatStatus === 'exempted' ? 'VAT Exempted' : 
                    ($customerVatStatus === 'zero_rated' ? 'Zero-Rated' : 'VATable')
                }}</span>
            </div>
            @endif
            
            @if(!$customerName)
            <div class="item-row customer-warning">
                <span>Customer:</span>
                <span>Walk-in Customer</span>
            </div>
            @endif
        </div>
        
        <!-- Barcode -->
        <div class="barcode" id="barcode"></div>
        
        <!-- Items Section -->
        <div class="section">
            <div class="section-title">ITEMS</div>
            
            <!-- Items Header -->
            <div class="item-row bold">
                <div class="item-name">ITEM</div>
                <div class="item-qty">QTY</div>
                <div class="item-price">PRICE</div>
                <div class="item-total">TOTAL</div>
            </div>
            
            <!-- Items List -->
            @if(count($items) > 0)
                @foreach($items as $item)
                @php
                    $itemName = $item['name'] ?? 'Item';
                    $itemQty = $item['quantity'] ?? $item['qty'] ?? 1;
                    $itemPrice = $item['unit_price'] ?? $item['price'] ?? 0;
                    $itemTotal = $item['total'] ?? ($itemPrice * $itemQty);
                    $itemTaxRate = $item['tax_rate'] ?? 0;
                    $itemTaxAmount = $item['tax_amount'] ?? 0;
                    $isVatable = $item['is_vatable'] ?? ($itemTaxRate > 0);
                @endphp
                <div class="item-row">
                    <div class="item-name">{{ $itemName }}</div>
                    <div class="item-qty">{{ $itemQty }}</div>
                    <div class="item-price">KES {{ number_format($itemPrice, 2) }}</div>
                    <div class="item-total">KES {{ number_format($itemTotal, 2) }}</div>
                </div>
                
                @if($isVatable && $itemTaxRate > 0 && $itemTaxAmount > 0)
                <div class="vat-line">
                    <span>VAT {{ $itemTaxRate }}%</span>
                    <span>KES {{ number_format($itemTaxAmount, 2) }}</span>
                </div>
                @endif
                @endforeach
            @else
                <div class="item-row">
                    <div class="item-name">No items</div>
                    <div class="item-total">0.00</div>
                </div>
            @endif
        </div>
        
        <!-- Totals Section -->
        <div class="section">
            <div class="section-title">TOTALS</div>
            
            <div class="total-row">
                <span class="total-label">Subtotal:</span>
                <span class="total-value">KES {{ number_format($subtotal, 2) }}</span>
            </div>
            
            @if($discount > 0)
            <div class="total-row">
                <span class="total-label">Discount:</span>
                <span class="total-value">-KES {{ number_format($discount, 2) }}</span>
            </div>
            @endif
            
            <!-- VAT Breakdown -->
            @if(count($vatBreakdown) > 0)
                @foreach($vatBreakdown as $vat)
                <div class="vat-line">
                    <span>VAT {{ $vat['rate'] ?? 16 }}%:</span>
                    <span>KES {{ number_format($vat['amount'] ?? 0, 2) }}</span>
                </div>
                @endforeach
            @elseif($vatAmount > 0)
            <div class="total-row">
                <span class="total-label">VAT:</span>
                <span class="total-value">KES {{ number_format($vatAmount, 2) }}</span>
            </div>
            @endif
            
            <!-- Taxable/Non-taxable Summary -->
            @if($taxableAmount > 0)
            <div class="vat-line">
                <span>Taxable Amount:</span>
                <span>KES {{ number_format($taxableAmount, 2) }}</span>
            </div>
            @endif
            
            @if($nonTaxableAmount > 0)
            <div class="vat-line">
                <span>Non-taxable:</span>
                <span>KES {{ number_format($nonTaxableAmount, 2) }}</span>
            </div>
            @endif
            
            @if($exemptedAmount > 0)
            <div class="vat-line">
                <span>Exempted:</span>
                <span>KES {{ number_format($exemptedAmount, 2) }}</span>
            </div>
            @endif
            
            @if($zeroRatedAmount > 0)
            <div class="vat-line">
                <span>Zero-rated:</span>
                <span>KES {{ number_format($zeroRatedAmount, 2) }}</span>
            </div>
            @endif
            
            <!-- Grand Total -->
            <div class="total-row grand-total">
                <span class="total-label">TOTAL:</span>
                <span class="total-value">KES {{ number_format($grandTotal, 2) }}</span>
            </div>
        </div>
        
        <!-- Payment Details -->
        <div class="section">
            <div class="section-title">PAYMENT</div>
            
            <div class="payment-method">
                <div class="bold">{{ strtoupper($paymentMethod) }}</div>
                
                <div>Amount Paid: KES {{ number_format($amountPaid, 2) }}</div>
                
                @if($paymentMethod === 'cash' && $cashReceived)
                <div>Cash Received: KES {{ number_format($cashReceived, 2) }}</div>
                @endif
                
                @if($paymentMethod === 'cash' && $change > 0)
                <div class="bold">Change: KES {{ number_format($change, 2) }}</div>
                @endif
                
                @if($paymentMethod === 'mpesa' && $mpesaDetails)
                <div>Phone: {{ $mpesaDetails['phone'] ?? '' }}</div>
                <div>Transaction ID: {{ $mpesaDetails['transaction_id'] ?? '' }}</div>
                @endif
                
                @if($transactionId)
                <div>Transaction ID: {{ $transactionId }}</div>
                @endif
            </div>
        </div>
        
        <!-- VAT Status Information -->
        @if($isExempted || $isZeroRated || !$isVatApplicable)
        <div class="section">
            <div class="section-title">VAT INFORMATION</div>
            <div class="text-center">
                @if($isExempted)
                <div class="bold">VAT EXEMPTED CUSTOMER</div>
                <div>No VAT Applied</div>
                @elseif($isZeroRated)
                <div class="bold">ZERO-RATED CUSTOMER</div>
                <div>0% VAT Rate Applied</div>
                @elseif(!$isVatApplicable)
                <div class="bold">NO VAT APPLICABLE</div>
                <div>VAT Not Applied</div>
                @endif
            </div>
        </div>
        @endif
        
        <!-- QR Code for verification -->
        <div class="qrcode" id="qrcode"></div>
        
        <!-- Footer -->
        <div class="footer">
            @foreach($notes as $note)
            <div>{{ $note }}</div>
            @endforeach
            
            <div style="margin-top: 10px;">
                <div>Generated: {{ date('Y-m-d H:i:s') }}</div>
                <div>System: {{ config('app.name', 'Laravel') }}</div>
            </div>
        </div>
    </div>
    
    <!-- Print Controls (only visible on screen) -->
    <div class="no-print" style="text-align: center; margin-top: 20px;">
        <button onclick="window.print()" style="padding: 10px 20px; margin: 5px; background: #007bff; color: white; border: none; border-radius: 5px; cursor: pointer;">
            🖨️ Print Receipt
        </button>
        <button onclick="window.close()" style="padding: 10px 20px; margin: 5px; background: #6c757d; color: white; border: none; border-radius: 5px; cursor: pointer;">
            ❌ Close
        </button>
        <button onclick="location.reload()" style="padding: 10px 20px; margin: 5px; background: #28a745; color: white; border: none; border-radius: 5px; cursor: pointer;">
            🔄 Reload
        </button>
        
        @if($data['invoice_no'] ?? false)
        <div style="margin-top: 10px;">
            <small>Receipt No: {{ $data['invoice_no'] }}</small>
        </div>
        @endif
    </div>
    
    <script>
        // Handle print events
        window.addEventListener('afterprint', function() {
            console.log('Receipt printed');
            // Send analytics
            fetch('/api/receipt/printed', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    invoice_no: '{{ $data["invoice_no"] ?? "" }}',
                    printed_at: new Date().toISOString()
                })
            }).catch(() => {});
        });
    </script>
</body>
</html>