@extends('layouts.app')

@section('title', 'Edit Profile')

@push('style')
<style>
    .profile-image-container {
        position: relative;
        display: inline-block;
    }
    .profile-image-overlay {
        position: absolute;
        bottom: 0;
        left: 0;
        right: 0;
        background: rgba(0,0,0,0.5);
        color: white;
        text-align: center;
        padding: 5px;
        border-radius: 0 0 50% 50%;
        cursor: pointer;
    }
</style>
@endpush

@section('content')
@php
    // Use Auth::user() as fallback if $user is not passed
    $user = $user ?? Auth::user();
@endphp

<div class="page-wrapper">
    <div class="page-content">
        <!-- Start breadcrumb -->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">Profile</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item active" aria-current="page">Edit Profile</li>
                    </ol>
                </nav>
            </div>
        </div>
        <!-- End breadcrumb -->

        <div class="row">
            <div class="col-md-4">
                <!-- Profile Image Card -->
                <div class="card radius-10">
                    <div class="card-body text-center">
                        <div class="profile-image-container mb-3">
                            <img src="{{ $user->profile_image_url }}" 
                                 class="rounded-circle" width="150" height="150"
                                 onerror="this.src='{{ asset('assets/images/default-avatar.png') }}'"
                                 id="profileImagePreview">
                            <div class="profile-image-overlay" onclick="document.getElementById('profileImageInput').click()">
                                <i class="bx bx-camera"></i> Change
                            </div>
                        </div>
                        <h4 class="mb-1">{{ $user->name }}</h4>
                        <p class="text-muted mb-2">{{ $user->position ?? 'No position specified' }}</p>
                        
                        @if($user->employee_id)
                            <p class="text-muted mb-3"><i class="bx bx-id-card"></i> {{ $user->employee_id }}</p>
                        @endif
                        
                        <div class="d-flex justify-content-center gap-2 mb-3">
                            @foreach($user->roles as $role)
                            <span class="badge bg-{{ \App\Helpers\RoleHelper::getColor($role->name) }}">
                                <i class="bx bx-{{ \App\Helpers\RoleHelper::getIcon($role->name) }}"></i>
                                {{ ucfirst($role->name) }}
                            </span>
                            @endforeach
                        </div>

                        <!-- Change Profile Image Form (Hidden by default) -->
                        <form action="{{ route('profile.image.update') }}" method="POST" enctype="multipart/form-data" id="profileImageForm">
                            @csrf
                            <input type="file" name="profile_image" id="profileImageInput" 
                                   accept="image/*" style="display: none;" 
                                   onchange="handleProfileImageChange(event)">
                            <small class="text-muted d-block mt-2">Max file size: 2MB. Allowed: jpg, jpeg, png, gif</small>
                        </form>

                        <div class="mt-4">
                            <div class="row">
                                <div class="col-6">
                                    <div class="border rounded p-2">
                                        @php
                                            $totalSales = $user->sales()->count();
                                        @endphp
                                        <h6 class="mb-0">{{ $totalSales }}</h6>
                                        <small>Sales</small>
                                    </div>
                                </div>
                                <div class="col-6">
                                    <div class="border rounded p-2">
                                        @php
                                            $totalPurchases = $user->purchases()->count();
                                        @endphp
                                        <h6 class="mb-0">{{ $totalPurchases }}</h6>
                                        <small>Purchases</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Account Information -->
                <div class="card radius-10 mt-3">
                    <div class="card-body">
                        <h6 class="mb-3 text-primary">Account Information</h6>
                        <div class="list-group list-group-flush">
                            <div class="list-group-item d-flex align-items-center border-0 px-0">
                                <i class="bx bx-calendar me-2"></i>
                                <div>
                                    <small class="text-muted">Member Since</small>
                                    <p class="mb-0">{{ $user->created_at->format('M d, Y') }}</p>
                                </div>
                            </div>
                            <div class="list-group-item d-flex align-items-center border-0 px-0">
                                <i class="bx bx-user-circle me-2"></i>
                                <div>
                                    <small class="text-muted">Account Status</small>
                                    <p class="mb-0">
                                        <span class="badge bg-{{ $user->is_active ? 'success' : 'danger' }}">
                                            {{ $user->is_active ? 'Active' : 'Inactive' }}
                                        </span>
                                    </p>
                                </div>
                            </div>
                            <div class="list-group-item d-flex align-items-center border-0 px-0">
                                <i class="bx bx-log-in me-2"></i>
                                <div>
                                    <small class="text-muted">Last Login</small>
                                    <p class="mb-0">
                                        @if($user->last_login_at)
                                            {{ $user->last_login_at->format('M d, Y h:i A') }}
                                        @else
                                            Never
                                        @endif
                                    </p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-md-8">
                <!-- Edit Profile Form -->
                <div class="card radius-10">
                    <div class="card-body">
                        <h6 class="mb-3 text-primary">Edit Profile Information</h6>
                        <form action="{{ route('profile.update') }}" method="POST" id="profileForm">
                            @csrf
                            @method('PUT')
                            
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Name <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control @error('name') is-invalid @enderror" 
                                           name="name" value="{{ old('name', $user->name) }}" required>
                                    @error('name')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Email <span class="text-danger">*</span></label>
                                    <input type="email" class="form-control @error('email') is-invalid @enderror" 
                                           name="email" value="{{ old('email', $user->email) }}" required>
                                    @error('email')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Phone</label>
                                    <input type="text" class="form-control @error('phone') is-invalid @enderror" 
                                           name="phone" value="{{ old('phone', $user->phone) }}">
                                    @error('phone')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Gender</label>
                                    <select class="form-select @error('gender') is-invalid @enderror" name="gender">
                                        <option value="">Select Gender</option>
                                        <option value="male" {{ old('gender', $user->gender) == 'male' ? 'selected' : '' }}>Male</option>
                                        <option value="female" {{ old('gender', $user->gender) == 'female' ? 'selected' : '' }}>Female</option>
                                        <option value="other" {{ old('gender', $user->gender) == 'other' ? 'selected' : '' }}>Other</option>
                                    </select>
                                    @error('gender')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Department</label>
                                    <input type="text" class="form-control @error('department') is-invalid @enderror" 
                                           name="department" value="{{ old('department', $user->department) }}">
                                    @error('department')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Position</label>
                                    <input type="text" class="form-control @error('position') is-invalid @enderror" 
                                           name="position" value="{{ old('position', $user->position) }}">
                                    @error('position')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Date of Birth</label>
                                    <input type="date" class="form-control @error('date_of_birth') is-invalid @enderror" 
                                           name="date_of_birth" 
                                           value="{{ old('date_of_birth', $user->date_of_birth ? $user->date_of_birth->format('Y-m-d') : '') }}">
                                    @error('date_of_birth')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Employee ID</label>
                                    <input type="text" class="form-control @error('employee_id') is-invalid @enderror" 
                                           name="employee_id" value="{{ old('employee_id', $user->employee_id) }}" 
                                           {{ $user->employee_id ? 'readonly' : '' }}>
                                    @error('employee_id')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                    @if($user->employee_id)
                                        <small class="text-muted">Employee ID cannot be changed</small>
                                    @endif
                                </div>
                                <div class="col-12 mb-3">
                                    <label class="form-label">Address</label>
                                    <textarea class="form-control @error('address') is-invalid @enderror" 
                                              name="address" rows="3">{{ old('address', $user->address) }}</textarea>
                                    @error('address')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                                <div class="col-12">
                                    <button type="submit" class="btn btn-primary">
                                        <i class="bx bx-save"></i> Update Profile
                                    </button>
                                    <a href="{{ route('dashboard') }}" class="btn btn-secondary">
                                        <i class="bx bx-x"></i> Cancel
                                    </a>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Change Password Card -->
                <div class="card radius-10 mt-3">
                    <div class="card-body">
                        <h6 class="mb-3 text-primary">Change Password</h6>
                        <form action="{{ route('profile.password.update') }}" method="POST" id="passwordForm">
                            @csrf
                            <div class="row">
                                <div class="col-md-4 mb-3">
                                    <label class="form-label">Current Password <span class="text-danger">*</span></label>
                                    <input type="password" class="form-control @error('current_password') is-invalid @enderror" 
                                           name="current_password" required>
                                    @error('current_password')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                                <div class="col-md-4 mb-3">
                                    <label class="form-label">New Password <span class="text-danger">*</span></label>
                                    <input type="password" class="form-control @error('password') is-invalid @enderror" 
                                           name="password" required>
                                    @error('password')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                                <div class="col-md-4 mb-3">
                                    <label class="form-label">Confirm Password <span class="text-danger">*</span></label>
                                    <input type="password" class="form-control" name="password_confirmation" required>
                                </div>
                                <div class="col-12">
                                    <button type="submit" class="btn btn-primary">
                                        <i class="bx bx-lock-alt"></i> Update Password
                                    </button>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Login History -->
                <div class="card radius-10 mt-3">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-center mb-3">
                            <h6 class="mb-0 text-primary">Recent Login History</h6>
                            <a href="javascript:;" class="small">View All</a>
                        </div>
                        @php
                            $loginHistory = $user->loginHistory()->latest()->take(10)->get();
                        @endphp
                        
                        @if($loginHistory->count() > 0)
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead>
                                    <tr>
                                        <th>Date & Time</th>
                                        <th>IP Address</th>
                                        <th>Device/Browser</th>
                                        <th>Status</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach($loginHistory as $login)
                                    <tr>
                                        <td>{{ $login->login_at ? $login->login_at->format('M d, Y h:i A') : 'N/A' }}</td>
                                        <td>{{ $login->ip_address ?? 'N/A' }}</td>
                                        <td>
                                            @if($login->user_agent)
                                                <small class="text-muted">{{ Str::limit($login->user_agent, 30) }}</small>
                                            @else
                                                <small class="text-muted">N/A</small>
                                            @endif
                                        </td>
                                        <td>
                                            <span class="badge bg-{{ $login->success ? 'success' : 'danger' }}">
                                                {{ $login->success ? 'Success' : 'Failed' }}
                                            </span>
                                            @if(!$login->success && $login->failure_reason)
                                                <br><small>{{ $login->failure_reason }}</small>
                                            @endif
                                        </td>
                                    </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                        @else
                            <div class="text-center py-4">
                                <i class="bx bx-history bx-lg text-muted mb-3"></i>
                                <p class="text-muted">No login history available.</p>
                            </div>
                        @endif
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@push('script')
<script>
// Handle profile image change
function handleProfileImageChange(event) {
    const file = event.target.files[0];
    if (!file) return;

    // Validate file size (2MB)
    if (file.size > 2 * 1024 * 1024) {
        Swal.fire({
            icon: 'error',
            title: 'File too large',
            text: 'Maximum file size is 2MB'
        });
        return;
    }

    // Validate file type
    const validTypes = ['image/jpeg', 'image/png', 'image/jpg', 'image/gif'];
    if (!validTypes.includes(file.type)) {
        Swal.fire({
            icon: 'error',
            title: 'Invalid file type',
            text: 'Only JPEG, PNG, JPG, and GIF files are allowed'
        });
        return;
    }

    // Show preview
    const reader = new FileReader();
    reader.onload = function(e) {
        document.getElementById('profileImagePreview').src = e.target.result;
        
        // Ask for confirmation
        Swal.fire({
            title: 'Update Profile Image?',
            text: 'Are you sure you want to update your profile picture?',
            icon: 'question',
            showCancelButton: true,
            confirmButtonColor: '#3085d6',
            cancelButtonColor: '#d33',
            confirmButtonText: 'Yes, update it!'
        }).then((result) => {
            if (result.isConfirmed) {
                // Submit the form
                document.getElementById('profileImageForm').submit();
            } else {
                // Reset preview
                document.getElementById('profileImagePreview').src = '{{ $user->profile_image_url }}';
                document.getElementById('profileImageInput').value = '';
            }
        });
    };
    reader.readAsDataURL(file);
}

// Form validation
$(document).ready(function() {
    // Profile form validation
    $('#profileForm').validate({
        rules: {
            name: {
                required: true,
                minlength: 3
            },
            email: {
                required: true,
                email: true
            },
            phone: {
                pattern: /^[\d\s\-\+\(\)]+$/ // Allow digits, spaces, -, +, (, )
            }
        },
        messages: {
            name: {
                required: "Please enter your name",
                minlength: "Name must be at least 3 characters"
            },
            email: {
                required: "Please enter your email address",
                email: "Please enter a valid email address"
            },
            phone: {
                pattern: "Please enter a valid phone number"
            }
        },
        errorElement: 'div',
        errorPlacement: function(error, element) {
            error.addClass('invalid-feedback');
            element.closest('.form-group').append(error);
        },
        highlight: function(element, errorClass, validClass) {
            $(element).addClass('is-invalid');
        },
        unhighlight: function(element, errorClass, validClass) {
            $(element).removeClass('is-invalid');
        }
    });

    // Password form validation
    $('#passwordForm').validate({
        rules: {
            current_password: {
                required: true
            },
            password: {
                required: true,
                minlength: 8,
                strongPassword: true
            },
            password_confirmation: {
                required: true,
                equalTo: "input[name='password']"
            }
        },
        messages: {
            current_password: {
                required: "Please enter your current password"
            },
            password: {
                required: "Please enter a new password",
                minlength: "Password must be at least 8 characters",
                strongPassword: "Password must contain at least one uppercase letter, one lowercase letter, one number and one special character"
            },
            password_confirmation: {
                required: "Please confirm your password",
                equalTo: "Passwords do not match"
            }
        },
        errorElement: 'div',
        errorPlacement: function(error, element) {
            error.addClass('invalid-feedback');
            element.closest('.form-group').append(error);
        },
        highlight: function(element, errorClass, validClass) {
            $(element).addClass('is-invalid');
        },
        unhighlight: function(element, errorClass, validClass) {
            $(element).removeClass('is-invalid');
        }
    });

    // Custom validation for strong password
    $.validator.addMethod("strongPassword", function(value, element) {
        return this.optional(element) || 
            /^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]{8,}$/.test(value);
    }, "Password must contain at least one uppercase letter, one lowercase letter, one number and one special character");
});

// Show success messages if any
@if(session('success'))
Swal.fire({
    icon: 'success',
    title: 'Success!',
    text: '{{ session("success") }}',
    timer: 3000,
    showConfirmButton: false
});
@endif

@if(session('error'))
Swal.fire({
    icon: 'error',
    title: 'Error!',
    text: '{{ session("error") }}'
});
@endif
</script>
@endpush