@extends('layouts.app')

@section('style')
<style>
    .batch-form-section {
        background: #f8f9fa;
        border-radius: 10px;
        padding: 20px;
        margin-bottom: 20px;
        border-left: 4px solid #667eea;
    }
    .form-section-title {
        font-weight: 600;
        margin-bottom: 15px;
        color: #4a5568;
        padding-bottom: 10px;
        border-bottom: 2px solid #e2e8f0;
    }
    .required-field::after {
        content: " *";
        color: #e53e3e;
    }
    .batch-info-card {
        background: white;
        border-radius: 8px;
        padding: 15px;
        border: 1px solid #e2e8f0;
        margin-bottom: 15px;
    }
    .info-label {
        color: #718096;
        font-size: 12px;
        font-weight: 500;
        margin-bottom: 4px;
    }
    .info-value {
        color: #2d3748;
        font-weight: 500;
    }
</style>
@endsection

@section('content')
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">Batches</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item"><a href="{{ route('batches.index') }}">Product Batches</a></li>
                        <li class="breadcrumb-item active" aria-current="page">Create New Batch</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <a href="{{ route('batches.index') }}" class="btn btn-outline-secondary">
                    <i class="bx bx-arrow-back"></i> Back to Batches
                </a>
            </div>
        </div>
        <!--end breadcrumb-->

        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-body">
                        <h4 class="mb-4"><i class="bx bx-plus-circle"></i> Create New Batch</h4>

                        @if(session('success'))
                            <div class="alert alert-success alert-dismissible fade show" role="alert">
                                <i class="bx bx-check-circle"></i> {{ session('success') }}
                                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                            </div>
                        @endif

                        @if(session('error'))
                            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                                <i class="bx bx-error-circle"></i> {{ session('error') }}
                                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                            </div>
                        @endif

                        @if ($errors->any())
                            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                                <i class="bx bx-error-circle"></i> Please fix the following errors:
                                <ul class="mb-0">
                                    @foreach ($errors->all() as $error)
                                        <li>{{ $error }}</li>
                                    @endforeach
                                </ul>
                                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                            </div>
                        @endif

                        <form action="{{ route('batches.store') }}" method="POST" id="batchForm">
                            @csrf

                            <!-- Product Selection Section -->
                            <div class="batch-form-section">
                                <h5 class="form-section-title required-field">Product Information</h5>
                                <div class="row g-3">
                                    <div class="col-md-6">
                                        <label class="form-label required-field">Select Product</label>
                                        <select name="product_id" class="form-select @error('product_id') is-invalid @enderror" 
                                                id="productSelect" required>
                                            <option value="">-- Select Product --</option>
                                            @foreach($products as $product)
                                                <option value="{{ $product->id }}" 
                                                        data-cost="{{ $product->cost_price }}"
                                                        data-sale="{{ $product->sale_price }}"
                                                        {{ old('product_id') == $product->id ? 'selected' : '' }}>
                                                    {{ $product->name }} ({{ $product->sku }})
                                                    @if($product->has_expiry) - Expiry Tracked @endif
                                                </option>
                                            @endforeach
                                        </select>
                                        @error('product_id')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">Supplier</label>
                                        <select name="supplier_id" class="form-select @error('supplier_id') is-invalid @enderror" 
                                                id="supplierSelect">
                                            <option value="">-- Select Supplier --</option>
                                            @foreach($suppliers as $supplier)
                                                <option value="{{ $supplier->id }}" 
                                                        {{ old('supplier_id') == $supplier->id ? 'selected' : '' }}>
                                                    {{ $supplier->name }}
                                                </option>
                                            @endforeach
                                        </select>
                                        @error('supplier_id')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                            </div>

                            <!-- Batch Information Section -->
                            <div class="batch-form-section">
                                <h5 class="form-section-title required-field">Batch Identification</h5>
                                <div class="row g-3">
                                    <div class="col-md-4">
                                        <label class="form-label required-field">Batch Number</label>
                                        <input type="text" name="batch_number" 
                                               class="form-control @error('batch_number') is-invalid @enderror"
                                               value="{{ old('batch_number', 'BATCH-' . date('Ymd') . '-' . strtoupper(substr(md5(uniqid()), 0, 6))) }}"
                                               required>
                                        @error('batch_number')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        <small class="text-muted">Unique identifier for this batch</small>
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label">Lot Number</label>
                                        <input type="text" name="lot_number" 
                                               class="form-control @error('lot_number') is-invalid @enderror"
                                               value="{{ old('lot_number') }}">
                                        @error('lot_number')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        <small class="text-muted">Manufacturer's lot number</small>
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label">Serial Number</label>
                                        <input type="text" name="serial_number" 
                                               class="form-control @error('serial_number') is-invalid @enderror"
                                               value="{{ old('serial_number') }}">
                                        @error('serial_number')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        <small class="text-muted">Unique serial number (if applicable)</small>
                                    </div>
                                </div>
                            </div>

                            <!-- Dates Section -->
                            <div class="batch-form-section">
                                <h5 class="form-section-title">Dates Information</h5>
                                <div class="row g-3">
                                    <div class="col-md-6">
                                        <label class="form-label">Manufacture Date</label>
                                        <input type="date" name="manufacture_date" 
                                               class="form-control @error('manufacture_date') is-invalid @enderror"
                                               value="{{ old('manufacture_date', date('Y-m-d')) }}"
                                               max="{{ date('Y-m-d') }}">
                                        @error('manufacture_date')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        <small class="text-muted">Date when product was manufactured</small>
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">Expiry Date</label>
                                        <input type="date" name="expiry_date" 
                                               class="form-control @error('expiry_date') is-invalid @enderror"
                                               value="{{ old('expiry_date') }}"
                                               min="{{ date('Y-m-d') }}">
                                        @error('expiry_date')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        <small class="text-muted">Product expiry date (if applicable)</small>
                                    </div>
                                </div>
                            </div>

                            <!-- Stock Information Section -->
                            <div class="batch-form-section">
                                <h5 class="form-section-title required-field">Stock Information</h5>
                                <div class="row g-3">
                                    <div class="col-md-4">
                                        <label class="form-label required-field">Initial Quantity</label>
                                        <input type="number" name="initial_quantity" 
                                               class="form-control @error('initial_quantity') is-invalid @enderror"
                                               value="{{ old('initial_quantity', 1) }}"
                                               min="1" step="1" required>
                                        @error('initial_quantity')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        <small class="text-muted">Total quantity in this batch</small>
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label">Current Quantity</label>
                                        <input type="number" name="current_quantity" 
                                               class="form-control @error('current_quantity') is-invalid @enderror"
                                               value="{{ old('current_quantity', old('initial_quantity', 1)) }}"
                                               min="0" step="1">
                                        @error('current_quantity')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        <small class="text-muted">Available quantity (defaults to initial)</small>
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label">Warehouse Location</label>
                                        <input type="text" name="warehouse_location" 
                                               class="form-control @error('warehouse_location') is-invalid @enderror"
                                               value="{{ old('warehouse_location') }}"
                                               placeholder="e.g., Shelf A-1, Rack B-2">
                                        @error('warehouse_location')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        <small class="text-muted">Storage location for this batch</small>
                                    </div>
                                </div>
                            </div>

                            <!-- Pricing Section -->
                            <div class="batch-form-section">
                                <h5 class="form-section-title required-field">Pricing Information</h5>
                                <div class="row g-3">
                                    <div class="col-md-6">
                                        <label class="form-label required-field">Batch Cost Price ($)</label>
                                        <input type="number" name="batch_cost_price" 
                                               class="form-control @error('batch_cost_price') is-invalid @enderror"
                                               id="batchCostPrice"
                                               value="{{ old('batch_cost_price') }}" 
                                               min="0" step="0.01" required>
                                        @error('batch_cost_price')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        <small class="text-muted">Cost price per unit for this batch</small>
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">Batch Sale Price ($)</label>
                                        <input type="number" name="batch_sale_price" 
                                               class="form-control @error('batch_sale_price') is-invalid @enderror"
                                               id="batchSalePrice"
                                               value="{{ old('batch_sale_price') }}" 
                                               min="0" step="0.01">
                                        @error('batch_sale_price')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        <small class="text-muted">Sale price per unit for this batch</small>
                                    </div>
                                </div>
                            </div>

                            <!-- Quality Control Section -->
                            <div class="batch-form-section">
                                <h5 class="form-section-title">Quality Control</h5>
                                <div class="row g-3">
                                    <div class="col-md-6">
                                        <label class="form-label">Quality Status</label>
                                        <select name="quality_status" class="form-select @error('quality_status') is-invalid @enderror">
                                            <option value="pending" {{ old('quality_status') == 'pending' ? 'selected' : '' }}>Pending Review</option>
                                            <option value="approved" {{ old('quality_status', 'approved') == 'approved' ? 'selected' : '' }}>Approved</option>
                                            <option value="rejected" {{ old('quality_status') == 'rejected' ? 'selected' : '' }}>Rejected</option>
                                            <option value="quarantine" {{ old('quality_status') == 'quarantine' ? 'selected' : '' }}>Quarantine</option>
                                        </select>
                                        @error('quality_status')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">Quality Notes</label>
                                        <textarea name="quality_notes" class="form-control @error('quality_notes') is-invalid @enderror" 
                                                  rows="2">{{ old('quality_notes') }}</textarea>
                                        @error('quality_notes')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        <small class="text-muted">Any quality inspection notes</small>
                                    </div>
                                </div>
                            </div>

                            <!-- Additional Information -->
                            <div class="batch-form-section">
                                <h5 class="form-section-title">Additional Information</h5>
                                <div class="row g-3">
                                    <div class="col-12">
                                        <label class="form-label">Notes</label>
                                        <textarea name="notes" class="form-control @error('notes') is-invalid @enderror" 
                                                  rows="3" placeholder="Any additional notes about this batch...">{{ old('notes') }}</textarea>
                                        @error('notes')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                            </div>

                            <!-- Form Actions -->
                            <div class="row mt-4">
                                <div class="col-12">
                                    <div class="d-flex justify-content-between">
                                        <a href="{{ route('batches.index') }}" class="btn btn-secondary">
                                            <i class="bx bx-x"></i> Cancel
                                        </a>
                                        <div class="d-flex gap-2">
                                            <button type="button" class="btn btn-outline-primary" onclick="saveAsDraft()">
                                                <i class="bx bx-save"></i> Save as Draft
                                            </button>
                                            <button type="submit" class="btn btn-primary">
                                                <i class="bx bx-check"></i> Create Batch
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Product Quick View Modal -->
<div class="modal fade" id="productInfoModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Product Information</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div id="productInfoContent">
                    <div class="text-center py-5">
                        <i class="bx bx-package display-4 text-muted"></i>
                        <p class="text-muted mt-3">Select a product to view details</p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script>
    $(document).ready(function() {
        // Auto-generate batch number
        function generateBatchNumber() {
            const date = new Date().toISOString().slice(0,10).replace(/-/g, '');
            const random = Math.random().toString(36).substr(2, 6).toUpperCase();
            return `BATCH-${date}-${random}`;
        }

        // Update batch number if it starts with BATCH-
        $('input[name="batch_number"]').on('click', function() {
            if ($(this).val().startsWith('BATCH-')) {
                $(this).val(generateBatchNumber());
            }
        });

        // When product is selected
        $('#productSelect').change(function() {
            const selectedOption = $(this).find(':selected');
            const productId = selectedOption.val();
            const costPrice = selectedOption.data('cost');
            const salePrice = selectedOption.data('sale');
            
            if (productId) {
                // Update pricing fields
                if (!$('#batchCostPrice').val()) {
                    $('#batchCostPrice').val(costPrice);
                }
                if (!$('#batchSalePrice').val()) {
                    $('#batchSalePrice').val(salePrice);
                }
                
                // Load product details
                loadProductInfo(productId);
            } else {
                // Clear pricing fields if no product selected
                if (!$('#batchCostPrice').attr('data-manual')) {
                    $('#batchCostPrice').val('');
                }
                if (!$('#batchSalePrice').attr('data-manual')) {
                    $('#batchSalePrice').val('');
                }
                $('#productInfoContent').html(`
                    <div class="text-center py-5">
                        <i class="bx bx-package display-4 text-muted"></i>
                        <p class="text-muted mt-3">Select a product to view details</p>
                    </div>
                `);
            }
        });

        // Mark pricing fields as manually edited
        $('#batchCostPrice, #batchSalePrice').on('input', function() {
            $(this).attr('data-manual', 'true');
        });

        // Load product information
        function loadProductInfo(productId) {
            $.ajax({
                url: '{{ route("products.show", ":id") }}'.replace(':id', productId),
                type: 'GET',
                success: function(response) {
                    const product = response.product || response.data;
                    if (product) {
                        const productInfo = `
                            <div class="row">
                                <div class="col-md-4">
                                    <div class="batch-info-card">
                                        <div class="info-label">Product Name</div>
                                        <div class="info-value">${product.name}</div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="batch-info-card">
                                        <div class="info-label">SKU</div>
                                        <div class="info-value">${product.sku || 'N/A'}</div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="batch-info-card">
                                        <div class="info-label">Current Stock</div>
                                        <div class="info-value">${product.stock || 0} units</div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="batch-info-card">
                                        <div class="info-label">Cost Price</div>
                                        <div class="info-value">$${parseFloat(product.cost_price || 0).toFixed(2)}</div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="batch-info-card">
                                        <div class="info-label">Sale Price</div>
                                        <div class="info-value">$${parseFloat(product.sale_price || 0).toFixed(2)}</div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="batch-info-card">
                                        <div class="info-label">Expiry Tracking</div>
                                        <div class="info-value">
                                            <span class="badge ${product.has_expiry ? 'bg-success' : 'bg-secondary'}">
                                                ${product.has_expiry ? 'Enabled' : 'Disabled'}
                                            </span>
                                        </div>
                                    </div>
                                </div>
                                ${product.category ? `
                                <div class="col-md-6">
                                    <div class="batch-info-card">
                                        <div class="info-label">Category</div>
                                        <div class="info-value">${product.category.name}</div>
                                    </div>
                                </div>
                                ` : ''}
                                ${product.brand ? `
                                <div class="col-md-6">
                                    <div class="batch-info-card">
                                        <div class="info-label">Brand</div>
                                        <div class="info-value">${product.brand.name}</div>
                                    </div>
                                </div>
                                ` : ''}
                            </div>
                        `;
                        $('#productInfoContent').html(productInfo);
                    }
                },
                error: function() {
                    $('#productInfoContent').html(`
                        <div class="alert alert-danger">
                            <i class="bx bx-error"></i> Failed to load product information
                        </div>
                    `);
                }
            });
        }

        // Show product info modal on button click
        $(document).on('click', '.view-product-info', function() {
            const productId = $('#productSelect').val();
            if (productId) {
                loadProductInfo(productId);
                $('#productInfoModal').modal('show');
            } else {
                Swal.fire({
                    icon: 'warning',
                    title: 'No Product Selected',
                    text: 'Please select a product first'
                });
            }
        });

        // Auto-set current quantity when initial quantity changes
        $('input[name="initial_quantity"]').on('input', function() {
            const initialQty = $(this).val();
            const currentQty = $('input[name="current_quantity"]').val();
            
            // Only auto-set if current quantity is empty or equals previous initial quantity
            if (!currentQty || currentQty == $('input[name="current_quantity"]').data('prev-initial')) {
                $('input[name="current_quantity"]').val(initialQty);
                $('input[name="current_quantity"]').data('prev-initial', initialQty);
            }
        });

        // Save as draft functionality
        window.saveAsDraft = function() {
            // Set a hidden field or modify form data
            $('#batchForm').append('<input type="hidden" name="is_draft" value="1">');
            $('#batchForm').submit();
        };

        // Form validation before submit
        $('#batchForm').submit(function(e) {
            const productId = $('#productSelect').val();
            const batchNumber = $('input[name="batch_number"]').val();
            const initialQty = $('input[name="initial_quantity"]').val();
            const costPrice = $('#batchCostPrice').val();
            
            if (!productId) {
                e.preventDefault();
                Swal.fire({
                    icon: 'error',
                    title: 'Product Required',
                    text: 'Please select a product'
                });
                $('#productSelect').focus();
                return false;
            }
            
            if (!batchNumber) {
                e.preventDefault();
                Swal.fire({
                    icon: 'error',
                    title: 'Batch Number Required',
                    text: 'Please enter a batch number'
                });
                $('input[name="batch_number"]').focus();
                return false;
            }
            
            if (!initialQty || initialQty < 1) {
                e.preventDefault();
                Swal.fire({
                    icon: 'error',
                    title: 'Invalid Quantity',
                    text: 'Please enter a valid initial quantity'
                });
                $('input[name="initial_quantity"]').focus();
                return false;
            }
            
            if (!costPrice || costPrice < 0) {
                e.preventDefault();
                Swal.fire({
                    icon: 'error',
                    title: 'Invalid Cost Price',
                    text: 'Please enter a valid cost price'
                });
                $('#batchCostPrice').focus();
                return false;
            }
            
            // Show loading indicator
            $('button[type="submit"]').html('<i class="bx bx-loader bx-spin"></i> Creating...');
            $('button[type="submit"]').prop('disabled', true);
        });

        // Check batch number uniqueness
        $('input[name="batch_number"]').on('blur', function() {
            const batchNumber = $(this).val();
            if (batchNumber && batchNumber.length >= 5) {
                $.ajax({
                    url: '{{ route("batches.index") }}?check_batch=' + encodeURIComponent(batchNumber),
                    type: 'GET',
                    success: function(response) {
                        if (response.exists) {
                            Swal.fire({
                                icon: 'warning',
                                title: 'Batch Number Exists',
                                text: 'This batch number already exists. Please use a different one.',
                                timer: 3000
                            });
                            $('input[name="batch_number"]').addClass('is-invalid');
                        } else {
                            $('input[name="batch_number"]').removeClass('is-invalid');
                        }
                    }
                });
            }
        });

        // Initialize date pickers with restrictions
        const today = new Date().toISOString().split('T')[0];
        $('input[type="date"][max]').attr('max', today);
        $('input[name="expiry_date"]').attr('min', today);
    });
</script>
@endsection