@extends('layouts.app')

@section('style')
<style>
    .price-change-badge {
        padding: 4px 8px;
        border-radius: 20px;
        font-size: 11px;
        font-weight: 600;
    }
    .price-increase {
        background-color: #d1fae5;
        color: #065f46;
    }
    .price-decrease {
        background-color: #fee2e2;
        color: #991b1b;
    }
    .price-no-change {
        background-color: #e5e7eb;
        color: #6b7280;
    }
    .price-trend-chart {
        height: 200px;
        position: relative;
    }
    .price-stat-card {
        border: 1px solid #e5e7eb;
        border-radius: 8px;
        padding: 15px;
        text-align: center;
        transition: all 0.3s;
    }
    .price-stat-card:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(0,0,0,0.1);
    }
    .stat-value {
        font-size: 24px;
        font-weight: bold;
        margin: 10px 0;
    }
    .stat-label {
        font-size: 12px;
        color: #6b7280;
        text-transform: uppercase;
    }
</style>
@endsection

@section('content')
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">Products</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item"><a href="{{ route('products.index') }}">Products</a></li>
                        <li class="breadcrumb-item"><a href="{{ route('products.show', $product) }}">{{ $product->name }}</a></li>
                        <li class="breadcrumb-item active" aria-current="page">Price History</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <div class="btn-group">
                    <a href="{{ route('products.show', $product) }}" class="btn btn-outline-secondary">
                        <i class="bx bx-arrow-back"></i> Back to Product
                    </a>
                    <a href="{{ route('products.pricing.index') }}" class="btn btn-primary">
                        <i class="bx bx-dollar-circle"></i> Pricing Management
                    </a>
                </div>
            </div>
        </div>
        <!--end breadcrumb-->

        <!-- Product Info -->
        <div class="card mb-4">
            <div class="card-body">
                <div class="row">
                    <div class="col-md-8">
                        <h5 class="mb-2">{{ $product->name }}</h5>
                        <div class="row g-3">
                            <div class="col-sm-6 col-md-4">
                                <small class="text-muted d-block">SKU</small>
                                <strong>{{ $product->sku }}</strong>
                            </div>
                            <div class="col-sm-6 col-md-4">
                                <small class="text-muted d-block">Current Cost Price</small>
                                <strong>${{ number_format($product->cost_price, 2) }}</strong>
                            </div>
                            <div class="col-sm-6 col-md-4">
                                <small class="text-muted d-block">Current Sale Price</small>
                                <strong>${{ number_format($product->sale_price, 2) }}</strong>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="d-flex justify-content-end">
                            <div class="text-end">
                                <div class="h4 mb-1">{{ number_format($product->profit_margin, 1) }}%</div>
                                <small class="text-muted">Profit Margin</small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Statistics -->
        <div class="row row-cols-1 row-cols-md-4 row-cols-xl-4 mb-4">
            <div class="col">
                <div class="price-stat-card">
                    <div class="stat-label">Total Changes</div>
                    <div class="stat-value">{{ $priceHistories->total() }}</div>
                    <small class="text-muted">All time</small>
                </div>
            </div>
            <div class="col">
                <div class="price-stat-card">
                    <div class="stat-label">Last 30 Days</div>
                    <div class="stat-value">
                        {{ PriceHistory::where('product_id', $product->id)
                            ->where('created_at', '>=', now()->subDays(30))
                            ->count() }}
                    </div>
                    <small class="text-muted">Recent changes</small>
                </div>
            </div>
            <div class="col">
                <div class="price-stat-card">
                    <div class="stat-label">Highest Price</div>
                    <div class="stat-value">
                        ${{ number_format(PriceHistory::where('product_id', $product->id)
                            ->where('price_type', 'sale_price')
                            ->max('new_price') ?? $product->sale_price, 2) }}
                    </div>
                    <small class="text-muted">Sale price</small>
                </div>
            </div>
            <div class="col">
                <div class="price-stat-card">
                    <div class="stat-label">Lowest Price</div>
                    <div class="stat-value">
                        ${{ number_format(PriceHistory::where('product_id', $product->id)
                            ->where('price_type', 'sale_price')
                            ->min('new_price') ?? $product->sale_price, 2) }}
                    </div>
                    <small class="text-muted">Sale price</small>
                </div>
            </div>
        </div>

        <!-- Price History Table -->
        <div class="card">
            <div class="card-body">
                <h5 class="card-title mb-4">Price History</h5>

                @if($priceHistories->isEmpty())
                    <div class="text-center py-5">
                        <i class="bx bx-history display-4 text-muted"></i>
                        <h5 class="mt-3 text-muted">No price history found</h5>
                        <p class="text-muted">This product's prices haven't been changed yet.</p>
                    </div>
                @else
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>Date & Time</th>
                                    <th>Price Type</th>
                                    <th>Old Price</th>
                                    <th>New Price</th>
                                    <th>Change</th>
                                    <th>Changed By</th>
                                    <th>Reason</th>
                                    <th>Notes</th>
                                </tr>
                            </thead>
                            <tbody>
                                @foreach($priceHistories as $history)
                                    @php
                                        $change = $history->new_price - $history->old_price;
                                        $percentChange = $history->old_price > 0 ? 
                                            ($change / $history->old_price) * 100 : 0;
                                    @endphp
                                    <tr>
                                        <td>
                                            {{ $history->created_at->format('Y-m-d') }}<br>
                                            <small class="text-muted">{{ $history->created_at->format('H:i:s') }}</small>
                                        </td>
                                        <td>
                                            <span class="badge 
                                                @if($history->price_type == 'cost_price') bg-info
                                                @elseif($history->price_type == 'sale_price') bg-success
                                                @else bg-warning @endif">
                                                {{ $history->price_type_label }}
                                            </span>
                                        </td>
                                        <td>${{ number_format($history->old_price, 2) }}</td>
                                        <td>${{ number_format($history->new_price, 2) }}</td>
                                        <td>
                                            <div class="d-flex align-items-center">
                                                @if($change > 0)
                                                    <span class="price-change-badge price-increase me-2">
                                                        +${{ number_format($change, 2) }}
                                                    </span>
                                                    <small class="text-success">
                                                        +{{ number_format($percentChange, 1) }}%
                                                    </small>
                                                @elseif($change < 0)
                                                    <span class="price-change-badge price-decrease me-2">
                                                        -${{ number_format(abs($change), 2) }}
                                                    </span>
                                                    <small class="text-danger">
                                                        {{ number_format($percentChange, 1) }}%
                                                    </small>
                                                @else
                                                    <span class="price-change-badge price-no-change">
                                                        No Change
                                                    </span>
                                                @endif
                                            </div>
                                        </td>
                                        <td>{{ $history->user->name ?? 'System' }}</td>
                                        <td>{{ $history->change_reason }}</td>
                                        <td>
                                            @if($history->notes)
                                                <small class="text-muted">{{ $history->notes }}</small>
                                            @else
                                                <span class="text-muted">-</span>
                                            @endif
                                        </td>
                                    </tr>
                                @endforeach
                            </tbody>
                        </table>
                    </div>

                    <!-- Pagination -->
                    <div class="d-flex justify-content-between align-items-center mt-4">
                        <div class="pagination-info">
                            Showing {{ $priceHistories->firstItem() }} to {{ $priceHistories->lastItem() }} of {{ $priceHistories->total() }} entries
                        </div>
                        <nav>
                            {{ $priceHistories->links() }}
                        </nav>
                    </div>
                @endif
            </div>
        </div>

        <!-- Export Options -->
        <div class="card mt-4">
            <div class="card-body">
                <h5 class="card-title mb-4">Export Options</h5>
                <div class="d-flex gap-2">
                    <a href="{{ route('price-histories.export') }}?product_id={{ $product->id }}" 
                       class="btn btn-outline-primary">
                        <i class="bx bx-download"></i> Export as CSV
                    </a>
                    <a href="{{ route('price-histories.trends', $product) }}" 
                       class="btn btn-outline-info">
                        <i class="bx bx-line-chart"></i> View Trends
                    </a>
                    <a href="{{ route('price-histories.compare') }}?product_ids[]={{ $product->id }}" 
                       class="btn btn-outline-warning">
                        <i class="bx bx-compare"></i> Compare Prices
                    </a>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script>
    // ============================================
    // PRICE HISTORY JAVASCRIPT
    // ============================================
    
    $(document).ready(function() {
        console.log('Price history page loaded for product #{{ $product->id }}');
        
        // Initialize tooltips
        $('[data-bs-toggle="tooltip"]').tooltip();
        
        // Initialize date range picker if element exists
        if ($('#dateRangeFilter').length) {
            initializeDateRangePicker();
        }
        
        // Initialize filters
        initializeFilters();
        
        // Initialize export functionality
        initializeExportOptions();
        
        // Initialize price trend chart if data exists
        initializePriceChart();
    });

    function initializeDateRangePicker() {
        $('#dateRangeFilter').daterangepicker({
            opens: 'left',
            startDate: moment().subtract(30, 'days'),
            endDate: moment(),
            ranges: {
                'Today': [moment(), moment()],
                'Yesterday': [moment().subtract(1, 'days'), moment().subtract(1, 'days')],
                'Last 7 Days': [moment().subtract(6, 'days'), moment()],
                'Last 30 Days': [moment().subtract(29, 'days'), moment()],
                'This Month': [moment().startOf('month'), moment().endOf('month')],
                'Last Month': [moment().subtract(1, 'month').startOf('month'), moment().subtract(1, 'month').endOf('month')]
            },
            locale: {
                format: 'YYYY-MM-DD',
                separator: ' to ',
                applyLabel: 'Apply',
                cancelLabel: 'Cancel',
                fromLabel: 'From',
                toLabel: 'To',
                customRangeLabel: 'Custom',
                weekLabel: 'W',
                daysOfWeek: ['Su', 'Mo', 'Tu', 'We', 'Th', 'Fr', 'Sa'],
                monthNames: [
                    'January', 'February', 'March', 'April', 'May', 'June',
                    'July', 'August', 'September', 'October', 'November', 'December'
                ],
                firstDay: 1
            }
        }, function(start, end, label) {
            console.log('Date range selected:', start.format('YYYY-MM-DD'), 'to', end.format('YYYY-MM-DD'));
            
            // Get current URL parameters
            const urlParams = new URLSearchParams(window.location.search);
            urlParams.set('start_date', start.format('YYYY-MM-DD'));
            urlParams.set('end_date', end.format('YYYY-MM-DD'));
            
            // Reload page with new filters
            window.location.href = window.location.pathname + '?' + urlParams.toString();
        });
    }

    function initializeFilters() {
        // Price type filter
        $('select[name="price_type"]').change(function() {
            console.log('Price type filter changed:', $(this).val());
            submitFilterForm();
        });

        // User filter
        $('select[name="user_id"]').change(function() {
            console.log('User filter changed:', $(this).val());
            submitFilterForm();
        });

        // Change reason filter
        $('select[name="change_reason"]').change(function() {
            console.log('Change reason filter changed:', $(this).val());
            submitFilterForm();
        });

        // Sort by filter
        $('select[name="sort_by"]').change(function() {
            console.log('Sort by filter changed:', $(this).val());
            submitFilterForm();
        });

        // Sort order filter
        $('select[name="sort_order"]').change(function() {
            console.log('Sort order filter changed:', $(this).val());
            submitFilterForm();
        });
    }

    function submitFilterForm() {
        // Show loading indicator
        showLoading();
        
        // Submit the filter form
        $('#filterForm').submit();
    }

    function showLoading() {
        // Add loading overlay to table
        const table = $('table');
        if (!table.find('.loading-overlay').length) {
            table.css('position', 'relative').append(`
                <div class="loading-overlay" style="
                    position: absolute;
                    top: 0;
                    left: 0;
                    width: 100%;
                    height: 100%;
                    background: rgba(255, 255, 255, 0.8);
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    z-index: 1000;
                ">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                </div>
            `);
        }
    }

    function initializeExportOptions() {
        // CSV Export button
        $('.btn-export-csv').click(function(e) {
            e.preventDefault();
            
            const url = $(this).attr('href');
            const productId = {{ $product->id }};
            
            Swal.fire({
                title: 'Export Price History',
                text: 'Would you like to export the price history as CSV?',
                icon: 'question',
                showCancelButton: true,
                confirmButtonText: 'Export',
                cancelButtonText: 'Cancel',
                showLoaderOnConfirm: true,
                preConfirm: () => {
                    return new Promise((resolve) => {
                        // Create download link
                        const downloadLink = document.createElement('a');
                        downloadLink.href = url + '?product_id=' + productId + '&' + window.location.search.substring(1);
                        downloadLink.download = 'price-history-{{ $product->sku }}-' + moment().format('YYYY-MM-DD') + '.csv';
                        document.body.appendChild(downloadLink);
                        downloadLink.click();
                        document.body.removeChild(downloadLink);
                        
                        resolve();
                    });
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    Swal.fire({
                        icon: 'success',
                        title: 'Export Started',
                        text: 'Your CSV file download should begin shortly.',
                        timer: 2000,
                        showConfirmButton: false
                    });
                }
            });
        });

        // Print button
        $('.btn-print').click(function(e) {
            e.preventDefault();
            
            Swal.fire({
                title: 'Print Price History',
                text: 'Would you like to print the price history?',
                icon: 'question',
                showCancelButton: true,
                confirmButtonText: 'Print',
                cancelButtonText: 'Cancel'
            }).then((result) => {
                if (result.isConfirmed) {
                    window.print();
                }
            });
        });
    }

    function initializePriceChart() {
        // Check if we have price history data
        const priceData = @json($priceHistories->items());
        
        if (priceData.length > 0 && typeof Chart !== 'undefined') {
            createPriceTrendChart(priceData);
        }
    }

    function createPriceTrendChart(priceData) {
        // Group data by price type
        const salePriceData = priceData.filter(item => item.price_type === 'sale_price');
        const costPriceData = priceData.filter(item => item.price_type === 'cost_price');
        
        if (salePriceData.length === 0) return;
        
        // Prepare chart data
        const labels = salePriceData.map(item => 
            moment(item.created_at).format('MMM D')
        ).reverse();
        
        const salePrices = salePriceData.map(item => item.new_price).reverse();
        const costPrices = costPriceData.map(item => item.new_price).reverse();
        
        // Get canvas context
        const ctx = document.getElementById('priceTrendChart');
        if (!ctx) return;
        
        new Chart(ctx, {
            type: 'line',
            data: {
                labels: labels,
                datasets: [
                    {
                        label: 'Sale Price',
                        data: salePrices,
                        borderColor: '#28a745',
                        backgroundColor: 'rgba(40, 167, 69, 0.1)',
                        borderWidth: 2,
                        fill: true,
                        tension: 0.4
                    },
                    {
                        label: 'Cost Price',
                        data: costPrices,
                        borderColor: '#17a2b8',
                        backgroundColor: 'rgba(23, 162, 184, 0.1)',
                        borderWidth: 2,
                        fill: true,
                        tension: 0.4
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'top',
                    },
                    tooltip: {
                        mode: 'index',
                        intersect: false,
                        callbacks: {
                            label: function(context) {
                                return `${context.dataset.label}: $${context.parsed.y.toFixed(2)}`;
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: false,
                        ticks: {
                            callback: function(value) {
                                return '$' + value.toFixed(2);
                            }
                        }
                    }
                },
                interaction: {
                    intersect: false,
                    mode: 'nearest'
                }
            }
        });
    }

    // Keyboard shortcuts
    $(document).keydown(function(e) {
        // Ctrl/Cmd + P to print
        if ((e.ctrlKey || e.metaKey) && e.key === 'p') {
            e.preventDefault();
            window.print();
        }
        
        // Ctrl/Cmd + E to export
        if ((e.ctrlKey || e.metaKey) && e.key === 'e') {
            e.preventDefault();
            $('.btn-export-csv').click();
        }
        
        // Escape to clear filters
        if (e.key === 'Escape') {
            clearFilters();
        }
    });

    function clearFilters() {
        Swal.fire({
            title: 'Clear Filters',
            text: 'Are you sure you want to clear all filters?',
            icon: 'question',
            showCancelButton: true,
            confirmButtonText: 'Clear',
            cancelButtonText: 'Cancel'
        }).then((result) => {
            if (result.isConfirmed) {
                window.location.href = '{{ route("products.price-history", $product) }}';
            }
        });
    }

    // Initialize table sorting
    function initializeTableSorting() {
        $('th[data-sortable]').click(function() {
            const sortBy = $(this).data('sort-by');
            const currentSort = new URLSearchParams(window.location.search).get('sort_by');
            const currentOrder = new URLSearchParams(window.location.search).get('sort_order');
            
            let newOrder = 'asc';
            if (currentSort === sortBy && currentOrder === 'asc') {
                newOrder = 'desc';
            }
            
            // Update URL and reload
            const urlParams = new URLSearchParams(window.location.search);
            urlParams.set('sort_by', sortBy);
            urlParams.set('sort_order', newOrder);
            
            window.location.href = window.location.pathname + '?' + urlParams.toString();
        });
    }

    // Add chart canvas to your Blade template if you want charts:
    // <div class="card mt-4">
    //     <div class="card-body">
    //         <h5 class="card-title mb-4">Price Trends</h5>
    //         <canvas id="priceTrendChart" height="300"></canvas>
    //     </div>
    // </div>

    console.log('Price history JavaScript loaded successfully');
</script>

<!-- Include Chart.js if you want charts -->
<!-- <script src="https://cdn.jsdelivr.net/npm/chart.js"></script> -->

<!-- Include Moment.js for date handling -->
<!-- <script src="https://cdn.jsdelivr.net/npm/moment@2.29.1/min/moment.min.js"></script> -->

<!-- Include Date Range Picker -->
<!-- <script src="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.min.js"></script> -->
<!-- <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/daterangepicker/daterangepicker.css" /> -->
@endsection