@extends('layouts.app')

@section('style')
<style>
    .analytics-card {
        border: 1px solid #e5e7eb;
        border-radius: 10px;
        padding: 20px;
        margin-bottom: 20px;
        transition: all 0.3s;
    }
    .analytics-card:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(0,0,0,0.1);
    }
    .analytics-value {
        font-size: 28px;
        font-weight: bold;
        margin: 10px 0;
    }
    .analytics-label {
        font-size: 13px;
        color: #6b7280;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    .trend-up {
        color: #10b981;
    }
    .trend-down {
        color: #ef4444;
    }
    .chart-container {
        height: 300px;
        position: relative;
        margin: 20px 0;
    }
    .activity-timeline {
        border-left: 2px solid #3b82f6;
        padding-left: 20px;
        margin-left: 10px;
    }
    .activity-item {
        position: relative;
        padding-bottom: 20px;
    }
    .activity-item::before {
        content: '';
        position: absolute;
        left: -26px;
        top: 0;
        width: 12px;
        height: 12px;
        border-radius: 50%;
        background: #3b82f6;
        border: 2px solid white;
    }
    .activity-time {
        font-size: 12px;
        color: #6b7280;
    }
    .activity-desc {
        font-size: 14px;
        margin-top: 5px;
    }
</style>
@endsection

@section('content')
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">Analytics</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item"><a href="{{ route('price-histories.index') }}">Price History</a></li>
                        <li class="breadcrumb-item active" aria-current="page">Price Analytics</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <div class="btn-group">
                    <a href="{{ route('price-histories.export') }}" class="btn btn-outline-primary">
                        <i class="bx bx-export"></i> Export Data
                    </a>
                    <a href="{{ route('price-histories.index') }}" class="btn btn-primary">
                        <i class="bx bx-history"></i> View History
                    </a>
                </div>
            </div>
        </div>
        <!--end breadcrumb-->

        <!-- Quick Stats -->
        <div class="row row-cols-1 row-cols-md-3 row-cols-xl-3 mb-4">
            <div class="col">
                <div class="analytics-card">
                    <div class="analytics-label">Price Changes Today</div>
                    <div class="analytics-value">{{ $stats['total_changes_today'] }}</div>
                    <div class="d-flex justify-content-between align-items-center">
                        <small class="text-muted">Daily average: {{ round($stats['avg_daily_changes'], 1) }}</small>
                        @if($stats['total_changes_today'] > $stats['avg_daily_changes'])
                            <span class="trend-up"><i class="bx bx-up-arrow-alt"></i> Above average</span>
                        @else
                            <span class="trend-down"><i class="bx bx-down-arrow-alt"></i> Below average</span>
                        @endif
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="analytics-card">
                    <div class="analytics-label">Price Changes This Week</div>
                    <div class="analytics-value">{{ $stats['total_changes_week'] }}</div>
                    <div class="d-flex justify-content-between align-items-center">
                        <small class="text-muted">Weekly trend</small>
                        @php
                            $lastWeek = PriceHistory::where('created_at', '>=', now()->subDays(14))
                                ->where('created_at', '<', now()->subDays(7))
                                ->count();
                            $weekChange = $lastWeek > 0 ? 
                                (($stats['total_changes_week'] - $lastWeek) / $lastWeek) * 100 : 0;
                        @endphp
                        @if($weekChange > 0)
                            <span class="trend-up"><i class="bx bx-up-arrow-alt"></i> {{ round($weekChange, 1) }}%</span>
                        @else
                            <span class="trend-down"><i class="bx bx-down-arrow-alt"></i> {{ round(abs($weekChange), 1) }}%</span>
                        @endif
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="analytics-card">
                    <div class="analytics-label">Price Changes This Month</div>
                    <div class="analytics-value">{{ $stats['total_changes_month'] }}</div>
                    <div class="d-flex justify-content-between align-items-center">
                        <small class="text-muted">Monthly overview</small>
                        @php
                            $lastMonth = PriceHistory::where('created_at', '>=', now()->subDays(60))
                                ->where('created_at', '<', now()->subDays(30))
                                ->count();
                            $monthChange = $lastMonth > 0 ? 
                                (($stats['total_changes_month'] - $lastMonth) / $lastMonth) * 100 : 0;
                        @endphp
                        @if($monthChange > 0)
                            <span class="trend-up"><i class="bx bx-up-arrow-alt"></i> {{ round($monthChange, 1) }}%</span>
                        @else
                            <span class="trend-down"><i class="bx bx-down-arrow-alt"></i> {{ round(abs($monthChange), 1) }}%</span>
                        @endif
                    </div>
                </div>
            </div>
        </div>

        <!-- Main Content -->
        <div class="row">
            <!-- Recent Activity -->
            <div class="col-lg-8">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title mb-4">Recent Price Changes</h5>
                        
                        @if($recentChanges->isEmpty())
                            <div class="text-center py-5">
                                <i class="bx bx-line-chart display-4 text-muted"></i>
                                <h5 class="mt-3 text-muted">No recent price changes</h5>
                            </div>
                        @else
                            <div class="activity-timeline">
                                @foreach($recentChanges as $change)
                                    <div class="activity-item">
                                        <div class="activity-time">
                                            {{ $change->created_at->diffForHumans() }}
                                        </div>
                                        <div class="activity-desc">
                                            <strong>{{ $change->product->name }}</strong> - 
                                            {{ $change->price_type_label }} changed from 
                                            <strong>${{ number_format($change->old_price, 2) }}</strong> to 
                                            <strong>${{ number_format($change->new_price, 2) }}</strong>
                                            @php
                                                $changeAmount = $change->new_price - $change->old_price;
                                            @endphp
                                            @if($changeAmount > 0)
                                                <span class="badge bg-success ms-2">
                                                    +${{ number_format($changeAmount, 2) }}
                                                </span>
                                            @elseif($changeAmount < 0)
                                                <span class="badge bg-danger ms-2">
                                                    -${{ number_format(abs($changeAmount), 2) }}
                                                </span>
                                            @endif
                                            <br>
                                            <small class="text-muted">
                                                By {{ $change->user->name ?? 'System' }} • 
                                                {{ $change->change_reason }}
                                            </small>
                                        </div>
                                    </div>
                                @endforeach
                            </div>
                        @endif
                    </div>
                </div>

                <!-- Most Active Products -->
                <div class="card mt-4">
                    <div class="card-body">
                        <h5 class="card-title mb-4">Most Active Products (Last 30 Days)</h5>
                        
                        @if($stats['most_active_products']->isEmpty())
                            <div class="text-center py-4">
                                <i class="bx bx-package display-4 text-muted"></i>
                                <p class="text-muted mt-2">No price changes in the last 30 days</p>
                            </div>
                        @else
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>Product</th>
                                            <th>SKU</th>
                                            <th>Price Changes</th>
                                            <th>Current Price</th>
                                            <th>Last Change</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        @foreach($stats['most_active_products'] as $item)
                                            <tr>
                                                <td>
                                                    <a href="{{ route('products.show', $item->product) }}" class="text-dark">
                                                        {{ $item->product->name }}
                                                    </a>
                                                </td>
                                                <td>{{ $item->product->sku }}</td>
                                                <td>
                                                    <span class="badge bg-primary">{{ $item->changes }}</span>
                                                </td>
                                                <td>${{ number_format($item->product->sale_price, 2) }}</td>
                                                <td>
                                                    @php
                                                        $lastChange = PriceHistory::where('product_id', $item->product_id)
                                                            ->latest()
                                                            ->first();
                                                    @endphp
                                                    @if($lastChange)
                                                        {{ $lastChange->created_at->diffForHumans() }}
                                                    @else
                                                        <span class="text-muted">N/A</span>
                                                    @endif
                                                </td>
                                            </tr>
                                        @endforeach
                                    </tbody>
                                </table>
                            </div>
                        @endif
                    </div>
                </div>
            </div>

            <!-- Right Sidebar -->
            <div class="col-lg-4">
                <!-- Largest Changes -->
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title mb-4">Largest Price Changes</h5>
                        
                        <div class="mb-4">
                            <h6 class="text-success">Largest Increase</h6>
                            @if($stats['largest_increase'])
                                <div class="d-flex align-items-center mb-2">
                                    <div class="flex-shrink-0 me-3">
                                        <i class="bx bx-up-arrow-circle text-success" style="font-size: 24px;"></i>
                                    </div>
                                    <div class="flex-grow-1">
                                        <strong>{{ $stats['largest_increase']->product->name }}</strong>
                                        <div class="text-success">
                                            +${{ number_format($stats['largest_increase']->new_price - $stats['largest_increase']->old_price, 2) }}
                                        </div>
                                        <small class="text-muted">
                                            {{ $stats['largest_increase']->created_at->diffForHumans() }}
                                        </small>
                                    </div>
                                </div>
                            @else
                                <p class="text-muted">No price increases recorded</p>
                            @endif
                        </div>

                        <div>
                            <h6 class="text-danger">Largest Decrease</h6>
                            @if($stats['largest_decrease'])
                                <div class="d-flex align-items-center">
                                    <div class="flex-shrink-0 me-3">
                                        <i class="bx bx-down-arrow-circle text-danger" style="font-size: 24px;"></i>
                                    </div>
                                    <div class="flex-grow-1">
                                        <strong>{{ $stats['largest_decrease']->product->name }}</strong>
                                        <div class="text-danger">
                                            -${{ number_format($stats['largest_decrease']->old_price - $stats['largest_decrease']->new_price, 2) }}
                                        </div>
                                        <small class="text-muted">
                                            {{ $stats['largest_decrease']->created_at->diffForHumans() }}
                                        </small>
                                    </div>
                                </div>
                            @else
                                <p class="text-muted">No price decreases recorded</p>
                            @endif
                        </div>
                    </div>
                </div>

                <!-- Quick Actions -->
                <div class="card mt-4">
                    <div class="card-body">
                        <h5 class="card-title mb-4">Quick Actions</h5>
                        <div class="d-grid gap-2">
                            <a href="{{ route('products.pricing.index') }}" class="btn btn-primary">
                                <i class="bx bx-dollar-circle"></i> Pricing Management
                            </a>
                            <a href="{{ route('price-histories.compare') }}" class="btn btn-outline-info">
                                <i class="bx bx-compare"></i> Compare Prices
                            </a>
                            <a href="{{ route('price-histories.export') }}" class="btn btn-outline-secondary">
                                <i class="bx bx-download"></i> Export Data
                            </a>
                        </div>
                    </div>
                </div>

                <!-- Summary Stats -->
                <div class="card mt-4">
                    <div class="card-body">
                        <h5 class="card-title mb-4">Summary Statistics</h5>
                        <div class="row g-3">
                            <div class="col-6">
                                <div class="text-center p-3 border rounded">
                                    <div class="h5 mb-1">{{ PriceHistory::count() }}</div>
                                    <small class="text-muted">Total Changes</small>
                                </div>
                            </div>
                            <div class="col-6">
                                <div class="text-center p-3 border rounded">
                                    <div class="h5 mb-1">{{ Product::count() }}</div>
                                    <small class="text-muted">Total Products</small>
                                </div>
                            </div>
                            <div class="col-6">
                                <div class="text-center p-3 border rounded">
                                    <div class="h5 mb-1">
                                        {{ PriceHistory::distinct('product_id')->count() }}
                                    </div>
                                    <small class="text-muted">Products with Changes</small>
                                </div>
                            </div>
                            <div class="col-6">
                                <div class="text-center p-3 border rounded">
                                    <div class="h5 mb-1">{{ User::count() }}</div>
                                    <small class="text-muted">Users Made Changes</small>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script src="https://cdn.jsdelivr.net/npm/chartjs-adapter-date-fns/dist/chartjs-adapter-date-fns.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/luxon@3.0.4/build/global/luxon.min.js"></script>

<script>
    // ============================================
    // PRICE ANALYTICS JAVASCRIPT
    // ============================================
    
    $(document).ready(function() {
        console.log('Price analytics page loaded');
        
        // Initialize tooltips
        $('[data-bs-toggle="tooltip"]').tooltip();
        
        // Initialize charts if data exists
        initializeCharts();
        
        // Initialize date filters
        initializeDateFilters();
        
        // Initialize auto-refresh
        initializeAutoRefresh();
        
        // Initialize export functionality
        initializeExportOptions();
        
        // Initialize quick actions
        initializeQuickActions();
    });

    // ============================================
    // CHART INITIALIZATION
    // ============================================
    
    let priceTrendChart = null;
    let activityChart = null;
    let productActivityChart = null;

    function initializeCharts() {
        // Create price changes over time chart
        createPriceChangesChart();
        
        // Create activity by hour chart
        createActivityByHourChart();
        
        // Create product activity distribution chart
        createProductActivityChart();
    }

    function createPriceChangesChart() {
        const ctx = document.getElementById('priceChangesChart');
        if (!ctx) {
            console.log('Price changes chart container not found');
            return;
        }
        
        // Sample data - replace with actual data from your controller
        const data = {
            labels: @json($stats['daily_changes_labels'] ?? []),
            datasets: [{
                label: 'Price Changes',
                data: @json($stats['daily_changes_data'] ?? []),
                borderColor: '#3b82f6',
                backgroundColor: 'rgba(59, 130, 246, 0.1)',
                borderWidth: 2,
                fill: true,
                tension: 0.4
            }]
        };
        
        priceTrendChart = new Chart(ctx, {
            type: 'line',
            data: data,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'top',
                    },
                    tooltip: {
                        mode: 'index',
                        intersect: false,
                        callbacks: {
                            label: function(context) {
                                return `${context.dataset.label}: ${context.parsed.y} changes`;
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            precision: 0
                        }
                    }
                },
                interaction: {
                    intersect: false,
                    mode: 'nearest'
                }
            }
        });
    }

    function createActivityByHourChart() {
        const ctx = document.getElementById('activityByHourChart');
        if (!ctx) {
            console.log('Activity by hour chart container not found, creating placeholder');
            return;
        }
        
        // Sample data - replace with actual data from your controller
        const hours = Array.from({length: 24}, (_, i) => `${i}:00`);
        const activityData = Array.from({length: 24}, () => Math.floor(Math.random() * 10));
        
        activityChart = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: hours,
                datasets: [{
                    label: 'Price Changes by Hour',
                    data: activityData,
                    backgroundColor: 'rgba(59, 130, 246, 0.7)',
                    borderColor: 'rgba(59, 130, 246, 1)',
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            precision: 0
                        }
                    }
                }
            }
        });
    }

    function createProductActivityChart() {
        const ctx = document.getElementById('productActivityChart');
        if (!ctx) {
            console.log('Product activity chart container not found');
            return;
        }
        
        // Get data from table if available
        const productData = [];
        $('table tbody tr').each(function() {
            const productName = $(this).find('td:first-child').text().trim();
            const changeCount = parseInt($(this).find('td:nth-child(3) span').text()) || 0;
            if (productName && changeCount > 0) {
                productData.push({ product: productName, changes: changeCount });
            }
        });
        
        // Take top 10 products
        const topProducts = productData.sort((a, b) => b.changes - a.changes).slice(0, 10);
        
        productActivityChart = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: topProducts.map(p => p.product.substring(0, 20) + (p.product.length > 20 ? '...' : '')),
                datasets: [{
                    label: 'Price Changes',
                    data: topProducts.map(p => p.changes),
                    backgroundColor: [
                        'rgba(59, 130, 246, 0.7)',
                        'rgba(16, 185, 129, 0.7)',
                        'rgba(245, 158, 11, 0.7)',
                        'rgba(239, 68, 68, 0.7)',
                        'rgba(139, 92, 246, 0.7)',
                        'rgba(14, 165, 233, 0.7)',
                        'rgba(20, 184, 166, 0.7)',
                        'rgba(244, 63, 94, 0.7)',
                        'rgba(120, 113, 108, 0.7)',
                        'rgba(168, 85, 247, 0.7)'
                    ]
                }]
            },
            options: {
                indexAxis: 'y',
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    x: {
                        beginAtZero: true,
                        ticks: {
                            precision: 0
                        }
                    }
                }
            }
        });
    }

    // ============================================
    // DATE FILTERS
    // ============================================
    
    function initializeDateFilters() {
        // Date range picker for custom date ranges
        $('#dateRangePicker').daterangepicker({
            opens: 'left',
            startDate: moment().subtract(30, 'days'),
            endDate: moment(),
            ranges: {
                'Today': [moment(), moment()],
                'Yesterday': [moment().subtract(1, 'days'), moment().subtract(1, 'days')],
                'Last 7 Days': [moment().subtract(6, 'days'), moment()],
                'Last 30 Days': [moment().subtract(29, 'days'), moment()],
                'This Month': [moment().startOf('month'), moment().endOf('month')],
                'Last Month': [moment().subtract(1, 'month').startOf('month'), moment().subtract(1, 'month').endOf('month')]
            },
            locale: {
                format: 'YYYY-MM-DD',
                separator: ' to ',
                applyLabel: 'Apply',
                cancelLabel: 'Cancel',
                fromLabel: 'From',
                toLabel: 'To',
                customRangeLabel: 'Custom',
                weekLabel: 'W',
                daysOfWeek: ['Su', 'Mo', 'Tu', 'We', 'Th', 'Fr', 'Sa'],
                monthNames: [
                    'January', 'February', 'March', 'April', 'May', 'June',
                    'July', 'August', 'September', 'October', 'November', 'December'
                ],
                firstDay: 1
            }
        }, function(start, end, label) {
            console.log('Date range selected:', start.format('YYYY-MM-DD'), 'to', end.format('YYYY-MM-DD'));
            
            // Update URL with new date range
            updateAnalyticsWithDateRange(start, end);
        });

        // Quick date buttons
        $('.quick-date-btn').click(function(e) {
            e.preventDefault();
            const days = $(this).data('days');
            const startDate = moment().subtract(days, 'days');
            const endDate = moment();
            
            updateAnalyticsWithDateRange(startDate, endDate);
        });
    }

    function updateAnalyticsWithDateRange(startDate, endDate) {
        // Show loading state
        showLoading();
        
        // Update URL and reload
        const urlParams = new URLSearchParams(window.location.search);
        urlParams.set('start_date', startDate.format('YYYY-MM-DD'));
        urlParams.set('end_date', endDate.format('YYYY-MM-DD'));
        
        // Redirect to new URL with filters
        window.location.href = '{{ route("price-histories.analytics") }}?' + urlParams.toString();
    }

    function showLoading() {
        // Add loading overlay to page
        $('body').append(`
            <div class="loading-overlay" style="
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: rgba(255, 255, 255, 0.8);
                display: flex;
                align-items: center;
                justify-content: center;
                z-index: 9999;
            ">
                <div class="text-center">
                    <div class="spinner-border text-primary" style="width: 3rem; height: 3rem;" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    <p class="mt-3">Loading analytics data...</p>
                </div>
            </div>
        `);
    }

    // ============================================
    // AUTO REFRESH
    // ============================================
    
    function initializeAutoRefresh() {
        let autoRefreshEnabled = true;
        let refreshInterval = null;
        
        // Start auto-refresh (every 60 seconds)
        startAutoRefresh();
        
        // Toggle auto-refresh
        $('#toggleAutoRefresh').click(function() {
            autoRefreshEnabled = !autoRefreshEnabled;
            
            if (autoRefreshEnabled) {
                startAutoRefresh();
                $(this).html('<i class="bx bx-refresh"></i> Auto Refresh: ON');
                $(this).removeClass('btn-light').addClass('btn-primary');
            } else {
                stopAutoRefresh();
                $(this).html('<i class="bx bx-refresh"></i> Auto Refresh: OFF');
                $(this).removeClass('btn-primary').addClass('btn-light');
            }
        });
        
        function startAutoRefresh() {
            if (refreshInterval) clearInterval(refreshInterval);
            
            refreshInterval = setInterval(function() {
                refreshAnalyticsData();
            }, 60000); // 60 seconds
        }
        
        function stopAutoRefresh() {
            if (refreshInterval) {
                clearInterval(refreshInterval);
                refreshInterval = null;
            }
        }
        
        // Manual refresh button
        $('#refreshData').click(function() {
            refreshAnalyticsData();
        });
    }

    function refreshAnalyticsData() {
        console.log('Refreshing analytics data...');
        
        // Show refreshing indicator
        $('#refreshData').addClass('disabled');
        $('#refreshData').html('<i class="bx bx-loader-circle bx-spin"></i> Refreshing...');
        
        $.ajax({
            url: '{{ route("price-histories.analytics") }}',
            type: 'GET',
            data: window.location.search,
            success: function(response) {
                // Parse the HTML response
                const $response = $(response);
                
                // Update statistics cards
                updateStatisticsCards($response);
                
                // Update recent activity
                updateRecentActivity($response);
                
                // Update most active products table
                updateMostActiveProducts($response);
                
                // Update charts
                updateChartsFromResponse($response);
                
                console.log('Analytics data refreshed successfully');
            },
            error: function(xhr) {
                console.error('Failed to refresh analytics data:', xhr);
                showToast('Failed to refresh data. Please try again.', 'error');
            },
            complete: function() {
                // Restore refresh button
                $('#refreshData').removeClass('disabled');
                $('#refreshData').html('<i class="bx bx-refresh"></i> Refresh Now');
            }
        });
    }

    function updateStatisticsCards($response) {
        // Update each statistics card
        $('.analytics-value').each(function(index) {
            const newValue = $response.find('.analytics-value').eq(index).text();
            if ($(this).text() !== newValue) {
                // Animate value change
                animateValueChange($(this), $(this).text(), newValue);
            }
        });
        
        // Update trend indicators
        $('.trend-up, .trend-down').each(function(index) {
            const newTrend = $response.find('.trend-up, .trend-down').eq(index).text();
            $(this).text(newTrend);
        });
    }

    function updateRecentActivity($response) {
        const newActivity = $response.find('.activity-timeline').html();
        $('.activity-timeline').html(newActivity);
    }

    function updateMostActiveProducts($response) {
        const newTableBody = $response.find('table tbody').html();
        $('table tbody').html(newTableBody);
    }

    function updateChartsFromResponse($response) {
        // This would require the controller to return JSON data for charts
        // For now, we'll just note that charts need to be updated
        console.log('Charts data would be updated here');
    }

    function animateValueChange($element, oldValue, newValue) {
        // Simple animation for value changes
        $element.css('color', '#ef4444');
        setTimeout(() => {
            $element.text(newValue);
            $element.css('color', '#10b981');
            setTimeout(() => {
                $element.css('color', '');
            }, 1000);
        }, 500);
    }

    // ============================================
    // EXPORT FUNCTIONALITY
    // ============================================
    
    function initializeExportOptions() {
        // CSV Export
        $('.btn-export-csv').click(function(e) {
            e.preventDefault();
            
            Swal.fire({
                title: 'Export Analytics Data',
                text: 'Would you like to export the analytics data as CSV?',
                icon: 'question',
                showCancelButton: true,
                confirmButtonText: 'Export CSV',
                cancelButtonText: 'Cancel',
                showLoaderOnConfirm: true,
                preConfirm: () => {
                    return new Promise((resolve) => {
                        // Add current date range to export URL
                        const urlParams = new URLSearchParams(window.location.search);
                        const exportUrl = '{{ route("price-histories.export") }}?' + urlParams.toString();
                        
                        // Trigger download
                        const downloadLink = document.createElement('a');
                        downloadLink.href = exportUrl;
                        downloadLink.download = `price-analytics-${moment().format('YYYY-MM-DD')}.csv`;
                        document.body.appendChild(downloadLink);
                        downloadLink.click();
                        document.body.removeChild(downloadLink);
                        
                        resolve();
                    });
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    showToast('Export started. Your download should begin shortly.', 'success');
                }
            });
        });

        // PDF Export
        $('.btn-export-pdf').click(function(e) {
            e.preventDefault();
            
            Swal.fire({
                title: 'Export as PDF',
                text: 'This feature is coming soon!',
                icon: 'info',
                confirmButtonText: 'OK'
            });
        });

        // Chart Image Export
        $('.btn-export-chart').click(function(e) {
            e.preventDefault();
            
            if (!priceTrendChart) {
                showToast('No chart available to export', 'warning');
                return;
            }
            
            // Get chart as base64 image
            const chartImage = priceTrendChart.toBase64Image();
            
            // Create download link
            const downloadLink = document.createElement('a');
            downloadLink.href = chartImage;
            downloadLink.download = `price-trend-chart-${moment().format('YYYY-MM-DD')}.png`;
            document.body.appendChild(downloadLink);
            downloadLink.click();
            document.body.removeChild(downloadLink);
            
            showToast('Chart exported successfully', 'success');
        });
    }

    // ============================================
    // QUICK ACTIONS
    // ============================================
    
    function initializeQuickActions() {
        // Quick filter by price type
        $('.filter-price-type').click(function(e) {
            e.preventDefault();
            const priceType = $(this).data('price-type');
            
            const urlParams = new URLSearchParams(window.location.search);
            urlParams.set('price_type', priceType);
            
            window.location.href = '{{ route("price-histories.analytics") }}?' + urlParams.toString();
        });

        // Quick filter by change reason
        $('.filter-reason').click(function(e) {
            e.preventDefault();
            const reason = $(this).data('reason');
            
            const urlParams = new URLSearchParams(window.location.search);
            urlParams.set('change_reason', reason);
            
            window.location.href = '{{ route("price-histories.analytics") }}?' + urlParams.toString();
        });

        // Reset filters
        $('#resetFilters').click(function(e) {
            e.preventDefault();
            
            Swal.fire({
                title: 'Reset Filters',
                text: 'Are you sure you want to reset all filters?',
                icon: 'question',
                showCancelButton: true,
                confirmButtonText: 'Reset',
                cancelButtonText: 'Cancel'
            }).then((result) => {
                if (result.isConfirmed) {
                    window.location.href = '{{ route("price-histories.analytics") }}';
                }
            });
        });
    }

    // ============================================
    // UTILITY FUNCTIONS
    // ============================================
    
    function showToast(message, type = 'info') {
        const toast = $(`
            <div class="toast align-items-center text-bg-${type} border-0" role="alert" aria-live="assertive" aria-atomic="true">
                <div class="d-flex">
                    <div class="toast-body">
                        ${message}
                    </div>
                    <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
                </div>
            </div>
        `);
        
        $('.toast-container').append(toast);
        const bsToast = new bootstrap.Toast(toast[0]);
        bsToast.show();
        
        // Remove toast after it hides
        toast.on('hidden.bs.toast', function () {
            $(this).remove();
        });
    }

    // ============================================
    // KEYBOARD SHORTCUTS
    // ============================================
    
    $(document).keydown(function(e) {
        // Ctrl/Cmd + R to refresh
        if ((e.ctrlKey || e.metaKey) && e.key === 'r') {
            e.preventDefault();
            refreshAnalyticsData();
        }
        
        // Ctrl/Cmd + E to export
        if ((e.ctrlKey || e.metaKey) && e.key === 'e') {
            e.preventDefault();
            $('.btn-export-csv').click();
        }
        
        // Ctrl/Cmd + F to focus search
        if ((e.ctrlKey || e.metaKey) && e.key === 'f') {
            e.preventDefault();
            $('#searchInput').focus().select();
        }
        
        // Escape to close modals
        if (e.key === 'Escape') {
            $('.modal').modal('hide');
        }
    });

    // Initialize search
    $('#searchInput').on('keyup', function() {
        const searchTerm = $(this).val().toLowerCase();
        
        $('table tbody tr').each(function() {
            const productName = $(this).find('td:first-child').text().toLowerCase();
            const sku = $(this).find('td:nth-child(2)').text().toLowerCase();
            
            if (productName.includes(searchTerm) || sku.includes(searchTerm)) {
                $(this).show();
            } else {
                $(this).hide();
            }
        });
    });

    console.log('Price analytics JavaScript loaded successfully');
</script>

<!-- Add this HTML for additional UI elements -->
<style>
    .toast-container {
        position: fixed;
        top: 20px;
        right: 20px;
        z-index: 9999;
    }
    
    .btn-refresh {
        position: relative;
    }
    
    .btn-refresh .badge {
        position: absolute;
        top: -5px;
        right: -5px;
        font-size: 10px;
    }
    
    .chart-placeholder {
        height: 300px;
        display: flex;
        align-items: center;
        justify-content: center;
        background: #f8f9fa;
        border-radius: 8px;
        color: #6c757d;
    }
</style>

<!-- Add these elements to your Blade template for additional features -->
<!--
<div class="toast-container"></div>

<div class="card mt-4">
    <div class="card-body">
        <h5 class="card-title mb-4">Price Changes Over Time</h5>
        <canvas id="priceChangesChart" class="chart-container"></canvas>
    </div>
</div>

<div class="row mt-4">
    <div class="col-md-6">
        <div class="card">
            <div class="card-body">
                <h5 class="card-title mb-4">Activity by Hour</h5>
                <canvas id="activityByHourChart" class="chart-container"></canvas>
            </div>
        </div>
    </div>
    <div class="col-md-6">
        <div class="card">
            <div class="card-body">
                <h5 class="card-title mb-4">Top Products by Activity</h5>
                <canvas id="productActivityChart" class="chart-container"></canvas>
            </div>
        </div>
    </div>
</div>

<div class="mt-4 d-flex gap-2">
    <button id="toggleAutoRefresh" class="btn btn-primary">
        <i class="bx bx-refresh"></i> Auto Refresh: ON
    </button>
    <button id="refreshData" class="btn btn-outline-primary">
        <i class="bx bx-refresh"></i> Refresh Now
    </button>
    <button id="resetFilters" class="btn btn-outline-secondary">
        <i class="bx bx-reset"></i> Reset Filters
    </button>
</div>
-->

@endsection