<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Receipt - {{ $data['invoice_no'] ?? 'POS Receipt' }}</title>
    <style>
        /* Thermal printer friendly styling */
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Courier New', monospace;
            font-size: 12px;
            line-height: 1.2;
            max-width: 320px;
            margin: 0 auto;
            padding: 5px;
            color: #000;
        }
        
        /* Thermal printer specific */
        @media print {
            @page {
                margin: 0;
                size: 80mm auto;
            }
            
            body {
                width: 80mm;
                padding: 0;
            }
            
            .no-print {
                display: none !important;
            }
            
            .page-break {
                page-break-before: always;
            }
        }
        
        /* Common styles */
        .text-center { text-align: center; }
        .text-right { text-align: right; }
        .text-left { text-align: left; }
        .bold { font-weight: bold; }
        .underline { text-decoration: underline; }
        
        /* Layout */
        .receipt {
            width: 100%;
        }
        
        .header {
            text-align: center;
            padding-bottom: 5px;
            border-bottom: 1px dashed #000;
            margin-bottom: 5px;
        }
        
        .store-name {
            font-size: 14px;
            font-weight: bold;
            text-transform: uppercase;
            margin: 3px 0;
        }
        
        .store-details {
            font-size: 10px;
            margin: 2px 0;
        }
        
        .section {
            margin: 5px 0;
            padding: 3px 0;
        }
        
        .section-title {
            font-weight: bold;
            border-bottom: 1px solid #000;
            margin-bottom: 3px;
            padding-bottom: 2px;
        }
        
        .item-row {
            display: flex;
            justify-content: space-between;
            margin: 1px 0;
            padding: 1px 0;
        }
        
        .item-name {
            flex: 3;
            text-align: left;
        }
        
        .item-qty {
            flex: 1;
            text-align: center;
        }
        
        .item-price {
            flex: 2;
            text-align: right;
        }
        
        .item-total {
            flex: 2;
            text-align: right;
            font-weight: bold;
        }
        
        .total-row {
            display: flex;
            justify-content: space-between;
            margin: 2px 0;
            padding: 2px 0;
        }
        
        .total-label {
            flex: 2;
        }
        
        .total-value {
            flex: 1;
            text-align: right;
            font-weight: bold;
        }
        
        .grand-total {
            border-top: 2px solid #000;
            border-bottom: 2px solid #000;
            margin: 5px 0;
            padding: 5px 0;
            font-size: 14px;
        }
        
        .payment-method {
            text-align: center;
            margin: 5px 0;
            padding: 3px;
            border: 1px dashed #000;
        }
        
        .footer {
            text-align: center;
            font-size: 10px;
            margin-top: 10px;
            padding-top: 5px;
            border-top: 1px dashed #000;
        }
        
        .vat-breakdown {
            font-size: 10px;
            margin: 3px 0;
            padding: 5px;
            border: 1px dashed #999;
        }
        
        .vat-line {
            display: flex;
            justify-content: space-between;
            margin: 2px 0;
        }
        
        /* Responsive for screen */
        @media screen {
            body {
                background: #f5f5f5;
                padding: 20px;
            }
            
            .receipt {
                background: white;
                padding: 20px;
                box-shadow: 0 0 10px rgba(0,0,0,0.1);
                border-radius: 5px;
            }
        }
        
        /* Barcode style */
        .barcode {
            text-align: center;
            margin: 10px 0;
            font-family: 'Libre Barcode 128', cursive;
            font-size: 36px;
        }
        
        /* QR Code container - CENTERED */
        .qrcode {
            text-align: center;
            margin: 10px auto;
            width: 100px;
            height: 100px;
            display: flex;
            justify-content: center;
            align-items: center;
        }
        
        .qrcode canvas {
            display: block;
            margin: 0 auto;
        }
        
        .qrcode img {
            display: block;
            margin: 0 auto;
        }
        
        /* Customer info warning */
        .customer-warning {
            font-style: italic;
            color: #666;
        }
        
        /* VAT Status Colors */
        .vat-exempted { color: #dc3545; }
        .vat-zero-rated { color: #fd7e14; }
        .vat-vatable { color: #28a745; }
        
        /* VAT summary from original receipt */
        .vat-summary {
            margin: 8px 0;
            font-size: 11px;
        }
        
        .info-row {
            display: flex;
            justify-content: space-between;
            margin: 3px 0;
        }
    </style>
    
    <!-- Barcode font -->
    <link href="https://fonts.googleapis.com/css2?family=Libre+Barcode+128&display=swap" rel="stylesheet">
    
    <!-- QR Code library -->
    <script src="https://cdn.rawgit.com/davidshimjs/qrcodejs/gh-pages/qrcode.min.js"></script>
</head>
<body>
    @php
        // Safe data extraction with defaults
        $store = $data['store'] ?? [
            'name' => config('app.name', 'Kenyan Supermarket'),
            'address' => config('app.address', 'Nairobi CBD'),
            'phone' => config('app.phone', '0700 000 000'),
            'pin' => config('app.pin', 'P051234567N'),
            'vat_number' => config('app.vat_number', 'VAT001234567'),
        ];
        
        $customer = $data['customer'] ?? [];
        $items = $data['items'] ?? [];
        $totals = $data['totals'] ?? [];
        $payment = $data['payment'] ?? [];
        $vatInfo = $data['vat_info'] ?? [];
        
        // Customer VAT status handling
        $customerName = $customer['name'] ?? 'Walk-in Customer';
        $customerStatus = $customer['vat_status'] ?? 'vatable';
        $isExempted = $customerStatus === 'exempted';
        $isZeroRated = $customerStatus === 'zero_rated';
        $isVatable = !$isExempted && !$isZeroRated;
        
        // Safe totals
        $subtotal = $totals['subtotal'] ?? 0;
        $discount = $totals['discount'] ?? 0;
        $vatAmount = $totals['vat_amount'] ?? $totals['tax_total'] ?? 0;
        $grandTotal = $totals['grand_total'] ?? 0;
        
        // VAT breakdown and summary amounts - IMPORTANT: Using original structure
        $vatBreakdown = $vatInfo['breakdown'] ?? [];
        if (!is_array($vatBreakdown)) {
            $vatBreakdown = [];
        }
        
        // Original VAT summary fields from first receipt
        $taxableAmount = $totals['taxable_amount'] ?? 0;
        $zeroRatedAmount = $totals['zero_rated_amount'] ?? 0;
        $exemptedAmount = $totals['exempted_amount'] ?? 0;
        $nonTaxableAmount = $totals['non_taxable_amount'] ?? 0;
        
        // Payment data
        $paymentMethod = $payment['method'] ?? 'cash';
        $amountPaid = $payment['amount_paid'] ?? $grandTotal;
        $change = $payment['change'] ?? 0;
        $transactionId = $payment['transaction_id'] ?? null;
        $mpesaDetails = $payment['mpesa_details'] ?? null;
        
        // Notes
        $notes = $data['notes'] ?? [
            'Thank you for your business!',
            'Items sold are not returnable unless defective',
            'Please check goods before leaving the counter',
            'Valid for 7 days from date of purchase',
            'Keep this receipt for warranty claims',
            'VAT Invoice - Valid for Tax Purposes'
        ];
    @endphp
    
    <div class="receipt">
        <!-- Store Header -->
        <div class="header">
            <div class="store-name">{{ strtoupper($store['name']) }}</div>
            <div class="store-details">{{ $store['address'] ?? '' }}</div>
            <div class="store-details">Tel: {{ $store['phone'] ?? '' }}</div>
            @if(!empty($store['pin']))
            <div class="store-details">PIN: {{ $store['pin'] }}</div>
            @endif
            @if(!empty($store['vat_number']))
            <div class="store-details">VAT: {{ $store['vat_number'] }}</div>
            @endif
        </div>
        
        <!-- Transaction Info -->
        <div class="section">
            <div class="item-row">
                <span>Receipt No:</span>
                <span class="bold">{{ $data['invoice_no'] ?? $data['receipt_no'] ?? 'N/A' }}</span>
            </div>
            <div class="item-row">
                <span>Date:</span>
                <span>{{ $data['date'] ?? date('Y-m-d') }}</span>
            </div>
            <div class="item-row">
                <span>Time:</span>
                <span>{{ $data['time'] ?? date('H:i:s') }}</span>
            </div>
            <div class="item-row">
                <span>Cashier:</span>
                <span>{{ $data['cashier'] ?? 'Cashier' }}</span>
            </div>
            
            <!-- Customer Information -->
            <div class="item-row">
                <span>Customer:</span>
                <span>{{ $customerName }}</span>
            </div>
            
            @if(!empty($customer['phone']))
            <div class="item-row">
                <span>Phone:</span>
                <span>{{ $customer['phone'] }}</span>
            </div>
            @endif
            
            @if(!empty($customer['vat_number']))
            <div class="item-row">
                <span>VAT No:</span>
                <span>{{ $customer['vat_number'] }}</span>
            </div>
            @endif
            
            <!-- Customer VAT Status -->
            <div class="item-row">
                <span>VAT Status:</span>
                <span class="bold {{ $isExempted ? 'vat-exempted' : ($isZeroRated ? 'vat-zero-rated' : 'vat-vatable') }}">
                    {{ $isExempted ? 'EXEMPTED' : ($isZeroRated ? 'ZERO-RATED' : 'VATABLE') }}
                </span>
            </div>
            
            @if($customerName === 'Walk-in Customer')
            <div class="item-row customer-warning">
                <span>Customer:</span>
                <span>Walk-in Customer</span>
            </div>
            @endif
        </div>
        
        <!-- Barcode -->
        <div class="barcode" id="barcode"></div>
        
        <!-- Items Section -->
        <div class="section">
            <div class="section-title">ITEMS</div>
            
            <!-- Items Header -->
            <div class="item-row bold">
                <div class="item-name">ITEM</div>
                <div class="item-qty">QTY</div>
                <div class="item-price">PRICE</div>
                <div class="item-total">TOTAL</div>
            </div>
            
            <!-- Items List -->
            @if(count($items) > 0)
                @foreach($items as $item)
                @php
                    $quantity = $item['quantity'] ?? 1;
                    $unitPrice = $item['price'] ?? $item['unit_price'] ?? 0;
                    $itemTotal = $unitPrice * $quantity;
                    $taxRate = $item['tax_rate'] ?? 0;
                    $isItemVatable = $item['is_vatable'] ?? ($taxRate > 0);
                @endphp
                <div class="item-row">
                    <div class="item-name">{{ $item['name'] ?? 'Product' }}</div>
                    <div class="item-qty">{{ $quantity }}</div>
                    <div class="item-price">{{ number_format($unitPrice, 2) }}</div>
                    <div class="item-total">{{ number_format($itemTotal, 2) }}</div>
                </div>
                
                @if($isItemVatable && $taxRate > 0 && ($item['tax_amount'] ?? 0) > 0)
                <div class="vat-line">
                    <span>VAT {{ $taxRate }}%</span>
                    <span>KES {{ number_format($item['tax_amount'] ?? 0, 2) }}</span>
                </div>
                @endif
                @endforeach
            @else
                <div class="item-row">
                    <div class="item-name">No items</div>
                    <div class="item-total">0.00</div>
                </div>
            @endif
        </div>
        
        <!-- Totals Section -->
        <div class="section">
            <div class="section-title">TOTALS</div>
            
            <div class="total-row">
                <span class="total-label">Subtotal:</span>
                <span class="total-value">KES {{ number_format($subtotal, 2) }}</span>
            </div>
            
            @if($discount > 0)
            <div class="total-row">
                <span class="total-label">Discount:</span>
                <span class="total-value">-KES {{ number_format($discount, 2) }}</span>
            </div>
            @endif
            
            <!-- VAT Breakdown Section (from original receipt) -->
            @if(!$isExempted && !empty($vatBreakdown))
                <div class="vat-breakdown">
                    <div class="item-row bold" style="border-bottom: 1px dashed #ccc; padding-bottom: 3px;">
                        <span>VAT BREAKDOWN</span>
                        <span>Amount</span>
                    </div>
                    
                    @foreach($vatBreakdown as $details)
                        @php
                            // Safely get values with defaults
                            $rate = $details['rate'] ?? 0;
                            $taxableAmt = $details['taxable_amount'] ?? $details['taxableAmount'] ?? 0;
                            $vatAmt = $details['vat_amount'] ?? $details['amount'] ?? 0;
                            $type = $details['type'] ?? 'standard';
                        @endphp
                        
                        @if($taxableAmt > 0 && $rate >= 0)
                            <div class="vat-line">
                                <span>
                                    @if($type === 'zero_rated')
                                        Zero-Rated (0%)
                                    @else
                                        VAT {{ $rate }}%
                                    @endif
                                    on KES {{ number_format($taxableAmt, 2) }}:
                                </span>
                                <span>KES {{ number_format($vatAmt, 2) }}</span>
                            </div>
                        @endif
                    @endforeach
                    
                    @if($vatAmount > 0)
                        <div class="vat-line bold" style="border-top: 1px dashed #ccc; padding-top: 3px; margin-top: 3px;">
                            <span>Total VAT:</span>
                            <span>KES {{ number_format($vatAmount, 2) }}</span>
                        </div>
                    @endif
                </div>
            @endif
            
            <!-- ORIGINAL VAT SUMMARY SECTION -->
            @if($taxableAmount > 0 || $zeroRatedAmount > 0 || $exemptedAmount > 0 || $nonTaxableAmount > 0)
                <div class="vat-summary" style="margin: 8px 0; padding: 8px; border: 1px dashed #999; font-size: 11px;">
                    @if($taxableAmount > 0)
                        <div class="info-row">
                            <span>Taxable Amount:</span>
                            <span>KES {{ number_format($taxableAmount, 2) }}</span>
                        </div>
                    @endif
                    @if($zeroRatedAmount > 0)
                        <div class="info-row">
                            <span>Zero-Rated Amount:</span>
                            <span>KES {{ number_format($zeroRatedAmount, 2) }}</span>
                        </div>
                    @endif
                    @if($exemptedAmount > 0)
                        <div class="info-row">
                            <span>Exempted Amount:</span>
                            <span>KES {{ number_format($exemptedAmount, 2) }}</span>
                        </div>
                    @endif
                    @if($nonTaxableAmount > 0)
                        <div class="info-row">
                            <span>Non-Taxable Amount:</span>
                            <span>KES {{ number_format($nonTaxableAmount, 2) }}</span>
                        </div>
                    @endif
                </div>
            @endif
            
            <!-- VAT Amount -->
            @if($vatAmount > 0 && !$isExempted)
            <div class="total-row">
                <span class="total-label">VAT Amount:</span>
                <span class="total-value">KES {{ number_format($vatAmount, 2) }}</span>
            </div>
            @endif
            
            <!-- Grand Total -->
            <div class="total-row grand-total">
                <span class="total-label">TOTAL:</span>
                <span class="total-value">KES {{ number_format($grandTotal, 2) }}</span>
            </div>
        </div>
        
        <!-- Payment Details -->
        @if(isset($data['payment']))
        <div class="section">
            <div class="section-title">PAYMENT</div>
            
            <div class="payment-method">
                <div class="bold">{{ strtoupper($paymentMethod) }}</div>
                
                <div>Amount Paid: KES {{ number_format($amountPaid, 2) }}</div>
                
                @if($paymentMethod === 'cash' && isset($payment['cash_received']))
                <div>Cash Received: KES {{ number_format($payment['cash_received'], 2) }}</div>
                @endif
                
                @if($change > 0)
                <div class="bold">Change: KES {{ number_format($change, 2) }}</div>
                @endif
                
                @if($transactionId)
                <div>Transaction ID: {{ $transactionId }}</div>
                @endif
                
                @if($mpesaDetails)
                <div>Phone: {{ $mpesaDetails['phone'] ?? '' }}</div>
                @if(!empty($mpesaDetails['receipt_no']))
                <div>M-Pesa Receipt: {{ $mpesaDetails['receipt_no'] }}</div>
                @endif
                @endif
            </div>
        </div>
        @endif
        
        <!-- VAT Status Information -->
        @if($isExempted || $isZeroRated)
        <div class="section">
            <div class="section-title">VAT INFORMATION</div>
            <div class="text-center">
                @if($isExempted)
                <div class="bold vat-exempted">VAT EXEMPTED CUSTOMER</div>
                <div>No VAT Applied</div>
                @elseif($isZeroRated)
                <div class="bold vat-zero-rated">ZERO-RATED CUSTOMER</div>
                <div>0% VAT Rate Applied</div>
                @endif
            </div>
        </div>
        @endif
        
        <!-- QR Code for verification - CENTERED -->
        <div class="qrcode" id="qrcode"></div>
        
        <!-- Footer -->
        <div class="footer">
            @foreach($notes as $note)
            <div>{{ $note }}</div>
            @endforeach
            
            <div style="margin-top: 10px;">
                <div>Printed: {{ date('Y-m-d H:i:s') }}</div>
                <div>** COMPUTER GENERATED RECEIPT **</div>
                <div>Valid without signature</div>
                <div style="margin-top: 5px;">
                    <small>Powered by Eflix Digital</small>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Print Controls (only visible on screen) -->
    <div class="no-print" style="text-align: center; margin-top: 20px;">
        <button onclick="window.print()" style="padding: 10px 20px; margin: 5px; background: #007bff; color: white; border: none; border-radius: 5px; cursor: pointer;">
            🖨️ Print Receipt
        </button>
        <button onclick="window.close()" style="padding: 10px 20px; margin: 5px; background: #6c757d; color: white; border: none; border-radius: 5px; cursor: pointer;">
            ❌ Close
        </button>
        <button onclick="location.reload()" style="padding: 10px 20px; margin: 5px; background: #28a745; color: white; border: none; border-radius: 5px; cursor: pointer;">
            🔄 Reload
        </button>
        
        @if($data['invoice_no'] ?? false)
        <div style="margin-top: 10px;">
            <small>Receipt No: {{ $data['invoice_no'] }}</small>
        </div>
        @endif
    </div>
    
    <script>
        // Auto-print Script
        @if($autoprint ?? false)
        document.addEventListener('DOMContentLoaded', function() {
            setTimeout(function() {
                window.print();
                
                // Optionally close after printing
                setTimeout(function() {
                    if(!window.sessionStorage.getItem('receipt_keep_open')) {
                        window.close();
                    }
                }, 1000);
            }, 500);
        });
        @endif
        
        // Generate QR code
        document.addEventListener('DOMContentLoaded', function() {
            const invoiceNo = '{{ $data["invoice_no"] ?? $data["receipt_no"] ?? "" }}';
            
            // Generate barcode
            const barcodeContainer = document.getElementById('barcode');
            if (barcodeContainer && invoiceNo) {
                barcodeContainer.textContent = `*${invoiceNo}*`;
            }
            
            // Generate QR code - CENTERED
            if (invoiceNo && typeof QRCode !== 'undefined') {
                const qrContainer = document.getElementById('qrcode');
                if (qrContainer) {
                    // Clear any existing QR code
                    qrContainer.innerHTML = '';
                    
                    // Create new QR code centered
                    const qrCode = new QRCode(qrContainer, {
                        text: invoiceNo,
                        width: 100,
                        height: 100,
                        colorDark : "#000000",
                        colorLight : "#ffffff",
                        correctLevel : QRCode.CorrectLevel.H
                    });
                    
                    // Ensure it's centered
                    const qrImg = qrContainer.querySelector('img');
                    if (qrImg) {
                        qrImg.style.display = 'block';
                        qrImg.style.margin = '0 auto';
                    }
                    
                    const qrCanvas = qrContainer.querySelector('canvas');
                    if (qrCanvas) {
                        qrCanvas.style.display = 'block';
                        qrCanvas.style.margin = '0 auto';
                    }
                }
            }
        });
        
        // Keyboard shortcuts
        document.addEventListener('keydown', function(e) {
            // Ctrl+P or Cmd+P for print
            if ((e.ctrlKey || e.metaKey) && e.key === 'p') {
                e.preventDefault();
                window.print();
            }
            // Escape to close
            if (e.key === 'Escape') {
                window.close();
            }
            // Ctrl+S to keep window open
            if ((e.ctrlKey || e.metaKey) && e.key === 's') {
                e.preventDefault();
                window.sessionStorage.setItem('receipt_keep_open', 'true');
                alert('Window will stay open after printing');
            }
        });
        
        // Handle print events
        window.addEventListener('afterprint', function() {
            console.log('Receipt printed');
            // Send analytics if needed
            fetch('/api/receipt/printed', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    invoice_no: '{{ $data["invoice_no"] ?? "" }}',
                    printed_at: new Date().toISOString()
                })
            }).catch(() => {});
        });
    </script>
</body>
</html>