<div class="modal fade" id="heldSalesModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered modal-xl">
        <div class="modal-content">
            <!-- Modal Header -->
            <div class="modal-header bg-info text-white">
                <h5 class="modal-title">
                    <i class="bx bx-history me-2"></i>Held & Saved Sales
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            
            <!-- Modal Body -->
            <div class="modal-body">
                <!-- Quick Stats -->
                <div class="held-stats mb-4">
                    <div class="row g-2">
                        <div class="col-3">
                            <div class="card border-0 bg-light">
                                <div class="card-body p-2 text-center">
                                    <div class="small text-muted">Active Holds</div>
                                    <div class="fw-bold" id="activeHolds">0</div>
                                </div>
                            </div>
                        </div>
                        <div class="col-3">
                            <div class="card border-0 bg-light">
                                <div class="card-body p-2 text-center">
                                    <div class="small text-muted">Total Value</div>
                                    <div class="fw-bold" id="totalHeldValue">KES 0</div>
                                </div>
                            </div>
                        </div>
                        <div class="col-3">
                            <div class="card border-0 bg-light">
                                <div class="card-body p-2 text-center">
                                    <div class="small text-muted">Expiring Soon</div>
                                    <div class="fw-bold" id="expiringSoon">0</div>
                                </div>
                            </div>
                        </div>
                        <div class="col-3">
                            <div class="card border-0 bg-light">
                                <div class="card-body p-2 text-center">
                                    <div class="small text-muted">Today's Holds</div>
                                    <div class="fw-bold" id="todaysHolds">0</div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Search and Filter -->
                <div class="held-search mb-4">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <div class="input-group">
                                <span class="input-group-text bg-light">
                                    <i class="bx bx-search"></i>
                                </span>
                                <input type="text" class="form-control" id="heldSearch" 
                                       placeholder="Search by customer, reference, or items...">
                            </div>
                        </div>
                        <div class="col-md-3">
                            <select class="form-select" id="heldStatusFilter">
                                <option value="all">All Status</option>
                                <option value="active">Active</option>
                                <option value="expired">Expired</option>
                                <option value="converted">Converted to Sale</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <select class="form-select" id="heldDateFilter">
                                <option value="all">All Dates</option>
                                <option value="today">Today</option>
                                <option value="yesterday">Yesterday</option>
                                <option value="week">This Week</option>
                            </select>
                        </div>
                    </div>
                </div>
                
                <!-- Held Sales Table -->
                <div class="held-table">
                    <div class="table-responsive" style="max-height: 400px; overflow-y: auto;">
                        <table class="table table-hover">
                            <thead style="position: sticky; top: 0; background: white; z-index: 1;">
                                <tr>
                                    <th>Reference</th>
                                    <th>Customer</th>
                                    <th>Items</th>
                                    <th>Total</th>
                                    <th>Status</th>
                                    <th>Held By</th>
                                    <th>Held At</th>
                                    <th>Expires</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody id="heldTableBody">
                                <!-- Held sales will be loaded here -->
                                <tr>
                                    <td colspan="9" class="text-center py-5">
                                        <div class="spinner-border text-primary" role="status">
                                            <span class="visually-hidden">Loading held sales...</span>
                                        </div>
                                        <p class="mt-2 text-muted">Loading held sales...</p>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
                
                <!-- Bulk Actions -->
                <div class="bulk-actions mt-4" id="bulkActions" style="display: none;">
                    <div class="card border-primary">
                        <div class="card-body">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <span id="selectedCount">0</span> sales selected
                                    <span class="ms-3 text-muted">Total: <span id="selectedTotal">KES 0.00</span></span>
                                </div>
                                <div class="btn-group">
                                    <button class="btn btn-outline-primary btn-sm" onclick="loadSelectedSales()">
                                        <i class="bx bx-check-circle me-1"></i>Load Selected
                                    </button>
                                    <button class="btn btn-outline-danger btn-sm" onclick="deleteSelectedSales()">
                                        <i class="bx bx-trash me-1"></i>Delete Selected
                                    </button>
                                    <button class="btn btn-outline-secondary btn-sm" onclick="clearSelection()">
                                        <i class="bx bx-x me-1"></i>Clear
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Modal Footer -->
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                <button type="button" class="btn btn-primary" onclick="exportHeldSales()">
                    <i class="bx bx-download me-2"></i>Export
                </button>
                <button type="button" class="btn btn-danger" onclick="clearExpiredHolds()">
                    <i class="bx bx-trash me-2"></i>Clear Expired
                </button>
            </div>
        </div>
    </div>
</div>

<style>
    .held-stats .card {
        transition: transform 0.2s;
    }
    
    .held-stats .card:hover {
        transform: translateY(-2px);
    }
    
    .held-table table {
        border-collapse: separate;
        border-spacing: 0;
    }
    
    .held-table tbody tr {
        cursor: pointer;
        transition: background-color 0.2s;
    }
    
    .held-table tbody tr:hover {
        background-color: #f8fafc;
    }
    
    .held-table tbody tr.selected {
        background-color: #e0f2fe;
    }
    
    .reference-badge {
        font-family: monospace;
        font-size: 12px;
        background: #f1f5f9;
        padding: 2px 6px;
        border-radius: 4px;
        color: #475569;
    }
    
    .customer-cell {
        max-width: 150px;
        white-space: nowrap;
        overflow: hidden;
        text-overflow: ellipsis;
    }
    
    .items-cell {
        max-width: 100px;
    }
    
    .items-badge {
        background: #dbeafe;
        color: #1e40af;
        font-size: 11px;
        padding: 1px 6px;
        border-radius: 10px;
    }
    
    .status-badge {
        font-size: 10px;
        padding: 2px 8px;
        border-radius: 10px;
    }
    
    .status-active {
        background-color: #dcfce7;
        color: #166534;
    }
    
    .status-expired {
        background-color: #fee2e2;
        color: #991b1b;
    }
    
    .status-converted {
        background-color: #fef3c7;
        color: #92400e;
    }
    
    .held-by {
        font-size: 11px;
        color: #64748b;
    }
    
    .held-time {
        font-size: 11px;
        color: #475569;
    }
    
    .expires-cell {
        font-size: 11px;
    }
    
    .expires-soon {
        color: #dc2626;
        font-weight: bold;
    }
    
    .expires-normal {
        color: #059669;
    }
    
    .held-actions {
        display: flex;
        gap: 5px;
    }
    
    .held-actions .btn {
        padding: 3px 6px;
        font-size: 12px;
    }
    
    .bulk-actions .card {
        border-width: 2px;
    }
    
    @media (max-width: 768px) {
        .held-table table {
            display: block;
        }
        
        .held-table thead {
            display: none;
        }
        
        .held-table tbody tr {
            display: block;
            margin-bottom: 15px;
            border: 1px solid #e2e8f0;
            border-radius: 8px;
            padding: 10px;
        }
        
        .held-table tbody td {
            display: block;
            text-align: right;
            padding: 5px 10px;
            border: none;
        }
        
        .held-table tbody td::before {
            content: attr(data-label);
            float: left;
            font-weight: bold;
            color: #64748b;
        }
        
        .held-actions {
            justify-content: flex-end;
            margin-top: 10px;
        }
    }
</style>

<script>
    let selectedHolds = [];
    
    function showHeldSalesModal() {
        const modal = new bootstrap.Modal(document.getElementById('heldSalesModal'));
        modal.show();
        loadHeldSales();
        loadHeldStats();
    }
    
    function loadHeldSales() {
        fetch('/api/hold-sales')
            .then(response => response.json())
            .then(data => {
                renderHeldSalesTable(data.holds);
            })
            .catch(error => {
                console.error('Error loading held sales:', error);
                showToast('Failed to load held sales', 'error');
            });
    }
    
    function loadHeldStats() {
        fetch('/api/hold-sales/stats')
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    document.getElementById('activeHolds').textContent = data.active_holds;
                    document.getElementById('totalHeldValue').textContent = 'KES ' + data.total_value.toFixed(2);
                    document.getElementById('expiringSoon').textContent = data.expiring_soon;
                    document.getElementById('todaysHolds').textContent = data.todays_holds;
                }
            })
            .catch(error => {
                console.error('Stats error:', error);
            });
    }
    
    function renderHeldSalesTable(holds) {
        const tbody = document.getElementById('heldTableBody');
        
        if (holds.length === 0) {
            tbody.innerHTML = `
                <tr>
                    <td colspan="9" class="text-center py-4">
                        <i class="bx bx-package fs-1 text-muted mb-3"></i>
                        <p class="text-muted">No held sales found</p>
                        <p class="small text-muted">Hold sales from the POS to see them here</p>
                    </td>
                </tr>
            `;
            return;
        }
        
        let html = '';
        holds.forEach(hold => {
            const isExpired = new Date(hold.expires_at) < new Date();
            const isExpiringSoon = !isExpired && 
                (new Date(hold.expires_at) - new Date()) < 3600000; // Less than 1 hour
            
            const statusClass = isExpired ? 'status-expired' : 
                               hold.status === 'converted' ? 'status-converted' : 'status-active';
            const statusText = isExpired ? 'Expired' : 
                              hold.status === 'converted' ? 'Converted' : 'Active';
            
            const expiresClass = isExpiringSoon ? 'expires-soon' : 'expires-normal';
            const expiresText = isExpired ? 'Expired' : 
                               formatTimeRemaining(hold.expires_at);
            
            html += `
                <tr onclick="toggleHoldSelection(this, ${hold.id})" data-hold-id="${hold.id}">
                    <td data-label="Reference">
                        <div class="reference-badge">${hold.reference}</div>
                    </td>
                    <td data-label="Customer" class="customer-cell">
                        ${hold.customer_name || 'Walk-in'}
                    </td>
                    <td data-label="Items">
                        <span class="items-badge">${hold.item_count} items</span>
                    </td>
                    <td data-label="Total" class="fw-bold">
                        KES ${hold.total.toFixed(2)}
                    </td>
                    <td data-label="Status">
                        <span class="status-badge ${statusClass}">${statusText}</span>
                    </td>
                    <td data-label="Held By">
                        <div class="held-by">${hold.held_by_name}</div>
                    </td>
                    <td data-label="Held At">
                        <div class="held-time">
                            ${new Date(hold.created_at).toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'})}
                        </div>
                    </td>
                    <td data-label="Expires" class="${expiresClass}">
                        ${expiresText}
                    </td>
                    <td data-label="Actions">
                        <div class="held-actions">
                            <button class="btn btn-sm btn-outline-primary" 
                                    onclick="event.stopPropagation(); loadHoldSale(${hold.id})">
                                <i class="bx bx-upload"></i>
                            </button>
                            <button class="btn btn-sm btn-outline-secondary" 
                                    onclick="event.stopPropagation(); viewHoldDetails(${hold.id})">
                                <i class="bx bx-show"></i>
                            </button>
                            <button class="btn btn-sm btn-outline-danger" 
                                    onclick="event.stopPropagation(); deleteHoldSale(${hold.id})">
                                <i class="bx bx-trash"></i>
                            </button>
                        </div>
                    </td>
                </tr>
            `;
        });
        
        tbody.innerHTML = html;
    }
    
    function toggleHoldSelection(row, holdId) {
        row.classList.toggle('selected');
        
        const index = selectedHolds.indexOf(holdId);
        if (index === -1) {
            selectedHolds.push(holdId);
        } else {
            selectedHolds.splice(index, 1);
        }
        
        updateBulkActions();
    }
    
    function updateBulkActions() {
        const bulkActions = document.getElementById('bulkActions');
        const selectedCount = document.getElementById('selectedCount');
        const selectedTotal = document.getElementById('selectedTotal');
        
        if (selectedHolds.length > 0) {
            bulkActions.style.display = 'block';
            selectedCount.textContent = selectedHolds.length;
            
            // Calculate total (this would need server-side calculation)
            selectedTotal.textContent = 'KES 0.00'; // Placeholder
        } else {
            bulkActions.style.display = 'none';
        }
    }
    
    function clearSelection() {
        selectedHolds = [];
        document.querySelectorAll('#heldTableBody tr.selected').forEach(row => {
            row.classList.remove('selected');
        });
        updateBulkActions();
    }
    
    function loadHoldSale(holdId) {
        fetch(`/api/hold-sales/${holdId}/load`)
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Load items into cart
                    cart = data.items;
                    
                    // Load customer if exists
                    if (data.customer) {
                        setCurrentCustomer(data.customer);
                    }
                    
                    // Update UI
                    updateCartDisplay();
                    
                    // Close modal
                    const modal = bootstrap.Modal.getInstance(document.getElementById('heldSalesModal'));
                    modal.hide();
                    
                    showToast('Hold sale loaded successfully', 'success');
                }
            })
            .catch(error => {
                console.error('Load hold error:', error);
                showToast('Failed to load hold sale', 'error');
            });
    }
    
    function loadSelectedSales() {
        if (selectedHolds.length === 0) return;
        
        if (confirm(`Load ${selectedHolds.length} selected holds?`)) {
            // This would need backend support for loading multiple holds
            showToast('Loading selected holds...', 'info');
        }
    }
    
    function deleteHoldSale(holdId) {
        if (!confirm('Are you sure you want to delete this hold sale?')) return;
        
        fetch(`/api/hold-sales/${holdId}`, {
            method: 'DELETE',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showToast('Hold sale deleted', 'success');
                loadHeldSales();
                loadHeldStats();
            }
        })
        .catch(error => {
            console.error('Delete error:', error);
            showToast('Failed to delete hold sale', 'error');
        });
    }
    
    function deleteSelectedSales() {
        if (selectedHolds.length === 0) return;
        
        if (confirm(`Delete ${selectedHolds.length} selected hold sales?`)) {
            showToast('Deleting selected holds...', 'info');
            // Implement bulk deletion
        }
    }
    
    function viewHoldDetails(holdId) {
        fetch(`/api/hold-sales/${holdId}`)
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Show details in a modal or side panel
                    showHoldDetailsModal(data.hold);
                }
            })
            .catch(error => {
                console.error('View details error:', error);
                showToast('Failed to load details', 'error');
            });
    }
    
    function exportHeldSales() {
        const format = prompt('Export format (csv/json):', 'csv');
        if (!format) return;
        
        fetch(`/api/hold-sales/export?format=${format}`)
            .then(response => response.blob())
            .then(blob => {
                const url = window.URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = `held-sales-${new Date().toISOString().split('T')[0]}.${format}`;
                document.body.appendChild(a);
                a.click();
                window.URL.revokeObjectURL(url);
                document.body.removeChild(a);
            })
            .catch(error => {
                console.error('Export error:', error);
                showToast('Export failed', 'error');
            });
    }
    
    function clearExpiredHolds() {
        if (!confirm('Clear all expired hold sales?')) return;
        
        fetch('/api/hold-sales/clear-expired', {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showToast(`Cleared ${data.cleared_count} expired holds`, 'success');
                loadHeldSales();
                loadHeldStats();
            }
        })
        .catch(error => {
            console.error('Clear expired error:', error);
            showToast('Failed to clear expired holds', 'error');
        });
    }
    
    function formatTimeRemaining(expiryDate) {
        const now = new Date();
        const expiry = new Date(expiryDate);
        const diffMs = expiry - now;
        
        if (diffMs <= 0) return 'Expired';
        
        const diffHours = Math.floor(diffMs / (1000 * 60 * 60));
        const diffMinutes = Math.floor((diffMs % (1000 * 60 * 60)) / (1000 * 60));
        
        if (diffHours > 0) {
            return `${diffHours}h ${diffMinutes}m`;
        } else {
            return `${diffMinutes}m`;
        }
    }
    
    function showHoldDetailsModal(hold) {
        // Implement a modal to show hold details
        const items = JSON.parse(hold.items || '[]');
        
        let itemsHtml = '';
        items.forEach(item => {
            itemsHtml += `
                <tr>
                    <td>${item.name}</td>
                    <td class="text-center">${item.quantity}</td>
                    <td class="text-end">KES ${item.price.toFixed(2)}</td>
                    <td class="text-end">KES ${(item.quantity * item.price).toFixed(2)}</td>
                </tr>
            `;
        });
        
        const modalHtml = `
            <div class="modal fade" id="holdDetailsModal">
                <div class="modal-dialog modal-lg">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title">Hold Sale Details</h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body">
                            <div class="row mb-3">
                                <div class="col-md-6">
                                    <strong>Reference:</strong> ${hold.reference}<br>
                                    <strong>Customer:</strong> ${hold.customer_name || 'Walk-in'}<br>
                                    <strong>Held By:</strong> ${hold.held_by_name}
                                </div>
                                <div class="col-md-6">
                                    <strong>Created:</strong> ${new Date(hold.created_at).toLocaleString()}<br>
                                    <strong>Expires:</strong> ${new Date(hold.expires_at).toLocaleString()}<br>
                                    <strong>Status:</strong> ${hold.status}
                                </div>
                            </div>
                            <div class="table-responsive">
                                <table class="table table-sm">
                                    <thead>
                                        <tr>
                                            <th>Item</th>
                                            <th class="text-center">Qty</th>
                                            <th class="text-end">Price</th>
                                            <th class="text-end">Total</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        ${itemsHtml}
                                    </tbody>
                                    <tfoot>
                                        <tr>
                                            <td colspan="3" class="text-end"><strong>Total:</strong></td>
                                            <td class="text-end fw-bold">KES ${hold.total.toFixed(2)}</td>
                                        </tr>
                                    </tfoot>
                                </table>
                            </div>
                            ${hold.notes ? `
                                <div class="mt-3">
                                    <strong>Notes:</strong>
                                    <p class="text-muted">${hold.notes}</p>
                                </div>
                            ` : ''}
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        // Add modal to DOM and show it
        const modalContainer = document.createElement('div');
        modalContainer.innerHTML = modalHtml;
        document.body.appendChild(modalContainer);
        
        const modal = new bootstrap.Modal(modalContainer.querySelector('#holdDetailsModal'));
        modal.show();
        
        // Remove modal from DOM after it's hidden
        modalContainer.querySelector('#holdDetailsModal').addEventListener('hidden.bs.modal', function() {
            document.body.removeChild(modalContainer);
        });
    }
    
    // Event listeners
    document.getElementById('heldSearch').addEventListener('input', function() {
        // Implement search with debounce
        clearTimeout(window.holdSearchTimeout);
        window.holdSearchTimeout = setTimeout(searchHeldSales, 300);
    });
    
    document.getElementById('heldStatusFilter').addEventListener('change', searchHeldSales);
    document.getElementById('heldDateFilter').addEventListener('change', searchHeldSales);
    
    function searchHeldSales() {
        const search = document.getElementById('heldSearch').value;
        const status = document.getElementById('heldStatusFilter').value;
        const date = document.getElementById('heldDateFilter').value;
        
        let url = `/api/hold-sales/search?q=${encodeURIComponent(search)}`;
        if (status !== 'all') url += `&status=${status}`;
        if (date !== 'all') url += `&date=${date}`;
        
        fetch(url)
            .then(response => response.json())
            .then(data => {
                renderHeldSalesTable(data.holds);
            })
            .catch(error => {
                console.error('Search error:', error);
            });
    }
</script>