@extends('layouts.app')

@section('title', 'MPESA Transactions')

@section('style')
<style>
    .stat-card {
        border-radius: 10px;
        transition: transform 0.2s;
    }
    .stat-card:hover {
        transform: translateY(-2px);
    }
    .badge-filter {
        cursor: pointer;
    }
    .filter-badge {
        display: inline-flex;
        align-items: center;
        gap: 4px;
    }
    .table-actions {
        white-space: nowrap;
    }
</style>
@endsection

@section('content')
<!--start page wrapper -->
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">Payments</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('payments.mpesa.dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item active" aria-current="page">MPESA Transactions</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <div class="btn-group">
                    <a href="{{ route('payments.mpesa.dashboard') }}" class="btn btn-outline-primary">
                        <i class="bx bx-tachometer me-1"></i> Dashboard
                    </a>
                    <button type="button" class="btn btn-outline-primary" data-bs-toggle="modal" data-bs-target="#filterModal">
                        <i class="bx bx-filter me-1"></i> Filters
                    </button>
                    <button type="button" class="btn btn-outline-primary" data-bs-toggle="modal" data-bs-target="#exportModal">
                        <i class="bx bx-download me-1"></i> Export
                    </button>
                </div>
            </div>
        </div>
        <!--end breadcrumb-->

        <!-- Stats Cards -->
        <div class="row row-cols-1 row-cols-sm-2 row-cols-xl-3 row-cols-xxl-6 g-3 mb-4">
            <div class="col">
                <div class="card stat-card radius-10 border-start border-primary border-3">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-1">Total</p>
                                <h4 class="mb-0">{{ number_format($stats['total']) }}</h4>
                            </div>
                            <div class="ms-auto text-primary">
                                <i class="bx bx-transfer font-30"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="card stat-card radius-10 border-start border-success border-3">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-1">Completed</p>
                                <h4 class="mb-0">{{ number_format($stats['completed']) }}</h4>
                            </div>
                            <div class="ms-auto text-success">
                                <i class="bx bx-check-circle font-30"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="card stat-card radius-10 border-start border-warning border-3">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-1">Unmatched</p>
                                <h4 class="mb-0">{{ number_format($stats['unmatched']) }}</h4>
                            </div>
                            <div class="ms-auto text-warning">
                                <i class="bx bx-unlink font-30"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="card stat-card radius-10 border-start border-info border-3">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-1">Matched</p>
                                <h4 class="mb-0">{{ number_format($stats['matched']) }}</h4>
                            </div>
                            <div class="ms-auto text-info">
                                <i class="bx bx-link font-30"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="card stat-card radius-10 border-start border-danger border-3">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-1">Failed</p>
                                <h4 class="mb-0">{{ number_format($stats['failed']) }}</h4>
                            </div>
                            <div class="ms-auto text-danger">
                                <i class="bx bx-x-circle font-30"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="card stat-card radius-10 border-start border-secondary border-3">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-1">Total Amount</p>
                                <h4 class="mb-0">Ksh {{ number_format($stats['total_amount'], 2) }}</h4>
                            </div>
                            <div class="ms-auto text-secondary">
                                <i class="bx bx-money font-30"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div><!--end row-->

        <!-- Active Filters -->
        @if(request()->hasAny(['search', 'status', 'date', 'start_date', 'end_date', 'phone', 'min_amount', 'max_amount', 'unmatched', 'matched', 'failed']))
            <div class="alert alert-info alert-dismissible fade show mb-4" role="alert">
                <div class="d-flex align-items-center">
                    <i class="bx bx-filter font-30 text-info me-3"></i>
                    <div>
                        <strong>Active Filters:</strong>
                        <div class="mt-1">
                            @if(request('search'))
                                <span class="badge bg-light text-dark me-1 mb-1">
                                    Search: "{{ request('search') }}"
                                </span>
                            @endif
                            @if(request('status'))
                                <span class="badge bg-light text-dark me-1 mb-1">
                                    Status: {{ $statusOptions[request('status')] ?? request('status') }}
                                </span>
                            @endif
                            @if(request('date'))
                                <span class="badge bg-light text-dark me-1 mb-1">
                                    Date: {{ request('date') }}
                                </span>
                            @endif
                            @if(request('start_date') && request('end_date'))
                                <span class="badge bg-light text-dark me-1 mb-1">
                                    Range: {{ request('start_date') }} to {{ request('end_date') }}
                                </span>
                            @endif
                            @if(request('phone'))
                                <span class="badge bg-light text-dark me-1 mb-1">
                                    Phone: {{ request('phone') }}
                                </span>
                            @endif
                            @if(request('unmatched'))
                                <span class="badge bg-warning me-1 mb-1">Unmatched Only</span>
                            @endif
                            @if(request('matched'))
                                <span class="badge bg-success me-1 mb-1">Matched Only</span>
                            @endif
                            @if(request('failed'))
                                <span class="badge bg-danger me-1 mb-1">Failed Only</span>
                            @endif
                        </div>
                    </div>
                </div>
                <a href="{{ route('payments.mpesa.transactions') }}" class="btn btn-sm btn-outline-info mt-2">
                    <i class="bx bx-x me-1"></i> Clear Filters
                </a>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        @endif

        <!-- Transactions Table -->
        <div class="card radius-10">
            <div class="card-body">
                <div class="d-flex align-items-center">
                    <div>
                        <h6 class="mb-0">Transactions List</h6>
                    </div>
                    <div class="dropdown ms-auto">
                        <a class="dropdown-toggle dropdown-toggle-nocaret" href="#" data-bs-toggle="dropdown">
                            <i class='bx bx-dots-horizontal-rounded font-22 text-option'></i>
                        </a>
                        <ul class="dropdown-menu">
                            <li>
                                <a class="dropdown-item" href="{{ route('payments.mpesa.reports.unmatched') }}">
                                    <i class="bx bx-unlink me-2"></i>View Unmatched
                                </a>
                            </li>
                            <li>
                                <a class="dropdown-item" href="{{ route('payments.mpesa.reports.reconciliation') }}">
                                    <i class="bx bx-balance me-2"></i>Reconciliation Report
                                </a>
                            </li>
                            <li><hr class="dropdown-divider"></li>
                            <li>
                                <a class="dropdown-item" href="#" data-bs-toggle="modal" data-bs-target="#bulkMatchModal">
                                    <i class="bx bx-link me-2"></i>Bulk Match
                                </a>
                            </li>
                        </ul>
                    </div>
                </div>
                <hr>
                <div class="table-responsive">
                    <table class="table align-middle mb-0">
                        <thead class="table-light">
                            <tr>
                                <th>Transaction ID</th>
                                <th>Phone</th>
                                <th>Customer</th>
                                <th>Amount</th>
                                <th>Date</th>
                                <th>Status</th>
                                <th>Sale</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @forelse($transactions as $transaction)
                                <tr>
                                    <td>
                                        <code>{{ $transaction->transaction_id }}</code>
                                        @if($transaction->mpesa_receipt_number)
                                            <br><small class="text-muted">Receipt: {{ $transaction->mpesa_receipt_number }}</small>
                                        @endif
                                    </td>
                                    <td>
                                        <a href="{{ route('payments.mpesa.transactions.by-phone', $transaction->phone) }}" 
                                           class="text-primary">
                                            {{ $transaction->formatted_phone }}
                                        </a>
                                    </td>
                                    <td>
                                        @if($transaction->full_name)
                                            {{ $transaction->full_name }}
                                        @else
                                            <span class="text-muted">-</span>
                                        @endif
                                        @if($transaction->customer)
                                            <br><small class="text-success">
                                                <i class="bx bx-user-check"></i> Registered
                                            </small>
                                        @endif
                                    </td>
                                    <td class="fw-bold">Ksh {{ number_format($transaction->amount, 2) }}</td>
                                    <td>
                                        {{ $transaction->transaction_date->format('M d, Y') }}<br>
                                        <small class="text-muted">{{ $transaction->transaction_date->format('h:i A') }}</small>
                                    </td>
                                    <td>
                                        @if($transaction->status == 'completed')
                                            <span class="badge bg-success">Completed</span>
                                        @elseif($transaction->status == 'pending')
                                            <span class="badge bg-warning">Pending</span>
                                        @elseif($transaction->status == 'failed')
                                            <span class="badge bg-danger">Failed</span>
                                        @elseif($transaction->status == 'reversed')
                                            <span class="badge bg-secondary">Reversed</span>
                                        @elseif($transaction->status == 'matched')
                                            <span class="badge bg-primary">Matched</span>
                                        @endif
                                        
                                        @if($transaction->result_desc)
                                            <br><small class="text-muted">{{ $transaction->result_desc }}</small>
                                        @endif
                                    </td>
                                    <td>
                                        @if($transaction->sale_id)
                                            <a href="{{ route('sales.show', $transaction->sale_id) }}" 
                                               class="badge bg-info">
                                                Sale #{{ $transaction->sale_id }}
                                            </a>
                                            @if($transaction->match_type)
                                                <br><small class="text-muted">{{ ucfirst($transaction->match_type) }}</small>
                                            @endif
                                        @elseif($transaction->status == 'completed')
                                            <span class="badge bg-warning">Unmatched</span>
                                        @else
                                            <span class="badge bg-light text-dark">-</span>
                                        @endif
                                    </td>
                                    <td>
                                        <div class="d-flex order-actions gap-2">
                                            <a href="{{ route('payments.mpesa.transactions.show', $transaction->id) }}" 
                                               class="btn btn-sm btn-outline-primary" data-bs-toggle="tooltip" title="View Details">
                                                <i class="bx bx-show"></i>
                                            </a>
                                            @if(!$transaction->sale_id && $transaction->status == 'completed')
                                                <button class="btn btn-sm btn-outline-success match-transaction" 
                                                        data-id="{{ $transaction->id }}"
                                                        data-bs-toggle="tooltip" title="Match with Sale">
                                                    <i class="bx bx-link"></i>
                                                </button>
                                            @endif
                                            @if($transaction->canBeReversed() && auth()->user()->can('mpesa.manage'))
                                                <button class="btn btn-sm btn-outline-danger reverse-transaction"
                                                        data-id="{{ $transaction->id }}"
                                                        data-bs-toggle="tooltip" title="Reverse Transaction">
                                                    <i class="bx bx-undo"></i>
                                                </button>
                                            @endif
                                        </div>
                                    </td>
                                </tr>
                            @empty
                                <tr>
                                    <td colspan="8" class="text-center py-4">
                                        <div class="widgets-icons bg-light-secondary text-secondary rounded-circle mx-auto mb-3">
                                            <i class="bx bx-transfer"></i>
                                        </div>
                                        <h5 class="text-gray-700 mb-2">No transactions found</h5>
                                        <p class="text-muted">
                                            @if(request()->hasAny(['search', 'status', 'date', 'phone']))
                                                Try adjusting your filters
                                            @else
                                                No MPESA transactions have been recorded yet
                                            @endif
                                        </p>
                                    </td>
                                </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                @if($transactions->hasPages())
                    <div class="d-flex justify-content-between align-items-center mt-4">
                        <div class="text-muted">
                            Showing {{ $transactions->firstItem() }} to {{ $transactions->lastItem() }} of {{ $transactions->total() }} entries
                        </div>
                        <nav>
                            {{ $transactions->withQueryString()->links() }}
                        </nav>
                    </div>
                @endif
            </div>
        </div>
    </div>
</div>
<!--end page wrapper -->

<!-- Filter Modal -->
<div class="modal fade" id="filterModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="bx bx-filter me-2"></i>Filter Transactions</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form action="{{ route('payments.mpesa.transactions') }}" method="GET">
                <div class="modal-body">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label for="search" class="form-label">Search</label>
                            <input type="text" class="form-control" id="search" name="search" 
                                   value="{{ request('search') }}" placeholder="Transaction ID, Phone, Reference...">
                        </div>
                        <div class="col-md-6">
                            <label for="status" class="form-label">Status</label>
                            <select class="form-select" id="status" name="status">
                                <option value="">All Statuses</option>
                                @foreach($statusOptions as $value => $label)
                                    <option value="{{ $value }}" {{ request('status') == $value ? 'selected' : '' }}>
                                        {{ $label }}
                                    </option>
                                @endforeach
                            </select>
                        </div>
                    </div>

                    <div class="row g-3 mt-2">
                        <div class="col-md-6">
                            <label for="phone" class="form-label">Phone Number</label>
                            <input type="text" class="form-control" id="phone" name="phone" 
                                   value="{{ request('phone') }}" placeholder="e.g., 0712345678">
                        </div>
                        <div class="col-md-6">
                            <label for="date" class="form-label">Specific Date</label>
                            <input type="date" class="form-control" id="date" name="date" 
                                   value="{{ request('date') }}">
                        </div>
                    </div>

                    <div class="row g-3 mt-2">
                        <div class="col-md-6">
                            <label for="start_date" class="form-label">Start Date</label>
                            <input type="date" class="form-control" id="start_date" name="start_date" 
                                   value="{{ request('start_date') }}">
                        </div>
                        <div class="col-md-6">
                            <label for="end_date" class="form-label">End Date</label>
                            <input type="date" class="form-control" id="end_date" name="end_date" 
                                   value="{{ request('end_date') }}">
                        </div>
                    </div>

                    <div class="row g-3 mt-2">
                        <div class="col-md-6">
                            <label for="min_amount" class="form-label">Minimum Amount</label>
                            <input type="number" class="form-control" id="min_amount" name="min_amount" 
                                   value="{{ request('min_amount') }}" min="0" step="0.01">
                        </div>
                        <div class="col-md-6">
                            <label for="max_amount" class="form-label">Maximum Amount</label>
                            <input type="number" class="form-control" id="max_amount" name="max_amount" 
                                   value="{{ request('max_amount') }}" min="0" step="0.01">
                        </div>
                    </div>

                    <div class="row g-3 mt-2">
                        <div class="col-12">
                            <div class="form-check form-check-inline">
                                <input class="form-check-input" type="checkbox" id="unmatched" name="unmatched" 
                                       value="1" {{ request('unmatched') ? 'checked' : '' }}>
                                <label class="form-check-label" for="unmatched">Unmatched Only</label>
                            </div>
                            <div class="form-check form-check-inline">
                                <input class="form-check-input" type="checkbox" id="matched" name="matched" 
                                       value="1" {{ request('matched') ? 'checked' : '' }}>
                                <label class="form-check-label" for="matched">Matched Only</label>
                            </div>
                            <div class="form-check form-check-inline">
                                <input class="form-check-input" type="checkbox" id="failed" name="failed" 
                                       value="1" {{ request('failed') ? 'checked' : '' }}>
                                <label class="form-check-label" for="failed">Failed Only</label>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <a href="{{ route('payments.mpesa.transactions') }}" class="btn btn-outline-secondary">
                        Clear Filters
                    </a>
                    <button type="submit" class="btn btn-primary">
                        <i class="bx bx-filter me-1"></i> Apply Filters
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Export Modal -->
<div class="modal fade" id="exportModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="bx bx-download me-2"></i>Export Transactions</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form action="{{ route('payments.mpesa.transactions.export') }}" method="POST">
                @csrf
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="export_format" class="form-label">Format</label>
                        <select class="form-select" id="export_format" name="format" required>
                            <option value="csv">CSV</option>
                            <option value="excel">Excel</option>
                            <option value="pdf">PDF</option>
                        </select>
                    </div>

                    <div class="row g-3">
                        <div class="col-md-6">
                            <label for="export_start_date" class="form-label">Start Date</label>
                            <input type="date" class="form-control" id="export_start_date" name="start_date">
                        </div>
                        <div class="col-md-6">
                            <label for="export_end_date" class="form-label">End Date</label>
                            <input type="date" class="form-control" id="export_end_date" name="end_date">
                        </div>
                    </div>

                    <div class="mb-3 mt-3">
                        <label for="export_status" class="form-label">Status</label>
                        <select class="form-select" id="export_status" name="status">
                            <option value="">All Statuses</option>
                            @foreach($statusOptions as $value => $label)
                                <option value="{{ $value }}">{{ $label }}</option>
                            @endforeach
                        </select>
                    </div>

                    <div class="form-check mb-0">
                        <input class="form-check-input" type="checkbox" id="include_details" name="include_details" value="1">
                        <label class="form-check-label" for="include_details">
                            Include transaction details
                        </label>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">
                        <i class="bx bx-download me-1"></i> Export
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Bulk Match Modal -->
<div class="modal fade" id="bulkMatchModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="bx bx-link me-2"></i>Bulk Match Transactions</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div id="bulkMatchApp">
                    <div class="alert bg-light-info border-0">
                        <i class="bx bx-info-circle me-2"></i>
                        Select unmatched transactions and match them with sales in bulk.
                    </div>

                    <!-- Step 1: Select Transactions -->
                    <div id="step1">
                        <h6 class="fw-bold mb-3">Step 1: Select Unmatched Transactions</h6>
                        <div class="table-responsive">
                            <table class="table table-sm mb-0">
                                <thead class="table-light">
                                    <tr>
                                        <th width="50"><input type="checkbox" id="selectAll"></th>
                                        <th>Transaction ID</th>
                                        <th>Phone</th>
                                        <th>Amount</th>
                                        <th>Date</th>
                                        <th>Customer</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach($transactions->whereNull('sale_id')->where('status', 'completed') as $transaction)
                                        <tr>
                                            <td>
                                                <input type="checkbox" class="transaction-check" 
                                                       value="{{ $transaction->id }}"
                                                       data-amount="{{ $transaction->amount }}"
                                                       data-phone="{{ $transaction->phone }}">
                                            </td>
                                            <td>{{ $transaction->transaction_id }}</td>
                                            <td>{{ $transaction->formatted_phone }}</td>
                                            <td>Ksh {{ number_format($transaction->amount, 2) }}</td>
                                            <td>{{ $transaction->transaction_date->format('M d, Y') }}</td>
                                            <td>
                                                @if($transaction->customer)
                                                    <span class="text-success">{{ $transaction->customer->name }}</span>
                                                @else
                                                    <span class="text-muted">-</span>
                                                @endif
                                            </td>
                                        </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                    </div>

                    <!-- Step 2: Select Sales (hidden initially) -->
                    <div id="step2" class="d-none">
                        <h6 class="fw-bold mb-3">Step 2: Select Sales to Match</h6>
                        <div class="alert bg-light-warning border-0">
                            <i class="bx bx-error-circle me-2"></i>
                            Make sure the sale amounts match the transaction amounts as closely as possible.
                        </div>
                        <div id="salesList"></div>
                    </div>

                    <!-- Summary -->
                    <div class="mt-4 p-3 bg-light rounded">
                        <h6 class="fw-bold mb-2">Summary</h6>
                        <div class="row">
                            <div class="col-md-4">
                                <small class="text-muted">Selected Transactions</small>
                                <div class="fw-bold" id="selectedCount">0</div>
                            </div>
                            <div class="col-md-4">
                                <small class="text-muted">Total Amount</small>
                                <div class="fw-bold" id="totalAmount">Ksh 0.00</div>
                            </div>
                            <div class="col-md-4">
                                <small class="text-muted">Status</small>
                                <div class="fw-bold" id="bulkStatus">Ready</div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-outline-primary" id="prevStep" class="d-none">
                    <i class="bx bx-arrow-left me-1"></i> Previous
                </button>
                <button type="button" class="btn btn-primary" id="nextStep">
                    <i class="bx bx-arrow-right me-1"></i> Next
                </button>
                <button type="button" class="btn btn-success" id="confirmBulkMatch" class="d-none">
                    <i class="bx bx-check me-1"></i> Confirm Match
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Bulk Match functionality
        let currentStep = 1;
        let selectedTransactions = [];
        let selectedSales = {};

        // Select all transactions
        document.getElementById('selectAll')?.addEventListener('change', function() {
            const checkboxes = document.querySelectorAll('.transaction-check');
            check.forEach(checkbox => {
                checkbox.checked = this.checked;
                checkbox.dispatchEvent(new Event('change'));
            });
        });

        // Transaction selection
        document.querySelectorAll('.transaction-check').forEach(checkbox => {
            checkbox.addEventListener('change', updateSummary);
        });

        // Update summary
        function updateSummary() {
            selectedTransactions = [];
            let totalAmount = 0;
            
            document.querySelectorAll('.transaction-check:checked').forEach(checkbox => {
                const transaction = {
                    id: checkbox.value,
                    amount: parseFloat(checkbox.getAttribute('data-amount')),
                    phone: checkbox.getAttribute('data-phone')
                };
                
                selectedTransactions.push(transaction);
                totalAmount += transaction.amount;
            });
            
            document.getElementById('selectedCount').textContent = selectedTransactions.length;
            document.getElementById('totalAmount').textContent = 'Ksh ' + totalAmount.toLocaleString('en-US', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
            });
            
            // Enable/disable next button
            document.getElementById('nextStep').disabled = selectedTransactions.length === 0;
        }

        // Next step
        document.getElementById('nextStep').addEventListener('click', function() {
            if (currentStep === 1) {
                // Load sales for selected transactions
                loadSalesForTransactions();
                document.getElementById('step1').classList.add('d-none');
                document.getElementById('step2').classList.remove('d-none');
                document.getElementById('prevStep').classList.remove('d-none');
                document.getElementById('nextStep').classList.add('d-none');
                document.getElementById('confirmBulkMatch').classList.remove('d-none');
                currentStep = 2;
                document.getElementById('bulkStatus').textContent = 'Select sales to match';
            }
        });

        // Previous step
        document.getElementById('prevStep').addEventListener('click', function() {
            if (currentStep === 2) {
                document.getElementById('step1').classList.remove('d-none');
                document.getElementById('step2').classList.add('d-none');
                document.getElementById('prevStep').classList.add('d-none');
                document.getElementById('nextStep').classList.remove('d-none');
                document.getElementById('confirmBulkMatch').classList.add('d-none');
                currentStep = 1;
                document.getElementById('bulkStatus').textContent = 'Ready';
            }
        });

        // Load sales for transactions
        function loadSalesForTransactions() {
            const phoneList = [...new Set(selectedTransactions.map(t => t.phone))];
            const html = [];
            
            selectedTransactions.forEach(transaction => {
                html.push(`
                    <div class="card mb-3">
                        <div class="card-body">
                            <h6 class="card-title">Transaction: ${transaction.id}</h6>
                            <div class="row">
                                <div class="col-md-4">
                                    <small class="text-muted">Amount</small>
                                    <div class="fw-bold">Ksh ${transaction.amount.toFixed(2)}</div>
                                </div>
                                <div class="col-md-4">
                                    <small class="text-muted">Phone</small>
                                    <div>${transaction.phone}</div>
                                </div>
                                <div class="col-md-4">
                                    <div class="mb-3">
                                        <label for="sale_${transaction.id}" class="form-label">Select Sale</label>
                                        <select class="form-select sale-select" 
                                                data-transaction="${transaction.id}"
                                                id="sale_${transaction.id}">
                                            <option value="">-- Select Sale --</option>
                                        </select>
                                    </div>
                                </div>
                            </div>
                            <div class="mt-2 sales-list-${transaction.id}"></div>
                        </div>
                    </div>
                `);
            });
            
            document.getElementById('salesList').innerHTML = html.join('');
            
            // Load sales for each phone
            phoneList.forEach(phone => {
                loadSalesByPhone(phone);
            });
        }

        // Load sales by phone
        function loadSalesByPhone(phone) {
            fetch(`{{ route("api.sales.search") }}?phone=${phone}&payment_status=pending&limit=10`)
                .then(response => response.json())
                .then(response => {
                    if (response.success && response.sales.length > 0) {
                        // Update dropdowns for transactions with this phone
                        selectedTransactions
                            .filter(t => t.phone === phone)
                            .forEach(transaction => {
                                const select = document.getElementById(`sale_${transaction.id}`);
                                if (select) {
                                    select.innerHTML = '<option value="">-- Select Sale --</option>';
                                    
                                    response.sales.forEach(sale => {
                                        const option = document.createElement('option');
                                        option.value = sale.id;
                                        option.textContent = `Sale #${sale.id} - Ksh ${parseFloat(sale.grand_total).toFixed(2)} - ${sale.customer?.name || 'No Customer'}`;
                                        select.appendChild(option);
                                    });
                                }
                            });
                    }
                });
        }

        // Sale selection
        document.addEventListener('change', function(e) {
            if (e.target.classList.contains('sale-select')) {
                const transactionId = e.target.getAttribute('data-transaction');
                const saleId = e.target.value;
                
                if (saleId) {
                    selectedSales[transactionId] = saleId;
                } else {
                    delete selectedSales[transactionId];
                }
                
                // Enable/disable confirm button
                const canConfirm = Object.keys(selectedSales).length === selectedTransactions.length;
                document.getElementById('confirmBulkMatch').disabled = !canConfirm;
            }
        });

        // Confirm bulk match
        document.getElementById('confirmBulkMatch').addEventListener('click', function() {
            if (Object.keys(selectedSales).length !== selectedTransactions.length) {
                alert('Please select a sale for each transaction');
                return;
            }
            
            const matches = selectedTransactions.map(transaction => ({
                transaction_id: transaction.id,
                sale_id: selectedSales[transaction.id],
                match_type: 'manual'
            }));
            
            fetch('{{ route("payments.mpesa.transactions.bulk-match") }}', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                body: JSON.stringify({ matches: matches })
            })
            .then(response => response.json())
            .then(response => {
                if (response.success) {
                    alert(response.message);
                    const modal = bootstrap.Modal.getInstance(document.getElementById('bulkMatchModal'));
                    modal.hide();
                    location.reload();
                } else {
                    alert(response.message || 'Bulk match failed');
                }
            })
            .catch(() => {
                alert('An error occurred during bulk match');
            });
        });

        // Individual match transaction
        document.querySelectorAll('.match-transaction').forEach(button => {
            button.addEventListener('click', function() {
                const transactionId = this.getAttribute('data-id');
                showMatchModal(transactionId);
            });
        });

        // Reverse transaction
        document.querySelectorAll('.reverse-transaction').forEach(button => {
            button.addEventListener('click', function() {
                const transactionId = this.getAttribute('data-id');
                if (confirm('Are you sure you want to reverse this transaction?')) {
                    showReversalModal(transactionId);
                }
            });
        });

        // Initialize tooltips
        const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
        tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl);
        });
    });
</script>
@endsection