@extends('layouts.app')

@section('title', 'MPESA Settings')

@section('style')
<style>
    .settings-card {
        border-left: 4px solid #dee2e6;
        transition: all 0.3s;
    }
    .settings-card:hover {
        box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    }
    .section-divider {
        border-top: 2px solid #f8f9fa;
        margin: 2rem 0;
        padding-top: 2rem;
    }
    .setting-group {
        background-color: #f8f9fa;
        border-radius: 8px;
        padding: 1.5rem;
        margin-bottom: 1.5rem;
    }
</style>
@endsection

@section('content')
<!--start page wrapper -->
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">Payments</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('payments.mpesa.dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item"><a href="{{ route('payments.mpesa.configuration') }}">MPESA</a></li>
                        <li class="breadcrumb-item active" aria-current="page">Settings</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <div class="btn-group">
                    <a href="{{ route('payments.mpesa.dashboard') }}" class="btn btn-outline-primary">
                        <i class="bx bx-tachometer me-1"></i> Dashboard
                    </a>
                    <a href="{{ route('payments.mpesa.configuration') }}" class="btn btn-outline-primary">
                        <i class="bx bx-cog me-1"></i> Configuration
                    </a>
                </div>
            </div>
        </div>
        <!--end breadcrumb-->

        @if(session('success'))
            <div class="alert border-0 border-start border-5 border-success alert-dismissible fade show py-2">
                <div class="d-flex align-items-center">
                    <div class="font-35 text-success"><i class="bx bxs-check-circle"></i></div>
                    <div class="ms-3">
                        <h6 class="mb-0 text-success">Success!</h6>
                        <div>{{ session('success') }}</div>
                    </div>
                </div>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        @endif

        @if(session('settings_updated'))
            <div class="alert border-0 border-start border-5 border-info alert-dismissible fade show py-2">
                <div class="d-flex align-items-center">
                    <div class="font-35 text-info"><i class="bx bx-info-circle"></i></div>
                    <div class="ms-3">
                        <h6 class="mb-0 text-info">Settings Updated</h6>
                        <div>Settings have been saved successfully. Some changes may require a system restart to take effect.</div>
                    </div>
                </div>
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        @endif

        <div class="row">
            <div class="col-12">
                <div class="card radius-10">
                    <div class="card-body">
                        <div class="d-flex align-items-center mb-3">
                            <div>
                                <h6 class="mb-0">
                                    <i class="bx bx-cog me-2"></i>MPESA System Settings
                                </h6>
                            </div>
                            <div class="ms-auto">
                                <button type="button" class="btn btn-primary" onclick="document.getElementById('settingsForm').submit()">
                                    <i class="bx bx-save me-1"></i> Save Settings
                                </button>
                            </div>
                        </div>
                        <hr>
                        
                        <form id="settingsForm" method="POST" action="{{ route('mpesa.settings.update') }}">
                            @csrf
                            
                            <!-- Auto Processing Settings -->
                            <div class="setting-group">
                                <h5 class="mb-3">
                                    <i class="bx bx-refresh me-2"></i>Auto Processing Settings
                                </h5>
                                
                                <div class="row g-3">
                                    <div class="col-md-6">
                                        <div class="form-check form-switch mb-3">
                                        <input class="form-check-input" type="checkbox" id="auto_check_payments" 
                                            name="auto_check_payments" value="1" 
                                            {{ $settings['auto_check_payments'] == '1' || $settings['auto_check_payments'] === true ? 'checked' : '' }}>
                                                Auto Check for Payments
                                            </label>
                                            <div class="form-text">Automatically check for new MPESA payments</div>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label for="auto_check_minutes" class="form-label">Check Interval (Minutes)</label>
                                            <input type="number" class="form-control" id="auto_check_minutes" 
                                                   name="auto_check_minutes" min="1" max="1440" 
                                                   value="{{ $settings['auto_check_minutes'] }}">
                                            <div class="form-text">How often to check for new payments (1-1440 minutes)</div>
                                        </div>
                                    </div>
                                    
                                    <div class="col-md-6">
                                        <div class="form-check form-switch mb-3">
                                            <input class="form-check-input" type="checkbox" id="auto_match_payments" 
                                                   name="auto_match_payments" value="1" 
                                                   {{ $settings['auto_match_payments'] ? 'checked' : '' }}>
                                            <label class="form-check-label" for="auto_match_payments">
                                                Auto Match Payments
                                            </label>
                                            <div class="form-text">Automatically match payments to sales when possible</div>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label for="auto_sync_hours" class="form-label">Auto Sync Interval (Hours)</label>
                                            <input type="number" class="form-control" id="auto_sync_hours" 
                                                   name="auto_sync_hours" min="1" max="168" 
                                                   value="{{ $settings['auto_sync_hours'] }}">
                                            <div class="form-text">How often to sync with MPESA API (1-168 hours)</div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Notification Settings -->
                            <div class="setting-group">
                                <h5 class="mb-3">
                                    <i class="bx bx-bell me-2"></i>Notification Settings
                                </h5>
                                
                                <div class="row g-3">
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label for="notification_email" class="form-label">Notification Email</label>
                                            <input type="email" class="form-control" id="notification_email" 
                                                   name="notification_email" value="{{ $settings['notification_email'] ?? '' }}">
                                            <div class="form-text">Email to receive MPESA notifications</div>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label class="form-label">Notification Channels</label>
                                            <div class="form-check">
                                                <input class="form-check-input" type="checkbox" id="channel_email" 
                                                       name="notification_channels[]" value="email"
                                                       {{ in_array('email', $settings['notification_channels']) ? 'checked' : '' }}>
                                                <label class="form-check-label" for="channel_email">Email</label>
                                            </div>
                                            <div class="form-check">
                                                <input class="form-check-input" type="checkbox" id="channel_sms" 
                                                       name="notification_channels[]" value="sms"
                                                       {{ in_array('sms', $settings['notification_channels']) ? 'checked' : '' }}>
                                                <label class="form-check-label" for="channel_sms">SMS</label>
                                            </div>
                                            <div class="form-check">
                                                <input class="form-check-input" type="checkbox" id="channel_dashboard" 
                                                       name="notification_channels[]" value="dashboard"
                                                       {{ in_array('dashboard', $settings['notification_channels']) ? 'checked' : '' }}>
                                                <label class="form-check-label" for="channel_dashboard">Dashboard</label>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label for="daily_report_time" class="form-label">Daily Report Time</label>
                                            <input type="time" class="form-control" id="daily_report_time" 
                                                   name="daily_report_time" value="{{ $settings['daily_report_time'] ?? '09:00' }}">
                                            <div class="form-text">Time to send daily MPESA reports</div>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label for="reconciliation_days" class="form-label">Reconciliation Period (Days)</label>
                                            <input type="number" class="form-control" id="reconciliation_days" 
                                                   name="reconciliation_days" min="1" max="365" 
                                                   value="{{ $settings['reconciliation_days'] }}">
                                            <div class="form-text">Number of days to keep for reconciliation reports</div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Transaction Limits -->
                            <div class="setting-group">
                                <h5 class="mb-3">
                                    <i class="bx bx-money me-2"></i>Transaction Limits
                                </h5>
                                
                                <div class="row g-3">
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label for="min_amount" class="form-label">Minimum Amount (KES)</label>
                                                <input type="number" class="form-control" id="min_amount" 
                                                name="min_amount" min="10" max="150000" step="0.01" 
                                                value="{{ $settings['min_amount'] ?? 10 }}">
                                            <div class="form-text">Minimum transaction amount allowed</div>
                                        </div>
                                    </div>
                                    
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label for="max_amount" class="form-label">Maximum Amount (KES)</label>
                                            <input type="number" class="form-control" id="max_amount" 
                                                   name="max_amount" min="10" max="150000" step="0.01" 
                                                   value="{{ $settings['max_amount'] }}">
                                            <div class="form-text">Maximum transaction amount allowed</div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Retry Settings -->
                            <div class="setting-group">
                                <h5 class="mb-3">
                                    <i class="bx bx-reset me-2"></i>Retry Settings
                                </h5>
                                
                                <div class="row g-3">
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label for="max_retry_attempts" class="form-label">Max Retry Attempts</label>
                                            <input type="number" class="form-control" id="max_retry_attempts" 
                                                   name="max_retry_attempts" min="0" max="10" 
                                                   value="{{ $settings['max_retry_attempts'] }}">
                                            <div class="form-text">Maximum number of retry attempts for failed API calls</div>
                                        </div>
                                    </div>
                                    
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label for="retry_delay_minutes" class="form-label">Retry Delay (Minutes)</label>
                                            <input type="number" class="form-control" id="retry_delay_minutes" 
                                                   name="retry_delay_minutes" min="1" max="60" 
                                                   value="{{ $settings['retry_delay_minutes'] }}">
                                            <div class="form-text">Delay between retry attempts</div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Logging Settings -->
                            <div class="setting-group">
                                <h5 class="mb-3">
                                    <i class="bx bx-book me-2"></i>Logging Settings
                                </h5>
                                
                                <div class="row g-3">
                                    <div class="col-md-6">
                                        <div class="form-check form-switch mb-3">
                                            <input class="form-check-input" type="checkbox" id="enable_logging" 
                                                   name="enable_logging" value="1" 
                                                   {{ $settings['enable_logging'] ? 'checked' : '' }}>
                                            <label class="form-check-label" for="enable_logging">
                                                Enable Logging
                                            </label>
                                            <div class="form-text">Enable MPESA transaction logging</div>
                                        </div>
                                    </div>
                                    
                                    <div class="col-md-6">
                                        <div class="mb-3">
                                            <label for="log_level" class="form-label">Log Level</label>
                                            <select class="form-select" id="log_level" name="log_level">
                                                <option value="debug" {{ $settings['log_level'] == 'debug' ? 'selected' : '' }}>Debug</option>
                                                <option value="info" {{ $settings['log_level'] == 'info' ? 'selected' : '' }}>Info</option>
                                                <option value="warning" {{ $settings['log_level'] == 'warning' ? 'selected' : '' }}>Warning</option>
                                                <option value="error" {{ $settings['log_level'] == 'error' ? 'selected' : '' }}>Error</option>
                                            </select>
                                            <div class="form-text">Level of detail to log</div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Danger Zone -->
                            <div class="setting-group border border-danger">
                                <h5 class="mb-3 text-danger">
                                    <i class="bx bx-error-circle me-2"></i>Danger Zone
                                </h5>
                                
                                <div class="alert alert-danger">
                                    <h6 class="alert-heading mb-2"><i class="bx bx-info-circle me-2"></i>Warning</h6>
                                    <p class="mb-3">These actions cannot be undone. Proceed with caution.</p>
                                    
                                    <div class="row g-3">
                                        <div class="col-md-6">
                                            <button type="button" class="btn btn-outline-danger w-100" id="clearMpesaCache">
                                                <i class="bx bx-trash me-1"></i> Clear MPESA Cache
                                            </button>
                                            <div class="form-text mt-2">Clear all MPESA related caches</div>
                                        </div>
                                        
                                        <div class="col-md-6">
                                            <button type="button" class="btn btn-outline-danger w-100" id="resetMpesaSettings">
                                                <i class="bx bx-reset me-1"></i> Reset to Defaults
                                            </button>
                                            <div class="form-text mt-2">Reset all settings to default values</div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Save Button -->
                            <div class="row g-3 mt-4">
                                <div class="col-12">
                                    <div class="d-flex gap-2 justify-content-end">
                                        <a href="{{ route('payments.mpesa.dashboard') }}" class="btn btn-secondary px-4">
                                            <i class="bx bx-x me-1"></i> Cancel
                                        </a>
                                        <button type="submit" class="btn btn-primary px-4">
                                            <i class="bx bx-save me-1"></i> Save Settings
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div><!--end row-->
    </div>
</div>
<!--end page wrapper -->

<!-- Danger Zone Confirmation Modal -->
<div class="modal fade" id="dangerZoneModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header bg-danger text-white">
                <h5 class="modal-title"><i class="bx bx-error-circle me-2"></i>Warning</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="text-center mb-4">
                    <div class="widgets-icons bg-light-danger text-danger rounded-circle mx-auto mb-3" style="width: 80px; height: 80px; line-height: 80px;">
                        <i class="bx bx-error-alt font-35"></i>
                    </div>
                    <h5 id="dangerModalTitle"></h5>
                    <p id="dangerModalMessage" class="text-muted"></p>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-danger" id="confirmDangerAction">
                    <i class="bx bx-check me-1"></i> Confirm
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Clear MPESA Cache
        document.getElementById('clearMpesaCache').addEventListener('click', function() {
            showDangerModal(
                'Clear MPESA Cache',
                'This will clear all MPESA related caches. This may temporarily slow down the system while caches are rebuilt. Are you sure you want to proceed?',
                'clearCache'
            );
        });
        
        // Reset MPESA Settings
        document.getElementById('resetMpesaSettings').addEventListener('click', function() {
            showDangerModal(
                'Reset MPESA Settings',
                'This will reset all MPESA settings to their default values. This action cannot be undone. Are you sure you want to proceed?',
                'resetSettings'
            );
        });
        
        // Confirm Danger Action
        document.getElementById('confirmDangerAction').addEventListener('click', function() {
            const action = this.getAttribute('data-action');
            
            if (action === 'clearCache') {
                clearMpesaCache();
            } else if (action === 'resetSettings') {
                resetMpesaSettings();
            }
            
            // Hide modal
            const modal = bootstrap.Modal.getInstance(document.getElementById('dangerZoneModal'));
            modal.hide();
        });
        
        // Show danger modal
        function showDangerModal(title, message, action) {
            document.getElementById('dangerModalTitle').textContent = title;
            document.getElementById('dangerModalMessage').textContent = message;
            document.getElementById('confirmDangerAction').setAttribute('data-action', action);
            
            const modal = new bootstrap.Modal(document.getElementById('dangerZoneModal'));
            modal.show();
        }
        
        // Clear MPESA cache function
        function clearMpesaCache() {
            fetch('{{ route("mpesa.settings.clear-cache") }}', {                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                }
            })
            .then(response => response.json())
            .then(response => {
                if (response.success) {
                    showAlert('success', response.message);
                } else {
                    showAlert('danger', response.message || 'Failed to clear cache');
                }
            })
            .catch(error => {
                showAlert('danger', 'An error occurred while clearing cache');
            });
        }
        
        // Reset MPESA settings function
        function resetMpesaSettings() {
            fetch('{{ route("mpesa.settings.clear-cache") }}', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                }
            })
            .then(response => response.json())
            .then(response => {
                if (response.success) {
                    showAlert('success', response.message);
                    // Reload page after 2 seconds to show updated settings
                    setTimeout(() => {
                        location.reload();
                    }, 2000);
                } else {
                    showAlert('danger', response.message || 'Failed to reset settings');
                }
            })
            .catch(error => {
                showAlert('danger', 'An error occurred while resetting settings');
            });
        }
        
        // Helper function to show alerts
        function showAlert(type, message) {
            const alertHtml = `
                <div class="alert border-0 border-start border-5 border-${type} alert-dismissible fade show py-2">
                    <div class="d-flex align-items-center">
                        <div class="font-35 text-${type}"><i class="bx bx-${type === 'success' ? 'check-circle' : 'error-circle'}"></i></div>
                        <div class="ms-3">${message}</div>
                    </div>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            `;
            
            const container = document.querySelector('.page-content');
            container.insertAdjacentHTML('afterbegin', alertHtml);
            
            // Auto dismiss after 5 seconds
            setTimeout(() => {
                const alert = container.querySelector('.alert');
                if (alert) {
                    alert.remove();
                }
            }, 5000);
        }
    });
</script>
@endsection