@extends('layouts.app')

@section('title', 'Daily MPESA Report')

@section('content')
<div class="container-fluid">
    <!-- Page Header -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h1 class="h3 mb-0 text-gray-800">
            <i class="fas fa-calendar-day mr-2"></i> Daily MPESA Report
        </h1>
        <div class="btn-group">
            <a href="{{ route('payments.mpesa.reports.unmatched') }}" class="btn btn-outline-primary">
                <i class="fas fa-unlink"></i> Unmatched
            </a>
            <a href="{{ route('payments.mpesa.reports.reconciliation') }}" class="btn btn-outline-primary">
                <i class="fas fa-balance-scale"></i> Reconciliation
            </a>
            <button type="button" class="btn btn-outline-primary" onclick="window.print()">
                <i class="fas fa-print"></i> Print
            </button>
        </div>
    </div>

    <!-- Date Selector -->
    <div class="card shadow mb-4">
        <div class="card-body">
            <form action="{{ route('payments.mpesa.reports.daily') }}" method="GET" class="form-inline">
                <div class="form-group mr-3">
                    <label for="date" class="mr-2">Select Date:</label>
                    <input type="date" class="form-control" id="date" name="date" 
                           value="{{ $date }}" max="{{ date('Y-m-d') }}">
                </div>
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-search mr-2"></i> Load Report
                </button>
                @if($date != date('Y-m-d'))
                    <a href="{{ route('payments.mpesa.reports.daily') }}" class="btn btn-outline-secondary ml-2">
                        Today's Report
                    </a>
                @endif
            </form>
        </div>
    </div>

    <!-- Summary Cards -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-primary shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">
                                Total Transactions</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800">{{ $summary['total_count'] }}</div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-exchange-alt fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-success shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">
                                Total Amount</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800">Ksh {{ number_format($summary['total_amount'], 2) }}</div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-money-bill-wave fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-info shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-info text-uppercase mb-1">
                                Matched Transactions</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800">{{ $summary['matched_count'] }}</div>
                            <div class="mt-2 mb-0 text-muted text-xs">
                                <span>Ksh {{ number_format($summary['matched_amount'], 2) }}</span>
                            </div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-link fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-warning shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-warning text-uppercase mb-1">
                                Unmatched Transactions</div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800">{{ $summary['unmatched_count'] }}</div>
                            <div class="mt-2 mb-0 text-muted text-xs">
                                <span>Ksh {{ number_format($summary['unmatched_amount'], 2) }}</span>
                            </div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-unlink fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Additional Stats -->
    <div class="row mb-4">
        <div class="col-md-4">
            <div class="card shadow h-100">
                <div class="card-body">
                    <h6 class="font-weight-bold text-primary mb-3">
                        <i class="fas fa-chart-line mr-2"></i> Amount Statistics
                    </h6>
                    <div class="row">
                        <div class="col-6">
                            <small class="text-muted">Average Amount</small>
                            <div class="font-weight-bold">Ksh {{ number_format($summary['avg_amount'], 2) }}</div>
                        </div>
                        <div class="col-6">
                            <small class="text-muted">Maximum Amount</small>
                            <div class="font-weight-bold">Ksh {{ number_format($summary['max_amount'], 2) }}</div>
                        </div>
                    </div>
                    <div class="row mt-2">
                        <div class="col-6">
                            <small class="text-muted">Minimum Amount</small>
                            <div class="font-weight-bold">Ksh {{ number_format($summary['min_amount'], 2) }}</div>
                        </div>
                        <div class="col-6">
                            <small class="text-muted">Match Rate</small>
                            <div class="font-weight-bold">
                                {{ $summary['total_count'] > 0 ? number_format(($summary['matched_count'] / $summary['total_count']) * 100, 1) : 0 }}%
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-md-8">
            <div class="card shadow h-100">
                <div class="card-body">
                    <h6 class="font-weight-bold text-primary mb-3">
                        <i class="fas fa-clock mr-2"></i> Hourly Distribution
                    </h6>
                    <div class="chart-container" style="position: relative; height: 200px;">
                        <canvas id="hourlyChart"></canvas>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Transactions List -->
    <div class="card shadow mb-4">
        <div class="card-header py-3 d-flex justify-content-between align-items-center">
            <h6 class="m-0 font-weight-bold text-primary">
                Transactions for {{ \Carbon\Carbon::parse($date)->format('F d, Y') }}
                <span class="badge badge-primary ml-2">{{ $transactions->count() }} transactions</span>
            </h6>
            <div class="dropdown no-arrow">
                <a class="dropdown-toggle" href="#" role="button" id="dropdownMenuLink"
                    data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                    <i class="fas fa-ellipsis-v fa-sm fa-fw text-gray-400"></i>
                </a>
                <div class="dropdown-menu dropdown-menu-right shadow animated--fade-in"
                    aria-labelledby="dropdownMenuLink">
                    <a class="dropdown-item" href="#" id="exportDaily">
                        <i class="fas fa-download fa-sm fa-fw mr-2 text-gray-400"></i>
                        Export as CSV
                    </a>
                    <a class="dropdown-item" href="#" id="printReport">
                        <i class="fas fa-print fa-sm fa-fw mr-2 text-gray-400"></i>
                        Print Report
                    </a>
                </div>
            </div>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th>Time</th>
                            <th>Transaction ID</th>
                            <th>Phone</th>
                            <th>Customer</th>
                            <th>Amount</th>
                            <th>Status</th>
                            <th>Sale</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        @forelse($transactions as $transaction)
                            <tr>
                                <td>{{ $transaction->transaction_date->format('h:i A') }}</td>
                                <td>
                                    <code>{{ $transaction->transaction_id }}</code>
                                    @if($transaction->mpesa_receipt_number)
                                        <br><small class="text-muted">Receipt: {{ $transaction->mpesa_receipt_number }}</small>
                                    @endif
                                </td>
                                <td>
                                    <a href="{{ route('payments.mpesa.transactions.by-phone', $transaction->phone) }}" 
                                       class="text-primary">
                                        {{ $transaction->formatted_phone }}
                                    </a>
                                </td>
                                <td>
                                    @if($transaction->full_name)
                                        {{ $transaction->full_name }}
                                    @else
                                        <span class="text-muted">-</span>
                                    @endif
                                </td>
                                <td class="font-weight-bold">
                                    Ksh {{ number_format($transaction->amount, 2) }}
                                </td>
                                <td>
                                    @if($transaction->status == 'completed')
                                        <span class="badge badge-success">Completed</span>
                                    @elseif($transaction->status == 'pending')
                                        <span class="badge badge-warning">Pending</span>
                                    @elseif($transaction->status == 'failed')
                                        <span class="badge badge-danger">Failed</span>
                                    @endif
                                </td>
                                <td>
                                    @if($transaction->sale_id)
                                        <a href="{{ route('sales.show', $transaction->sale_id) }}" 
                                           class="badge badge-info">
                                            Sale #{{ $transaction->sale_id }}
                                        </a>
                                    @else
                                        <span class="badge badge-warning">Unmatched</span>
                                    @endif
                                </td>
                                <td>
                                    <a href="{{ route('payments.mpesa.transactions.show', $transaction->id) }}" 
                                       class="btn btn-sm btn-outline-primary">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                </td>
                            </tr>
                        @empty
                            <tr>
                                <td colspan="8" class="text-center py-4">
                                    <i class="fas fa-exchange-alt fa-3x text-gray-300 mb-3"></i>
                                    <h5 class="text-gray-700">No transactions for this date</h5>
                                    <p class="text-muted">No MPESA transactions were recorded on {{ \Carbon\Carbon::parse($date)->format('F d, Y') }}</p>
                                </td>
                            </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- Top Phones -->
    @if($topPhones->count() > 0)
    <div class="row">
        <div class="col-lg-6">
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">
                        <i class="fas fa-mobile-alt mr-2"></i> Top Phone Numbers
                    </h6>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-sm">
                            <thead>
                                <tr>
                                    <th>Phone</th>
                                    <th>Transactions</th>
                                    <th>Amount</th>
                                    <th>Matched</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @foreach($topPhones as $phone => $data)
                                    <tr>
                                        <td>
                                            {{ $phone }}
                                            @php
                                                $customer = \App\Models\Customer::where('phone', $phone)->first();
                                            @endphp
                                            @if($customer)
                                                <br><small class="text-success">{{ $customer->name }}</small>
                                            @endif
                                        </td>
                                        <td>{{ $data['count'] }}</td>
                                        <td class="font-weight-bold">Ksh {{ number_format($data['amount'], 2) }}</td>
                                        <td>
                                            <span class="badge badge-{{ $data['matched'] > 0 ? 'success' : 'warning' }}">
                                                {{ $data['matched'] }}/{{ $data['count'] }}
                                            </span>
                                        </td>
                                        <td>
                                            <a href="{{ route('payments.mpesa.transactions.by-phone', $phone) }}" 
                                               class="btn btn-sm btn-outline-primary">
                                                <i class="fas fa-list"></i>
                                            </a>
                                            @if($customer)
                                                <a href="{{ route('customers.show', $customer->id) }}" 
                                                   class="btn btn-sm btn-outline-info">
                                                    <i class="fas fa-user"></i>
                                                </a>
                                            @endif
                                        </td>
                                    </tr>
                                @endforeach
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-lg-6">
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">
                        <i class="fas fa-chart-pie mr-2"></i> Amount Distribution
                    </h6>
                </div>
                <div class="card-body">
                    <div class="chart-container" style="position: relative; height: 300px;">
                        <canvas id="amountDistributionChart"></canvas>
                    </div>
                </div>
            </div>
        </div>
    </div>
    @endif
</div>

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
    $(document).ready(function() {
        // Hourly Distribution Chart
        const hourlyCtx = document.getElementById('hourlyChart').getContext('2d');
        const hourlyChart = new Chart(hourlyCtx, {
            type: 'bar',
            data: {
                labels: @json(array_column($hourlyData, 'hour')),
                datasets: [{
                    label: 'Transaction Count',
                    data: @json(array_column($hourlyData, 'count')),
                    backgroundColor: 'rgba(54, 162, 235, 0.5)',
                    borderColor: 'rgba(54, 162, 235, 1)',
                    borderWidth: 1,
                    yAxisID: 'y'
                }, {
                    label: 'Amount (Ksh)',
                    data: @json(array_column($hourlyData, 'amount')),
                    backgroundColor: 'rgba(255, 99, 132, 0.5)',
                    borderColor: 'rgba(255, 99, 132, 1)',
                    borderWidth: 1,
                    type: 'line',
                    yAxisID: 'y1'
                }]
            },
            options: {
                responsive: true,
                scales: {
                    y: {
                        type: 'linear',
                        display: true,
                        position: 'left',
                        title: {
                            display: true,
                            text: 'Transaction Count'
                        }
                    },
                    y1: {
                        type: 'linear',
                        display: true,
                        position: 'right',
                        title: {
                            display: true,
                            text: 'Amount (Ksh)'
                        },
                        grid: {
                            drawOnChartArea: false
                        }
                    }
                }
            }
        });

        // Amount Distribution Chart
        @if($topPhones->count() > 0)
        const distributionCtx = document.getElementById('amountDistributionChart').getContext('2d');
        const distributionChart = new Chart(distributionCtx, {
            type: 'doughnut',
            data: {
                labels: @json(array_keys($topPhones->toArray())),
                datasets: [{
                    data: @json(array_column($topPhones->toArray(), 'amount')),
                    backgroundColor: [
                        '#FF6384', '#36A2EB', '#FFCE56', '#4BC0C0', 
                        '#9966FF', '#FF9F40', '#8AC926', '#1982C4',
                        '#6A4C93', '#FF595E'
                    ]
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        position: 'right',
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                let label = context.label || '';
                                if (label) {
                                    label += ': ';
                                }
                                label += 'Ksh ' + context.parsed.toLocaleString();
                                return label;
                            }
                        }
                    }
                }
            }
        });
        @endif

        // Export daily report
        $('#exportDaily').click(function(e) {
            e.preventDefault();
            window.location.href = '{{ route("payments.mpesa.transactions.export") }}?' + 
                'format=csv&start_date={{ $date }}&end_date={{ $date }}';
        });

        // Print report
        $('#printReport').click(function(e) {
            e.preventDefault();
            window.print();
        });
    });
</script>

<style>
    @media print {
        .btn, .form-inline, .dropdown, .no-print {
            display: none !important;
        }
        
        .card {
            border: none !important;
            box-shadow: none !important;
        }
        
        .container-fluid {
            padding: 0 !important;
        }
        
        .card-body {
            padding: 0 !important;
        }
        
        .table th, .table td {
            padding: 0.5rem !important;
        }
    }
</style>
@endpush
@endsection