@extends('layouts.app')

@section('title', 'MPESA Configuration')

@section('style')
<style>
    .credential-card {
        border-left: 4px solid #dee2e6;
        transition: all 0.3s;
    }
    .credential-card.active {
        border-left-color: #28a745;
    }
    .credential-card:hover {
        box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    }
    .masked-text {
        background-color: #f8f9fa;
        padding: 0.2rem 0.5rem;
        border-radius: 4px;
        font-family: monospace;
    }
</style>
@endsection

@section('content')
<!--start page wrapper -->
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">Payments</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('payments.mpesa.dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item active" aria-current="page">MPESA Configuration</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <div class="btn-group">
                    <a href="{{ route('payments.mpesa.dashboard') }}" class="btn btn-outline-primary">
                        <i class="bx bx-tachometer me-1"></i> Dashboard
                    </a>
                    <a href="{{ route('payments.mpesa.test') }}" class="btn btn-outline-primary">
                        <i class="bx bx-test-tube me-1"></i> Test
                    </a>
                </div>
            </div>
        </div>
        <!--end breadcrumb-->

        <!-- Active Credential Alert -->
        @if($activeCredential)
        <div class="alert border-0 border-start border-5 border-success alert-dismissible fade show py-2">
            <div class="d-flex align-items-center">
                <div class="font-35 text-success"><i class="bx bxs-check-circle"></i></div>
                <div class="ms-3">
                    <h6 class="mb-0 text-success">Active Credential</h6>
                    <div>{{ $activeCredential->name }} ({{ $activeCredential->environment_display }})</div>
                    @if($activeCredential->last_success)
                    <small class="text-muted">
                        <i class="bx bx-time"></i> Last successful connection: {{ $activeCredential->last_success->diffForHumans() }}
                    </small>
                    @endif
                </div>
            </div>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
        @else
        <div class="alert border-0 border-start border-5 border-warning alert-dismissible fade show py-2">
            <div class="d-flex align-items-center">
                <div class="font-35 text-warning"><i class="bx bxs-error-circle"></i></div>
                <div class="ms-3">
                    <h6 class="mb-0 text-warning">Warning: No Active Credentials</h6>
                    <div>MPESA payments will not work until you activate credentials.</div>
                </div>
            </div>
            <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
        </div>
        @endif

        <div class="row">
            <!-- Credentials List -->
            <div class="col-12 col-lg-5">
                <div class="card radius-10 mb-3">
                    <div class="card-body">
                        <div class="d-flex align-items-center mb-3">
                            <div>
                                <h6 class="mb-0">Configured Credentials</h6>
                            </div>
                            <div class="ms-auto">
                                <button type="button" class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#addCredentialModal">
                                    <i class="bx bx-plus"></i> Add New
                                </button>
                            </div>
                        </div>
                        
                        @if($credentials->count() > 0)
                            <div class="list-group">
                                @foreach($credentials as $credential)
                                    <div class="list-group-item list-group-item-action credential-card {{ $credential->is_active ? 'active' : '' }}">
                                        <div class="d-flex w-100 justify-content-between align-items-center">
                                            <div class="flex-grow-1">
                                                <h6 class="mb-1">
                                                    {{ $credential->name }}
                                                    @if($credential->is_active)
                                                        <span class="badge bg-success ms-2">Active</span>
                                                    @endif
                                                </h6>
                                                <div class="d-flex align-items-center flex-wrap">
                                                    <span class="masked-text me-3 mb-1">
                                                        <i class="bx bx-hash"></i> {{ $credential->shortcode_masked }}
                                                    </span>
                                                    <span class="badge bg-light text-dark me-3 mb-1">
                                                        <i class="bx bx-globe"></i> {{ $credential->environment_display }}
                                                    </span>
                                                    @if($credential->urls_registered_at)
                                                        <span class="badge bg-success mb-1">
                                                            <i class="bx bx-check"></i> URLs Registered
                                                        </span>
                                                    @endif
                                                </div>
                                                
                                                @if($credential->notes)
                                                    <p class="mb-1 mt-2 small text-muted">{{ $credential->notes }}</p>
                                                @endif
                                                
                                                <div class="d-flex align-items-center mt-2">
                                                    @if($credential->last_success)
                                                        <small class="text-success me-3">
                                                            <i class="bx bx-check-circle"></i> 
                                                            Last success: {{ $credential->last_success->diffForHumans() }}
                                                        </small>
                                                    @endif
                                                    @if($credential->last_failure)
                                                        <small class="text-danger">
                                                            <i class="bx bx-error-circle"></i> 
                                                            Last failure: {{ $credential->last_failure->diffForHumans() }}
                                                        </small>
                                                    @endif
                                                </div>
                                            </div>
                                            <div class="btn-group btn-group-sm ms-3">
                                                <button class="btn btn-outline-primary edit-credential" 
                                                        data-id="{{ $credential->id }}"
                                                        data-bs-toggle="tooltip" title="Edit">
                                                    <i class="bx bx-edit"></i>
                                                </button>
                                                @if(!$credential->is_active)
                                                    <button class="btn btn-outline-success activate-credential"
                                                            data-id="{{ $credential->id }}"
                                                            data-bs-toggle="tooltip" title="Activate">
                                                        <i class="bx bx-toggle-right"></i>
                                                    </button>
                                                @endif
                                                @if($credentials->count() > 1)
                                                    <button class="btn btn-outline-danger delete-credential"
                                                            data-id="{{ $credential->id }}"
                                                            data-name="{{ $credential->name }}"
                                                            data-bs-toggle="tooltip" title="Delete">
                                                        <i class="bx bx-trash"></i>
                                                    </button>
                                                @endif
                                            </div>
                                        </div>
                                    </div>
                                @endforeach
                            </div>
                        @else
                            <div class="text-center py-5">
                                <div class="widgets-icons bg-light-secondary text-secondary rounded-circle mx-auto mb-3">
                                    <i class="bx bx-key"></i>
                                </div>
                                <h5 class="text-gray-700 mb-3">No credentials configured</h5>
                                <p class="text-muted mb-4">Add your first MPESA credentials to get started</p>
                                <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addCredentialModal">
                                    <i class="bx bx-plus me-1"></i> Add Credentials
                                </button>
                            </div>
                        @endif
                    </div>
                </div>

                <!-- Information Card -->
                <div class="card radius-10">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <h6 class="mb-0"><i class="bx bx-info-circle me-2"></i>Important Notes</h6>
                            </div>
                        </div>
                        <hr>
                        <div class="alert bg-light-info border-0">
                            <ul class="mb-0 ps-3">
                                <li class="mb-2">Only one credential can be active at a time</li>
                                <li class="mb-2">Make sure callback URLs are publicly accessible</li>
                                <li class="mb-2">Test credentials in sandbox environment first</li>
                                <li class="mb-2">Keep your credentials secure and never share them</li>
                                <li>URLs must be registered for C2B payments to work</li>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Configuration Form -->
            <div class="col-12 col-lg-7">
                <div class="card radius-10">
                    <div class="card-body">
                        <div class="d-flex align-items-center mb-3">
                            <div>
                                <h6 class="mb-0" id="formTitle">
                                    <i class="bx bx-plus me-2"></i> Add New Credentials
                                </h6>
                            </div>
                        </div>
                        <hr>
                        
                        <form id="credentialForm" method="POST" action="{{ route('payments.mpesa.configuration.update') }}">
                            @csrf
                            <input type="hidden" id="credential_id" name="id">

                            <div class="row g-3">
                                <div class="col-md-6">
                                    <label for="name" class="form-label">Credential Name <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="name" name="name" 
                                           value="{{ $defaults['name'] }}" required>
                                    <div class="form-text">e.g., "Production", "Sandbox Test"</div>
                                </div>
                                <div class="col-md-6">
                                    <label for="business_shortcode" class="form-label">Business Shortcode <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="business_shortcode" 
                                           name="business_shortcode" value="{{ $defaults['business_shortcode'] }}" required>
                                    <div class="form-text">Your MPESA Paybill/Till number</div>
                                </div>
                            </div>

                            <div class="row g-3 mt-2">
                                <div class="col-md-6">
                                    <label for="consumer_key" class="form-label">Consumer Key <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="consumer_key" name="consumer_key" required>
                                    <div class="form-text">From Safaricom Daraja portal</div>
                                </div>
                                <div class="col-md-6">
                                    <label for="consumer_secret" class="form-label">Consumer Secret <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="consumer_secret" name="consumer_secret" required>
                                    <div class="form-text">From Safaricom Daraja portal</div>
                                </div>
                            </div>

                            <div class="row g-3 mt-2">
                                <div class="col-12">
                                    <label for="passkey" class="form-label">Passkey <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="passkey" name="passkey" required>
                                    <div class="form-text">Your MPESA passkey for STK Push</div>
                                </div>
                            </div>

                            <div class="row g-3 mt-2">
                                <div class="col-md-6">
                                    <label for="initiator_name" class="form-label">Initiator Name</label>
                                    <input type="text" class="form-control" id="initiator_name" name="initiator_name">
                                    <div class="form-text">For B2C and reversal transactions</div>
                                </div>
                                <div class="col-md-6">
                                    <label for="initiator_password" class="form-label">Initiator Password</label>
                                    <input type="password" class="form-control" id="initiator_password" name="initiator_password">
                                    <div class="form-text">Password for initiator</div>
                                </div>
                            </div>

                            <div class="row g-3 mt-2">
                                <div class="col-12">
                                    <label for="security_credential" class="form-label">Security Credential</label>
                                    <textarea class="form-control" id="security_credential" name="security_credential" 
                                              rows="2"></textarea>
                                    <div class="form-text">Encrypted security credential</div>
                                </div>
                            </div>

                            <div class="row g-3 mt-2">
                                <div class="col-md-6">
                                    <label for="environment" class="form-label">Environment <span class="text-danger">*</span></label>
                                    <select class="form-select" id="environment" name="environment" required>
                                        <option value="sandbox" {{ $defaults['environment'] == 'sandbox' ? 'selected' : '' }}>Sandbox (Testing)</option>
                                        <option value="production" {{ $defaults['environment'] == 'production' ? 'selected' : '' }}>Production (Live)</option>
                                    </select>
                                </div>
                                <div class="col-md-6">
                                    <div class="form-check mt-4">
                                        <input class="form-check-input" type="checkbox" id="is_active" name="is_active" 
                                               value="1" {{ $defaults['is_active'] ? 'checked' : '' }}>
                                        <label class="form-check-label" for="is_active">
                                            Set as active credentials
                                        </label>
                                        <div class="form-text">Only one credential can be active at a time</div>
                                    </div>
                                </div>
                            </div>

                            <div class="row g-3 mt-2">
                                <div class="col-12">
                                    <label for="callback_url" class="form-label">Callback URL <span class="text-danger">*</span></label>
                                    <input type="url" class="form-control" id="callback_url" name="callback_url" 
                                           value="{{ $defaults['callback_url'] }}" required>
                                    <div class="form-text">For STK Push and C2B callbacks</div>
                                </div>
                            </div>

                            <div class="row g-3 mt-2">
                                <div class="col-md-6">
                                    <label for="validation_url" class="form-label">Validation URL</label>
                                    <input type="url" class="form-control" id="validation_url" name="validation_url"
                                           value="{{ url('/api/mpesa/validation') }}">
                                    <div class="form-text">For C2B validation (optional)</div>
                                </div>
                                <div class="col-md-6">
                                    <label for="confirmation_url" class="form-label">Confirmation URL</label>
                                    <input type="url" class="form-control" id="confirmation_url" name="confirmation_url"
                                           value="{{ url('/api/mpesa/confirmation') }}">
                                    <div class="form-text">For C2B confirmation (optional)</div>
                                </div>
                            </div>

                            <div class="row g-3 mt-2">
                                <div class="col-12">
                                    <label for="notes" class="form-label">Notes</label>
                                    <textarea class="form-control" id="notes" name="notes" rows="3"></textarea>
                                    <div class="form-text">Any additional notes about these credentials</div>
                                </div>
                            </div>

                            <div class="row g-3 mt-3">
                                <div class="col-12">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="test_connection" name="test_connection" value="1">
                                        <label class="form-check-label" for="test_connection">
                                            Test connection after saving
                                        </label>
                                    </div>
                                </div>
                            </div>

                            <div class="row g-3 mt-4">
                                <div class="col-12">
                                    <div class="d-flex gap-2">
                                        <button type="submit" class="btn btn-primary px-4">
                                            <i class="bx bx-save me-1"></i> Save Credentials
                                        </button>
                                        <button type="button" class="btn btn-secondary" id="resetForm">
                                            <i class="bx bx-reset me-1"></i> Reset
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Connection Test Results -->
                <div class="card radius-10 mt-3 d-none" id="testResultsCard">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <h6 class="mb-0">
                                    <i class="bx bx-test-tube me-2"></i> Connection Test Results
                                </h6>
                            </div>
                        </div>
                        <hr>
                        <div id="testResults"></div>
                    </div>
                </div>
            </div>
        </div><!--end row-->
    </div>
</div>
<!--end page wrapper -->

<!-- Test Connection Modal -->
<div class="modal fade" id="testConnectionModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Test Connection</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="mb-3">
                    <label for="test_consumer_key" class="form-label">Consumer Key</label>
                    <input type="text" class="form-control" id="test_consumer_key">
                </div>
                <div class="mb-3">
                    <label for="test_consumer_secret" class="form-label">Consumer Secret</label>
                    <input type="text" class="form-control" id="test_consumer_secret">
                </div>
                <div class="mb-3">
                    <label for="test_environment" class="form-label">Environment</label>
                    <select class="form-select" id="test_environment">
                        <option value="sandbox">Sandbox</option>
                        <option value="production">Production</option>
                    </select>
                </div>
                <div class="alert bg-light-info border-0">
                    <i class="bx bx-info-circle me-2"></i>
                    This will test if the credentials can obtain an access token from MPESA API.
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" id="runConnectionTest">
                    <i class="bx bx-play me-1"></i> Test Connection
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Reset form
        document.getElementById('resetForm').addEventListener('click', function() {
            document.getElementById('credentialForm').reset();
            document.getElementById('credential_id').value = '';
            document.getElementById('formTitle').innerHTML = '<i class="bx bx-plus me-2"></i> Add New Credentials';
        });

        // Edit credential
        document.querySelectorAll('.edit-credential').forEach(button => {
            button.addEventListener('click', function() {
                const id = this.getAttribute('data-id');
                
                fetch('{{ route("mpesa.configuration.get") }}?id=' + id)
                    .then(response => response.json())
                    .then(response => {
                        if (response.success) {
                            const credential = response.credential;
                            
                            // Fill form
                            document.getElementById('credential_id').value = credential.id;
                            document.getElementById('name').value = credential.name;
                            document.getElementById('business_shortcode').value = credential.business_shortcode;
                            document.getElementById('consumer_key').value = '';
                            document.getElementById('consumer_secret').value = '';
                            document.getElementById('passkey').value = '';
                            document.getElementById('initiator_name').value = credential.initiator_name || '';
                            document.getElementById('initiator_password').value = '';
                            document.getElementById('security_credential').value = credential.security_credential || '';
                            document.getElementById('environment').value = credential.environment;
                            document.getElementById('is_active').checked = credential.is_active;
                            document.getElementById('callback_url').value = credential.callback_url;
                            document.getElementById('validation_url').value = credential.validation_url || '';
                            document.getElementById('confirmation_url').value = credential.confirmation_url || '';
                            document.getElementById('notes').value = credential.notes || '';
                            
                            document.getElementById('formTitle').innerHTML = '<i class="bx bx-edit me-2"></i> Edit Credentials';
                            
                            // Scroll to form
                            document.getElementById('credentialForm').scrollIntoView({ behavior: 'smooth' });
                        }
                    });
            });
        });

        // Activate credential
        document.querySelectorAll('.activate-credential').forEach(button => {
            button.addEventListener('click', function() {
                const id = this.getAttribute('data-id');
                
                if (confirm('Are you sure you want to activate these credentials?')) {
                    fetch('{{ route("mpesa.configuration.activate") }}', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-CSRF-TOKEN': '{{ csrf_token() }}'
                        },
                        body: JSON.stringify({ id: id })
                    })
                    .then(response => response.json())
                    .then(response => {
                        if (response.success) {
                            location.reload();
                        } else {
                            alert(response.message || 'Failed to activate credential');
                        }
                    });
                }
            });
        });

        // Delete credential
        document.querySelectorAll('.delete-credential').forEach(button => {
            button.addEventListener('click', function() {
                const id = this.getAttribute('data-id');
                const name = this.getAttribute('data-name');
                
                if (confirm(`Are you sure you want to delete "${name}"? This action cannot be undone.`)) {
                    fetch('{{ route("mpesa.configuration.delete") }}', {
                        method: 'DELETE',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-CSRF-TOKEN': '{{ csrf_token() }}'
                        },
                        body: JSON.stringify({ id: id })
                    })
                    .then(response => response.json())
                    .then(response => {
                        if (response.success) {
                            location.reload();
                        } else {
                            alert(response.message || 'Failed to delete credential');
                        }
                    });
                }
            });
        });

        // Submit form
        document.getElementById('credentialForm').addEventListener('submit', function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            const submitBtn = this.querySelector('button[type="submit"]');
            const originalText = submitBtn.innerHTML;
            
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="bx bx-loader bx-spin me-1"></i>Saving...';
            
            fetch(this.action, {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(response => {
                if (response.success) {
                    // Show success message
                    showAlert('success', response.message);
                    
                    // Show test results if available
                    if (response.test_result) {
                        document.getElementById('testResultsCard').classList.remove('d-none');
                        const result = response.test_result;
                        
                        let html = `<div class="alert alert-${result.success ? 'success' : 'danger'}">`;
                        html += `<h6><i class="bx bx-${result.success ? 'check-circle' : 'error-circle'} me-2"></i>`;
                        html += `${result.success ? 'Connection Successful' : 'Connection Failed'}</h6>`;
                        html += `<p>${result.message}</p>`;
                        
                        if (result.data) {
                            html += `<div class="mt-3"><strong>Details:</strong>`;
                            html += `<pre class="bg-light p-2 mt-2 small">`;
                            html += `Access Token: ${result.data.access_token}\n`;
                            html += `Expires In: ${result.data.expires_in} seconds\n`;
                            html += `Environment: ${result.data.environment}\n`;
                            html += `Timestamp: ${result.data.timestamp}`;
                            html += `</pre></div>`;
                        }
                        
                        if (result.error_details) {
                            html += `<div class="mt-3"><strong>Error Details:</strong>`;
                            html += `<pre class="bg-light p-2 mt-2 small">${result.error_details}</pre></div>`;
                        }
                        
                        html += `</div>`;
                        document.getElementById('testResults').innerHTML = html;
                    }
                    
                    // Reload page after 2 seconds
                    setTimeout(() => {
                        location.reload();
                    }, 2000);
                } else {
                    // Show error message
                    if (response.errors) {
                        let errorHtml = '<div class="alert alert-danger"><ul class="mb-0">';
                        for (const [key, errors] of Object.entries(response.errors)) {
                            errors.forEach(error => {
                                errorHtml += `<li>${error}</li>`;
                            });
                        }
                        errorHtml += '</ul></div>';
                        showAlert('danger', errorHtml);
                    } else {
                        showAlert('danger', response.message || 'Failed to save credentials');
                    }
                    submitBtn.disabled = false;
                    submitBtn.innerHTML = originalText;
                }
            })
            .catch(error => {
                showAlert('danger', 'An error occurred while saving credentials');
                submitBtn.disabled = false;
                submitBtn.innerHTML = originalText;
            });
        });

        // Standalone connection test
        if (document.getElementById('runConnectionTest')) {
            document.getElementById('runConnectionTest').addEventListener('click', function() {
                const btn = this;
                const originalText = btn.innerHTML;
                
                btn.disabled = true;
                btn.innerHTML = '<i class="bx bx-loader bx-spin me-1"></i>Testing...';
                
                fetch('{{ route("payments.mpesa.test-connection") }}', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    },
                    body: JSON.stringify({
                        consumer_key: document.getElementById('test_consumer_key').value,
                        consumer_secret: document.getElementById('test_consumer_secret').value,
                        environment: document.getElementById('test_environment').value
                    })
                })
                .then(response => response.json())
                .then(response => {
                    document.getElementById('testResultsCard').classList.remove('d-none');
                    
                    let html = `<div class="alert alert-${response.success ? 'success' : 'danger'}">`;
                    html += `<h6><i class="bx bx-${response.success ? 'check-circle' : 'error-circle'} me-2"></i>`;
                    html += `${response.success ? 'Connection Successful' : 'Connection Failed'}</h6>`;
                    html += `<p>${response.message}</p>`;
                    
                    if (response.data) {
                        html += `<div class="mt-3"><strong>Details:</strong>`;
                        html += `<pre class="bg-light p-2 mt-2 small">`;
                        html += `Access Token: ${response.data.access_token}\n`;
                        html += `Expires In: ${response.data.expires_in} seconds\n`;
                        html += `Environment: ${response.data.environment}\n`;
                        html += `Timestamp: ${response.data.timestamp}`;
                        html += `</pre></div>`;
                    }
                    
                    if (response.error_details) {
                        html += `<div class="mt-3"><strong>Error Details:</strong>`;
                        html += `<pre class="bg-light p-2 mt-2 small">${response.error_details}</pre></div>`;
                    }
                    
                    html += `</div>`;
                    document.getElementById('testResults').innerHTML = html;
                    
                    btn.disabled = false;
                    btn.innerHTML = originalText;
                    
                    // Hide modal
                    const modal = bootstrap.Modal.getInstance(document.getElementById('testConnectionModal'));
                    modal.hide();
                })
                .catch(error => {
                    showAlert('danger', 'Failed to run connection test');
                    btn.disabled = false;
                    btn.innerHTML = originalText;
                });
            });
        }

        // Helper function to show alerts
        function showAlert(type, message) {
            const alertHtml = `
                <div class="alert border-0 border-start border-5 border-${type} alert-dismissible fade show py-2">
                    <div class="d-flex align-items-center">
                        <div class="font-35 text-${type}"><i class="bx bx-${type === 'success' ? 'check-circle' : 'error-circle'}"></i></div>
                        <div class="ms-3">${message}</div>
                    </div>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            `;
            
            const container = document.querySelector('.page-content');
            container.insertAdjacentHTML('afterbegin', alertHtml);
            
            // Auto dismiss after 5 seconds
            setTimeout(() => {
                const alert = container.querySelector('.alert');
                if (alert) {
                    alert.remove();
                }
            }, 5000);
        }

        // Initialize tooltips
        const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
        tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl);
        });
    });
</script>
@endsection