@extends('layouts.app')

@section('style')
    <style>
        .form-section {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        .form-section h6 {
            color: #495057;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 2px solid #dee2e6;
        }
        .required-field::after {
            content: " *";
            color: #dc3545;
        }
        .customer-avatar {
            width: 80px;
            height: 80px;
            background: linear-gradient(45deg, #667eea 0%, #764ba2 100%);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 30px;
            font-weight: bold;
            margin: 0 auto 20px;
        }
        .customer-type-badge {
            font-size: 0.75rem;
            padding: 0.25rem 0.75rem;
        }
        .stat-badge {
            background: #e9ecef;
            color: #495057;
            padding: 0.5rem 1rem;
            border-radius: 5px;
            margin: 0.25rem;
            display: inline-block;
        }
        .vat-toggle {
            display: flex;
            align-items: center;
            gap: 10px;
        }
    </style>
@endsection

@section('content')
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">Customers</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item"><a href="{{ route('customers.index') }}">Customers</a></li>
                        <li class="breadcrumb-item"><a href="{{ route('customers.show', $customer) }}">{{ $customer->name }}</a></li>
                        <li class="breadcrumb-item active" aria-current="page">Edit Customer</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <div class="btn-group">
                    <a href="{{ route('customers.show', $customer) }}" class="btn btn-outline-info">
                        <i class="bx bx-show"></i> View Details
                    </a>
                    <a href="{{ route('customers.index') }}" class="btn btn-outline-secondary">
                        <i class="bx bx-arrow-back"></i> Back to List
                    </a>
                </div>
            </div>
        </div>
        <!--end breadcrumb-->

        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-body">
                        <div class="text-center mb-4">
                            <div class="customer-avatar">
                                {{ strtoupper(substr($customer->name, 0, 1)) }}
                            </div>
                            <h5 class="card-title mb-1">Edit Customer: {{ $customer->name }}</h5>
                            <p class="text-muted">Customer ID: {{ $customer->customer_code }}</p>
                            <div class="mt-2">
                                @if($customer->customer_type == 'walk-in')
                                    <span class="badge bg-warning text-dark customer-type-badge">
                                        <i class="bx bx-walk"></i> Walk-in Customer
                                    </span>
                                @elseif($customer->customer_type == 'regular')
                                    <span class="badge bg-success customer-type-badge">
                                        <i class="bx bx-user"></i> Regular Customer
                                    </span>
                                @else
                                    <span class="badge bg-primary customer-type-badge">
                                        <i class="bx bx-building"></i> Business Customer
                                    </span>
                                @endif
                            </div>
                        </div>
                        
                        @if($errors->any())
                            <div class="alert alert-danger border-0 bg-danger alert-dismissible fade show py-2">
                                <div class="d-flex align-items-center">
                                    <div class="font-35 text-white">
                                        <i class="bx bxs-error"></i>
                                    </div>
                                    <div class="ms-3">
                                        <h6 class="mb-0 text-white">Please fix the following errors:</h6>
                                        <ul class="text-white mb-0">
                                            @foreach($errors->all() as $error)
                                                <li>{{ $error }}</li>
                                            @endforeach
                                        </ul>
                                    </div>
                                </div>
                                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                            </div>
                        @endif

                        @if(session('success'))
                            <div class="alert alert-success border-0 bg-success alert-dismissible fade show py-2">
                                <div class="d-flex align-items-center">
                                    <div class="font-35 text-white">
                                        <i class="bx bxs-check-circle"></i>
                                    </div>
                                    <div class="ms-3">
                                        <h6 class="mb-0 text-white">Success</h6>
                                        <div class="text-white">{{ session('success') }}</div>
                                    </div>
                                </div>
                                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                            </div>
                        @endif

                        <form method="POST" action="{{ route('customers.update', $customer) }}" id="customerForm">
                            @csrf
                            @method('PUT')
                            
                            <!-- Basic Information -->
                            <div class="form-section">
                                <h6><i class="bx bx-info-circle"></i> Basic Information</h6>
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label for="name" class="form-label required-field">Full Name</label>
                                        <input type="text" class="form-control @error('name') is-invalid @enderror" 
                                               id="name" name="name" value="{{ old('name', $customer->name) }}" 
                                               placeholder="Enter customer name" required>
                                        @error('name')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label for="customer_type" class="form-label">Customer Type</label>
                                        <select class="form-select @error('customer_type') is-invalid @enderror" 
                                                id="customer_type" name="customer_type">
                                            <option value="regular" {{ old('customer_type', $customer->customer_type) == 'regular' ? 'selected' : '' }}>Regular Customer</option>
                                            <option value="business" {{ old('customer_type', $customer->customer_type) == 'business' ? 'selected' : '' }}>Business Customer</option>
                                            <option value="walk-in" {{ old('customer_type', $customer->customer_type) == 'walk-in' ? 'selected' : '' }}>Walk-in Customer</option>
                                        </select>
                                        @error('customer_type')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label for="company_name" class="form-label">Company Name</label>
                                        <input type="text" class="form-control @error('company_name') is-invalid @enderror" 
                                               id="company_name" name="company_name" value="{{ old('company_name', $customer->company_name) }}"
                                               placeholder="Enter company name">
                                        @error('company_name')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label for="contact_person" class="form-label">Contact Person</label>
                                        <input type="text" class="form-control @error('contact_person') is-invalid @enderror" 
                                               id="contact_person" name="contact_person" value="{{ old('contact_person', $customer->contact_person) }}"
                                               placeholder="Primary contact person">
                                        @error('contact_person')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                            </div>

                            <!-- Contact Information -->
                            <div class="form-section">
                                <h6><i class="bx bx-phone"></i> Contact Information</h6>
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label for="email" class="form-label">Email Address</label>
                                        <input type="email" class="form-control @error('email') is-invalid @enderror" 
                                               id="email" name="email" value="{{ old('email', $customer->email) }}"
                                               placeholder="customer@example.com">
                                        @error('email')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label for="phone" class="form-label required-field">Phone Number</label>
                                        <input type="text" class="form-control @error('phone') is-invalid @enderror" 
                                               id="phone" name="phone" value="{{ old('phone', $customer->phone) }}"
                                               placeholder="+254 7XX XXX XXX" required>
                                        @error('phone')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                    
                                    <div class="col-12 mb-3">
                                        <label for="address" class="form-label">Address</label>
                                        <textarea class="form-control @error('address') is-invalid @enderror" 
                                                  id="address" name="address" rows="3" 
                                                  placeholder="Full address">{{ old('address', $customer->address) }}</textarea>
                                        @error('address')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                            </div>

                            <!-- VAT Information -->
                            <div class="form-section">
                                <h6><i class="bx bx-receipt"></i> VAT Information</h6>
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <div class="vat-toggle mb-3">
                                            <div class="form-check form-switch">
                                                <input class="form-check-input" type="checkbox" 
                                                       id="is_vat_registered" name="is_vat_registered"
                                                       value="1" {{ old('is_vat_registered', $customer->is_vat_registered) ? 'checked' : '' }}>
                                                <label class="form-check-label" for="is_vat_registered">
                                                    <strong>VAT Registered Customer</strong>
                                                </label>
                                            </div>
                                        </div>
                                        <div class="form-text">Enable if customer is VAT registered</div>
                                    </div>
                                    
                                    <div class="col-md-6 mb-3 vat-field">
                                        <label for="vat_number" class="form-label">VAT Number</label>
                                        <input type="text" class="form-control @error('vat_number') is-invalid @enderror" 
                                               id="vat_number" name="vat_number" value="{{ old('vat_number', $customer->vat_number) }}"
                                               placeholder="VAT-123456789">
                                        <div class="form-text">Format: VAT- followed by numbers</div>
                                        @error('vat_number')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                            </div>

                            <!-- Business Information -->
                            <div class="form-section">
                                <h6><i class="bx bx-briefcase"></i> Business Information</h6>
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label for="website" class="form-label">Website</label>
                                        <input type="url" class="form-control @error('website') is-invalid @enderror" 
                                               id="website" name="website" value="{{ old('website', $customer->website) }}"
                                               placeholder="https://example.com">
                                        @error('website')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label for="tax_id" class="form-label">Tax ID</label>
                                        <input type="text" class="form-control @error('tax_id') is-invalid @enderror" 
                                               id="tax_id" name="tax_id" value="{{ old('tax_id', $customer->tax_id) }}"
                                               placeholder="TAX-123456789">
                                        @error('tax_id')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                            </div>

                            <!-- Account Settings -->
                            <div class="form-section">
                                <h6><i class="bx bx-cog"></i> Account Settings</h6>
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label for="status" class="form-label">Status</label>
                                        <select class="form-select @error('status') is-invalid @enderror" 
                                                id="status" name="status">
                                            <option value="active" {{ old('status', $customer->status) == 'active' ? 'selected' : '' }}>Active</option>
                                            <option value="inactive" {{ old('status', $customer->status) == 'inactive' ? 'selected' : '' }}>Inactive</option>
                                        </select>
                                        @error('status')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label for="credit_limit" class="form-label">Credit Limit</label>
                                        <div class="input-group">
                                            <span class="input-group-text">KES</span>
                                            <input type="number" class="form-control @error('credit_limit') is-invalid @enderror" 
                                                   id="credit_limit" name="credit_limit" value="{{ old('credit_limit', $customer->credit_limit) }}"
                                                   placeholder="0.00" step="0.01" min="0">
                                            @error('credit_limit')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>
                                        <div class="form-text">
                                            Current Balance: KES{{ number_format($customer->balance, 2) }}
                                            @if($customer->credit_limit)
                                                | Available Credit: KES{{ number_format($customer->available_credit, 2) }}
                                            @endif
                                        </div>
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label for="discount_percent" class="form-label">Discount Percentage</label>
                                        <div class="input-group">
                                            <input type="number" class="form-control @error('discount_percent') is-invalid @enderror" 
                                                   id="discount_percent" name="discount_percent" value="{{ old('discount_percent', $customer->discount_percent) }}"
                                                   placeholder="0.00" step="0.01" min="0" max="100">
                                            <span class="input-group-text">%</span>
                                            @error('discount_percent')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>
                                        <div class="form-text">Percentage discount on all purchases (0-100%)</div>
                                    </div>
                                </div>
                            </div>

                            <!-- Customer Statistics (Read-only) -->
                            <div class="form-section">
                                <h6><i class="bx bx-stats"></i> Customer Statistics (Read-only)</h6>
                                <div class="row">
                                    <div class="col-md-3 mb-3">
                                        <div class="stat-badge text-center">
                                            <div class="text-muted small">Total Spent</div>
                                            <div class="h6 mb-0">KES{{ number_format($customer->total_spent, 2) }}</div>
                                        </div>
                                    </div>
                                    <div class="col-md-3 mb-3">
                                        <div class="stat-badge text-center">
                                            <div class="text-muted small">Total Orders</div>
                                            <div class="h6 mb-0">{{ $customer->total_orders }}</div>
                                        </div>
                                    </div>
                                    <div class="col-md-3 mb-3">
                                        <div class="stat-badge text-center">
                                            <div class="text-muted small">Avg. Order</div>
                                            <div class="h6 mb-0">KES{{ number_format($customer->average_order_value, 2) }}</div>
                                        </div>
                                    </div>
                                    <div class="col-md-3 mb-3">
                                        <div class="stat-badge text-center">
                                            <div class="text-muted small">Loyalty Points</div>
                                            <div class="h6 mb-0">{{ $customer->loyalty_points }}</div>
                                        </div>
                                    </div>
                                    <div class="col-md-6 mb-3">
                                        <div class="stat-badge">
                                            <div class="text-muted small">Taxable Sales</div>
                                            <div class="h6 mb-0">KES{{ number_format($customer->taxable_sales, 2) }}</div>
                                        </div>
                                    </div>
                                    <div class="col-md-6 mb-3">
                                        <div class="stat-badge">
                                            <div class="text-muted small">VAT Collected</div>
                                            <div class="h6 mb-0">KES{{ number_format($customer->vat_collected, 2) }}</div>
                                        </div>
                                    </div>
                                    <div class="col-12">
                                        <div class="text-center">
                                            <small class="text-muted">Last Purchase: 
                                                @if($customer->last_purchase_date)
                                                    {{ $customer->last_purchase_date->format('M d, Y H:i') }}
                                                @else
                                                    Never
                                                @endif
                                            </small>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Additional Notes -->
                            <div class="form-section">
                                <h6><i class="bx bx-note"></i> Additional Information</h6>
                                <div class="row">
                                    <div class="col-12 mb-3">
                                        <label for="notes" class="form-label">Notes</label>
                                        <textarea class="form-control @error('notes') is-invalid @enderror" 
                                                  id="notes" name="notes" rows="4" 
                                                  placeholder="Any additional information about this customer...">{{ old('notes', $customer->notes) }}</textarea>
                                        @error('notes')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                            </div>

                            <!-- Form Actions -->
                            <div class="row mt-4">
                                <div class="col-12">
                                    <div class="d-flex justify-content-between align-items-center">
                                        <div>
                                            <button type="submit" class="btn btn-primary px-5">
                                                <i class="bx bx-save"></i> Update Customer
                                            </button>
                                            <button type="reset" class="btn btn-outline-secondary px-5">
                                                <i class="bx bx-reset"></i> Reset Changes
                                            </button>
                                        </div>
                                        <div>
                                            <a href="{{ route('customers.show', $customer) }}" class="btn btn-outline-info px-4">
                                                <i class="bx bx-show"></i> View
                                            </a>
                                            <a href="{{ route('pos.index') }}?customer_id={{ $customer->id }}" 
                                               class="btn btn-success px-4">
                                                <i class="bx bx-cart-add"></i> New Sale
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script>
    $(document).ready(function() {
        // VAT registration toggle
        function toggleVatFields() {
            const isVatRegistered = $('#is_vat_registered').is(':checked');
            if (isVatRegistered) {
                $('.vat-field').show();
                $('#vat_number').prop('required', true);
            } else {
                $('.vat-field').hide();
                $('#vat_number').prop('required', false);
            }
        }
        
        // Initialize VAT fields
        toggleVatFields();
        
        // Event listener for VAT toggle
        $('#is_vat_registered').change(toggleVatFields);
        
        // Form validation
        $('#customerForm').validate({
            rules: {
                name: {
                    required: true,
                    minlength: 2,
                    maxlength: 255
                },
                email: {
                    email: true
                },
                phone: {
                    required: true,
                    maxlength: 20
                },
                website: {
                    url: true
                },
                credit_limit: {
                    min: 0
                },
                discount_percent: {
                    min: 0,
                    max: 100
                },
                vat_number: {
                    pattern: /^VAT-[0-9]{6,10}$/i
                }
            },
            messages: {
                name: {
                    required: "Customer name is required",
                    minlength: "Customer name must be at least 2 characters"
                },
                phone: {
                    required: "Phone number is required",
                    maxlength: "Phone number cannot exceed 20 characters"
                },
                email: {
                    email: "Please enter a valid email address"
                },
                website: {
                    url: "Please enter a valid website URL"
                },
                vat_number: {
                    pattern: "VAT number must be in format: VAT- followed by 6-10 digits"
                }
            },
            errorElement: 'span',
            errorPlacement: function (error, element) {
                error.addClass('invalid-feedback');
                element.closest('.mb-3').append(error);
            },
            highlight: function (element, errorClass, validClass) {
                $(element).addClass('is-invalid');
            },
            unhighlight: function (element, errorClass, validClass) {
                $(element).removeClass('is-invalid');
            }
        });

        // Phone number formatting
        $('#phone').on('input', function() {
            $(this).val($(this).val().replace(/[^0-9+()\- ]/g, ''));
        });

        // VAT number validation
        $('#vat_number').on('blur', function() {
            const currentVatNumber = '{{ $customer->vat_number }}';
            const newVatNumber = $(this).val().trim();
            
            if (newVatNumber !== '' && newVatNumber !== currentVatNumber) {
                $.ajax({
                    url: '/api/validate-vat',
                    type: 'POST',
                    data: {
                        _token: '{{ csrf_token() }}',
                        vat_number: newVatNumber
                    },
                    success: function(response) {
                        if (response.success && !response.is_valid) {
                            Swal.fire({
                                title: 'VAT Number Issue',
                                text: response.message,
                                icon: 'warning',
                                confirmButtonColor: '#3085d6',
                            });
                        }
                    }
                });
            }
        });

        // Show/hide business fields based on customer type
        function toggleBusinessFields() {
            const customerType = $('#customer_type').val();
            if (customerType === 'business') {
                $('.business-field').show();
            } else {
                $('.business-field').hide();
            }
        }
        
        $('#customer_type').change(toggleBusinessFields);
        toggleBusinessFields(); // Initialize on page load

        // Warn before leaving page with unsaved changes
        let formChanged = false;
        $('#customerForm input, #customerForm textarea, #customerForm select').on('change input', function() {
            formChanged = true;
        });

        $('button[type="reset"]').click(function(e) {
            if (formChanged && !confirm('Are you sure you want to reset all changes?')) {
                e.preventDefault();
            } else {
                formChanged = false;
            }
        });

        $(window).on('beforeunload', function() {
            if (formChanged) {
                return 'You have unsaved changes. Are you sure you want to leave?';
            }
        });

        $('#customerForm').on('submit', function() {
            formChanged = false;
        });
    });
</script>
@endsection