@extends('layouts.app')

@section('style')
    <style>
        .form-section {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        .form-section h6 {
            color: #495057;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 2px solid #dee2e6;
        }
        .required-field::after {
            content: " *";
            color: #dc3545;
        }
        .color-preview {
            width: 30px;
            height: 30px;
            border-radius: 5px;
            border: 2px solid #dee2e6;
            display: inline-block;
            margin-left: 10px;
            vertical-align: middle;
        }
        .image-preview {
            width: 100px;
            height: 100px;
            border: 2px solid #dee2e6;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            overflow: hidden;
            margin-top: 10px;
        }
        .image-preview img {
            max-width: 100%;
            max-height: 100%;
            object-fit: cover;
        }
        .predefined-colors {
            display: flex;
            gap: 5px;
            flex-wrap: wrap;
            margin-top: 10px;
        }
        .color-option {
            width: 25px;
            height: 25px;
            border-radius: 50%;
            cursor: pointer;
            border: 2px solid transparent;
            transition: transform 0.2s;
        }
        .color-option:hover {
            transform: scale(1.1);
        }
        .color-option.selected {
            border-color: #495057;
            transform: scale(1.1);
        }
        .category-avatar {
            width: 80px;
            height: 80px;
            border-radius: 15px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 30px;
            font-weight: bold;
            margin: 0 auto 20px;
        }
    </style>
@endsection

@section('content')
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">Categories</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item"><a href="{{ route('categories.index') }}">Categories</a></li>
                        <li class="breadcrumb-item"><a href="{{ route('categories.show', $category) }}">{{ $category->name }}</a></li>
                        <li class="breadcrumb-item active" aria-current="page">Edit Category</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <div class="btn-group">
                    <a href="{{ route('categories.show', $category) }}" class="btn btn-outline-info">
                        <i class="bx bx-show"></i> View Details
                    </a>
                    <a href="{{ route('categories.index') }}" class="btn btn-outline-secondary">
                        <i class="bx bx-arrow-back"></i> Back to List
                    </a>
                </div>
            </div>
        </div>
        <!--end breadcrumb-->

        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-body">
                        <div class="text-center mb-4">
                            <div class="category-avatar" style="background: {{ $category->color ?? '#667eea' }}">
                                <i class="bx bxs-category"></i>
                            </div>
                            <h5 class="card-title mb-1">Edit Category: {{ $category->name }}</h5>
                            <p class="text-muted">Category ID: #{{ str_pad($category->id, 5, '0', STR_PAD_LEFT) }}</p>
                        </div>
                        
                        @if($errors->any())
                            <div class="alert alert-danger border-0 bg-danger alert-dismissible fade show py-2">
                                <div class="d-flex align-items-center">
                                    <div class="font-35 text-white">
                                        <i class="bx bxs-error"></i>
                                    </div>
                                    <div class="ms-3">
                                        <h6 class="mb-0 text-white">Please fix the following errors:</h6>
                                        <ul class="text-white mb-0">
                                            @foreach($errors->all() as $error)
                                                <li>{{ $error }}</li>
                                            @endforeach
                                        </ul>
                                    </div>
                                </div>
                                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                            </div>
                        @endif

                        @if(session('success'))
                            <div class="alert alert-success border-0 bg-success alert-dismissible fade show py-2">
                                <div class="d-flex align-items-center">
                                    <div class="font-35 text-white">
                                        <i class="bx bxs-check-circle"></i>
                                    </div>
                                    <div class="ms-3">
                                        <h6 class="mb-0 text-white">Success</h6>
                                        <div class="text-white">{{ session('success') }}</div>
                                    </div>
                                </div>
                                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                            </div>
                        @endif

                        <form method="POST" action="{{ route('categories.update', $category) }}" id="categoryForm" enctype="multipart/form-data">
                            @csrf
                            @method('PUT')
                            
                            <!-- Basic Information -->
                            <div class="form-section">
                                <h6><i class="bx bx-info-circle"></i> Basic Information</h6>
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label for="name" class="form-label required-field">Category Name</label>
                                        <input type="text" class="form-control @error('name') is-invalid @enderror" 
                                               id="name" name="name" value="{{ old('name', $category->name) }}" 
                                               placeholder="Enter category name" required>
                                        @error('name')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label for="parent_id" class="form-label">Parent Category</label>
                                        <select class="form-select @error('parent_id') is-invalid @enderror" 
                                                id="parent_id" name="parent_id">
                                            <option value="">None (Main Category)</option>
                                            @foreach($parentCategories as $parentCat)
                                                <option value="{{ $parentCat->id }}" 
                                                    {{ old('parent_id', $category->parent_id) == $parentCat->id ? 'selected' : '' }}
                                                    {{ $parentCat->id == $category->id ? 'disabled' : '' }}>
                                                    {{ $parentCat->name }}
                                                </option>
                                            @endforeach
                                        </select>
                                        @error('parent_id')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                            </div>

                            <!-- Description -->
                            <div class="form-section">
                                <h6><i class="bx bx-detail"></i> Description</h6>
                                <div class="row">
                                    <div class="col-12 mb-3">
                                        <label for="description" class="form-label">Description</label>
                                        <textarea class="form-control @error('description') is-invalid @enderror" 
                                                  id="description" name="description" rows="3" 
                                                  placeholder="Describe this category...">{{ old('description', $category->description) }}</textarea>
                                        @error('description')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                            </div>

                            <!-- Appearance -->
                            <div class="form-section">
                                <h6><i class="bx bx-palette"></i> Appearance</h6>
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label for="color" class="form-label">Category Color</label>
                                        <div class="input-group">
                                            <span class="input-group-text">#</span>
                                            <input type="text" class="form-control @error('color') is-invalid @enderror" 
                                                   id="color" name="color" value="{{ old('color', $category->color ? str_replace('#', '', $category->color) : '667eea') }}"
                                                   placeholder="667eea" maxlength="6">
                                            <div class="color-preview" id="colorPreview" 
                                                 style="background: {{ old('color', $category->color ?? '#667eea') }}"></div>
                                            @error('color')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>
                                        
                                        <!-- Predefined Colors -->
                                        <div class="predefined-colors">
                                            @php
                                                $predefinedColors = [
                                                    '#667eea', '#764ba2', '#f093fb', '#4facfe', '#00f2fe',
                                                    '#43e97b', '#38f9d7', '#fa709a', '#fee140', '#fa709a',
                                                    '#ff6b6b', '#4ecdc4', '#45b7d1', '#96ceb4', '#ffeaa7',
                                                    '#dda0dd', '#98d8c8', '#f7b731', '#eb3b5a', '#26de81'
                                                ];
                                            @endphp
                                            @foreach($predefinedColors as $color)
                                                <div class="color-option" 
                                                     style="background: {{ $color }}"
                                                     onclick="selectColor('{{ $color }}')"
                                                     title="{{ $color }}">
                                                </div>
                                            @endforeach
                                        </div>
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label for="image" class="form-label">Category Image</label>
                                        <input type="file" class="form-control @error('image') is-invalid @enderror" 
                                               id="image" name="image" accept="image/*">
                                        @error('image')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        
                                        <!-- Image Preview -->
                                        <div class="image-preview" id="imagePreview">
                                            @if($category->image)
                                                <img src="{{ asset('storage/' . $category->image) }}" alt="{{ $category->name }}">
                                            @else
                                                <i class="bx bx-image text-muted" style="font-size: 30px;"></i>
                                            @endif
                                        </div>
                                        
                                        @if($category->image)
                                            <div class="form-check mt-2">
                                                <input class="form-check-input" type="checkbox" id="remove_image" name="remove_image" value="1">
                                                <label class="form-check-label" for="remove_image">
                                                    Remove current image
                                                </label>
                                            </div>
                                        @endif
                                    </div>
                                </div>
                            </div>

                            <!-- Settings -->
                            <div class="form-section">
                                <h6><i class="bx bx-cog"></i> Settings</h6>
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <div class="form-check form-switch">
                                            <input class="form-check-input" type="checkbox" role="switch" 
                                                   id="is_active" name="is_active" value="1" 
                                                   {{ old('is_active', $category->is_active) ? 'checked' : '' }}>
                                            <label class="form-check-label" for="is_active">
                                                Active Category
                                            </label>
                                        </div>
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label for="sort_order" class="form-label">Sort Order</label>
                                        <input type="number" class="form-control @error('sort_order') is-invalid @enderror" 
                                               id="sort_order" name="sort_order" value="{{ old('sort_order', $category->sort_order) }}"
                                               min="0" max="999">
                                        @error('sort_order')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                            </div>

                            <!-- Category Statistics -->
                            <div class="form-section">
                                <h6><i class="bx bx-stats"></i> Category Statistics</h6>
                                <div class="row">
                                    <div class="col-md-3 mb-3">
                                        <div class="card bg-light">
                                            <div class="card-body text-center">
                                                <h6 class="text-muted">Total Products</h6>
                                                <h4 class="mb-0">{{ $category->products_count ?? 0 }}</h4>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-3 mb-3">
                                        <div class="card bg-light">
                                            <div class="card-body text-center">
                                                <h6 class="text-muted">Created</h6>
                                                <h6 class="mb-0">{{ $category->created_at->format('M d, Y') }}</h6>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-3 mb-3">
                                        <div class="card bg-light">
                                            <div class="card-body text-center">
                                                <h6 class="text-muted">Last Updated</h6>
                                                <h6 class="mb-0">{{ $category->updated_at->format('M d, Y') }}</h6>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-3 mb-3">
                                        <div class="card bg-light">
                                            <div class="card-body text-center">
                                                <h6 class="text-muted">Status</h6>
                                                <h6 class="mb-0">
                                                    @if($category->is_active)
                                                        <span class="badge bg-success">Active</span>
                                                    @else
                                                        <span class="badge bg-secondary">Inactive</span>
                                                    @endif
                                                </h6>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Form Actions -->
                            <div class="row mt-4">
                                <div class="col-12">
                                    <div class="d-flex justify-content-between align-items-center">
                                        <div>
                                            <button type="submit" class="btn btn-primary px-5">
                                                <i class="bx bx-save"></i> Update Category
                                            </button>
                                            <button type="reset" class="btn btn-outline-secondary px-5">
                                                <i class="bx bx-reset"></i> Reset Changes
                                            </button>
                                        </div>
                                        <div>
                                            <a href="{{ route('categories.show', $category) }}" class="btn btn-outline-info px-4">
                                                <i class="bx bx-show"></i> View
                                            </a>
                                            <a href="{{ route('products.create') }}?category_id={{ $category->id }}" 
                                               class="btn btn-success px-4">
                                                <i class="bx bx-plus"></i> Add Product
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script>
    $(document).ready(function() {
        // Color picker functionality
        function updateColorPreview() {
            let color = $('#color').val();
            if (!color.startsWith('#')) {
                color = '#' + color;
            }
            $('#colorPreview').css('background', color);
            $('#color').val(color.replace('#', ''));
        }
        
        function selectColor(color) {
            $('#color').val(color.replace('#', ''));
            updateColorPreview();
            $('.color-option').removeClass('selected');
            $(`.color-option[title="${color}"]`).addClass('selected');
        }
        
        $('#color').on('input', updateColorPreview);
        
        // Select current color
        const currentColor = '{{ $category->color ?? '#667eea' }}';
        selectColor(currentColor);
        
        // Image preview
        $('#image').change(function() {
            const file = this.files[0];
            if (file) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    $('#imagePreview').html(`<img src="${e.target.result}" alt="Preview">`);
                }
                reader.readAsDataURL(file);
            }
        });
        
        // Remove image checkbox
        $('#remove_image').change(function() {
            if (this.checked) {
                $('#imagePreview').html('<i class="bx bx-image text-muted" style="font-size: 30px;"></i>');
            } else {
                @if($category->image)
                    $('#imagePreview').html('<img src="{{ asset("storage/" . $category->image) }}" alt="{{ $category->name }}">');
                @endif
            }
        });
        
        // Form validation
        $('#categoryForm').validate({
            rules: {
                name: {
                    required: true,
                    minlength: 2,
                    maxlength: 255
                },
                description: {
                    maxlength: 500
                },
                color: {
                    pattern: /^([0-9a-fA-F]{6})?$/
                },
                sort_order: {
                    number: true,
                    min: 0,
                    max: 999
                }
            },
            messages: {
                name: {
                    required: "Category name is required",
                    minlength: "Category name must be at least 2 characters"
                },
                color: {
                    pattern: "Please enter a valid hex color (e.g., 667eea)"
                }
            },
            errorElement: 'span',
            errorPlacement: function (error, element) {
                error.addClass('invalid-feedback');
                element.closest('.mb-3').append(error);
            },
            highlight: function (element, errorClass, validClass) {
                $(element).addClass('is-invalid');
            },
            unhighlight: function (element, errorClass, validClass) {
                $(element).removeClass('is-invalid');
            }
        });

        // Warn before leaving page with unsaved changes
        let formChanged = false;
        $('#categoryForm input, #categoryForm textarea, #categoryForm select').on('change input', function() {
            formChanged = true;
        });

        $('button[type="reset"]').click(function(e) {
            if (formChanged && !confirm('Are you sure you want to reset all changes?')) {
                e.preventDefault();
            } else {
                formChanged = false;
            }
        });

        $(window).on('beforeunload', function() {
            if (formChanged) {
                return 'You have unsaved changes. Are you sure you want to leave?';
            }
        });

        $('#categoryForm').on('submit', function() {
            formChanged = false;
        });
    });
</script>
@endsection