@extends('layouts.app')

@section('style')
    <style>
        .form-section {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        .form-section h6 {
            color: #495057;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 2px solid #dee2e6;
        }
        .required-field::after {
            content: " *";
            color: #dc3545;
        }
        .color-preview {
            width: 30px;
            height: 30px;
            border-radius: 5px;
            border: 2px solid #dee2e6;
            display: inline-block;
            margin-left: 10px;
            vertical-align: middle;
        }
        .image-preview {
            width: 100px;
            height: 100px;
            border: 2px dashed #dee2e6;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            overflow: hidden;
            margin-top: 10px;
        }
        .image-preview img {
            max-width: 100%;
            max-height: 100%;
            object-fit: cover;
        }
        .predefined-colors {
            display: flex;
            gap: 5px;
            flex-wrap: wrap;
            margin-top: 10px;
        }
        .color-option {
            width: 25px;
            height: 25px;
            border-radius: 50%;
            cursor: pointer;
            border: 2px solid transparent;
            transition: transform 0.2s;
        }
        .color-option:hover {
            transform: scale(1.1);
        }
        .color-option.selected {
            border-color: #495057;
            transform: scale(1.1);
        }
        .color-input-group {
            display: flex;
            align-items: center;
        }
        .color-input-group .input-group-text {
            border-top-right-radius: 0;
            border-bottom-right-radius: 0;
            background-color: #f8f9fa;
            border-right: none;
        }
        .color-input-group .form-control {
            border-top-left-radius: 0;
            border-bottom-left-radius: 0;
            border-left: none;
        }
    </style>
@endsection

@section('content')
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">Categories</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item"><a href="{{ route('categories.index') }}">Categories</a></li>
                        <li class="breadcrumb-item active" aria-current="page">Create New Category</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <a href="{{ route('categories.index') }}" class="btn btn-outline-secondary">
                    <i class="bx bx-arrow-back"></i> Back to List
                </a>
            </div>
        </div>
        <!--end breadcrumb-->

        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title mb-4">Add New Category</h5>
                        
                        @if($errors->any())
                            <div class="alert alert-danger border-0 bg-danger alert-dismissible fade show py-2">
                                <div class="d-flex align-items-center">
                                    <div class="font-35 text-white">
                                        <i class="bx bxs-error"></i>
                                    </div>
                                    <div class="ms-3">
                                        <h6 class="mb-0 text-white">Please fix the following errors:</h6>
                                        <ul class="text-white mb-0">
                                            @foreach($errors->all() as $error)
                                                <li>{{ $error }}</li>
                                            @endforeach
                                        </ul>
                                    </div>
                                </div>
                                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                            </div>
                        @endif

                        <form method="POST" action="{{ route('categories.store') }}" id="categoryForm" enctype="multipart/form-data">
                            @csrf
                            
                            <!-- Basic Information -->
                            <div class="form-section">
                                <h6><i class="bx bx-info-circle"></i> Basic Information</h6>
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label for="name" class="form-label required-field">Category Name</label>
                                        <input type="text" class="form-control @error('name') is-invalid @enderror" 
                                               id="name" name="name" value="{{ old('name') }}" 
                                               placeholder="Enter category name" required>
                                        @error('name')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        <div class="form-text">Unique name for the category</div>
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label for="parent_id" class="form-label">Parent Category</label>
                                        <select class="form-select @error('parent_id') is-invalid @enderror" 
                                                id="parent_id" name="parent_id">
                                            <option value="">None (Main Category)</option>
                                            @php
                                                $categories = \App\Models\Category::whereNull('parent_id')->get();
                                            @endphp
                                            @foreach($categories as $cat)
                                                <option value="{{ $cat->id }}" {{ old('parent_id') == $cat->id ? 'selected' : '' }}>
                                                    {{ $cat->name }}
                                                </option>
                                            @endforeach
                                        </select>
                                        @error('parent_id')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        <div class="form-text">Leave empty to create a main category</div>
                                    </div>
                                </div>
                            </div>

                            <!-- Description -->
                            <div class="form-section">
                                <h6><i class="bx bx-detail"></i> Description</h6>
                                <div class="row">
                                    <div class="col-12 mb-3">
                                        <label for="description" class="form-label">Description</label>
                                        <textarea class="form-control @error('description') is-invalid @enderror" 
                                                  id="description" name="description" rows="3" 
                                                  placeholder="Describe this category...">{{ old('description') }}</textarea>
                                        @error('description')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        <div class="form-text">Optional description for the category</div>
                                    </div>
                                </div>
                            </div>

                            <!-- Appearance -->
                            <div class="form-section">
                                <h6><i class="bx bx-palette"></i> Appearance</h6>
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label for="color" class="form-label">Category Color</label>
                                        <div class="color-input-group">
                                            <div class="input-group">
                                                <span class="input-group-text">#</span>
                                                <input type="text" class="form-control @error('color') is-invalid @enderror" 
                                                       id="color" name="color" value="{{ old('color', '667eea') }}"
                                                       placeholder="667eea or f00" maxlength="6">
                                                <div class="color-preview" id="colorPreview" style="background: #{{ old('color', '667eea') }}"></div>
                                            </div>
                                            @error('color')
                                                <div class="invalid-feedback d-block">{{ $message }}</div>
                                            @enderror
                                        </div>
                                        
                                        <!-- Predefined Colors -->
                                        <div class="predefined-colors">
                                            @php
                                                $predefinedColors = [
                                                    '#667eea', '#764ba2', '#f093fb', '#4facfe', '#00f2fe',
                                                    '#43e97b', '#38f9d7', '#fa709a', '#fee140', '#fa709a',
                                                    '#ff6b6b', '#4ecdc4', '#45b7d1', '#96ceb4', '#ffeaa7',
                                                    '#dda0dd', '#98d8c8', '#f7b731', '#eb3b5a', '#26de81'
                                                ];
                                            @endphp
                                            @foreach($predefinedColors as $color)
                                                <div class="color-option" 
                                                     data-color="{{ str_replace('#', '', $color) }}"
                                                     style="background: {{ $color }}"
                                                     onclick="selectColor('{{ str_replace('#', '', $color) }}')"
                                                     title="{{ $color }}">
                                                </div>
                                            @endforeach
                                        </div>
                                        <div class="form-text">Click a color or enter a hex value (with or without #)</div>
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label for="image" class="form-label">Category Image</label>
                                        <input type="file" class="form-control @error('image') is-invalid @enderror" 
                                               id="image" name="image" accept="image/*">
                                        @error('image')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        
                                        <!-- Image Preview -->
                                        <div class="image-preview" id="imagePreview">
                                            <i class="bx bx-image-add text-muted" style="font-size: 30px;"></i>
                                        </div>
                                        
                                        <div class="form-text">Optional. Max size: 2MB. Recommended: 300x300 pixels</div>
                                    </div>
                                </div>
                            </div>

                            <!-- Settings -->
                            <div class="form-section">
                                <h6><i class="bx bx-cog"></i> Settings</h6>
                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <div class="form-check form-switch">
                                            <input class="form-check-input" type="checkbox" role="switch" 
                                                   id="is_active" name="is_active" value="1" {{ old('is_active', true) ? 'checked' : '' }}>
                                            <label class="form-check-label" for="is_active">
                                                Active Category
                                            </label>
                                        </div>
                                        <div class="form-text">Inactive categories won't appear in dropdowns</div>
                                    </div>
                                    
                                    <div class="col-md-6 mb-3">
                                        <label for="sort_order" class="form-label">Sort Order</label>
                                        <input type="number" class="form-control @error('sort_order') is-invalid @enderror" 
                                               id="sort_order" name="sort_order" value="{{ old('sort_order', 0) }}"
                                               min="0" max="999">
                                        @error('sort_order')
                                            <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        <div class="form-text">Lower numbers appear first</div>
                                    </div>
                                </div>
                            </div>

                            <!-- Form Actions -->
                            <div class="row mt-4">
                                <div class="col-12">
                                    <div class="d-flex justify-content-between">
                                        <div>
                                            <button type="submit" class="btn btn-primary px-5">
                                                <i class="bx bx-save"></i> Save Category
                                            </button>
                                            <button type="reset" class="btn btn-outline-secondary px-5" id="resetButton">
                                                <i class="bx bx-reset"></i> Reset Form
                                            </button>
                                        </div>
                                        <div>
                                            <a href="{{ route('categories.index') }}" class="btn btn-light px-5">
                                                <i class="bx bx-x"></i> Cancel
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script src="https://cdnjs.cloudflare.com/ajax/libs/jquery-validate/1.19.5/jquery.validate.min.js"></script>
<script>
    $(document).ready(function() {
        // Color picker functionality
        function updateColorPreview() {
            let color = $('#color').val().trim();
            
            // Remove any # characters
            color = color.replace(/#/g, '');
            
            // Validate hex color
            const hexRegex = /^([0-9a-fA-F]{3}|[0-9a-fA-F]{6})$/;
            
            if (color === '' || hexRegex.test(color)) {
                // Update preview
                $('#colorPreview').css('background', '#' + color);
                
                // Update input value (clean, without #)
                $('#color').val(color);
                
                // Update selected state for predefined colors
                $('.color-option').removeClass('selected');
                $('.color-option[data-color="' + color.toLowerCase() + '"]').addClass('selected');
                
                return true;
            } else {
                // Invalid color
                $('#colorPreview').css('background', '#ccc');
                $('.color-option').removeClass('selected');
                return false;
            }
        }
        
        function selectColor(color) {
            // Remove any # characters
            color = color.replace(/#/g, '');
            
            // Set the color value
            $('#color').val(color);
            
            // Update preview
            updateColorPreview();
            
            // Highlight selected color
            $('.color-option').removeClass('selected');
            $(`.color-option[data-color="${color.toLowerCase()}"]`).addClass('selected');
        }
        
        // Initialize color preview
        updateColorPreview();
        
        // Update preview on input
        $('#color').on('input', function() {
            updateColorPreview();
        });
        
        // Select predefined color if it matches old value
        @if(old('color'))
            selectColor('{{ old('color') }}');
        @endif
        
        // Image preview
        $('#image').change(function() {
            const file = this.files[0];
            if (file) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    $('#imagePreview').html(`<img src="${e.target.result}" alt="Preview">`);
                }
                reader.readAsDataURL(file);
            } else {
                $('#imagePreview').html('<i class="bx bx-image-add text-muted" style="font-size: 30px;"></i>');
            }
        });
        
        // Form validation
        $('#categoryForm').validate({
            rules: {
                name: {
                    required: true,
                    minlength: 2,
                    maxlength: 255
                },
                description: {
                    maxlength: 500
                },
                color: {
                    pattern: /^([0-9a-fA-F]{3}|[0-9a-fA-F]{6})?$/  // Hex color without #
                },
                sort_order: {
                    number: true,
                    min: 0,
                    max: 999
                }
            },
            messages: {
                name: {
                    required: "Category name is required",
                    minlength: "Category name must be at least 2 characters"
                },
                color: {
                    pattern: "Please enter a valid hex color (3 or 6 characters, e.g., 667eea or f00)"
                },
                sort_order: {
                    number: "Please enter a valid number",
                    min: "Sort order cannot be negative",
                    max: "Sort order cannot exceed 999"
                }
            },
            errorElement: 'span',
            errorPlacement: function (error, element) {
                error.addClass('invalid-feedback');
                element.closest('.mb-3').append(error);
            },
            highlight: function (element, errorClass, validClass) {
                $(element).addClass('is-invalid');
            },
            unhighlight: function (element, errorClass, validClass) {
                $(element).removeClass('is-invalid');
            },
            submitHandler: function(form) {
                // Ensure color is valid before submitting
                if ($('#color').val() && !updateColorPreview()) {
                    $('#color').addClass('is-invalid');
                    $('#color').closest('.mb-3').find('.invalid-feedback').remove();
                    $('#color').closest('.mb-3').append(
                        '<div class="invalid-feedback">Please enter a valid hex color (3 or 6 characters, e.g., 667eea or f00)</div>'
                    );
                    return false;
                }
                
                // Show loading state
                const submitButton = $(form).find('button[type="submit"]');
                const originalText = submitButton.html();
                submitButton.prop('disabled', true).html('<i class="bx bx-loader bx-spin"></i> Saving...');
                
                // Submit the form
                form.submit();
            }
        });

        // Form reset functionality
        $('#resetButton').click(function(e) {
            e.preventDefault();
            
            if (confirm('Are you sure you want to reset the form? All entered data will be lost.')) {
                // Reset form
                $('#categoryForm')[0].reset();
                
                // Reset color preview to default
                selectColor('667eea');
                
                // Reset image preview
                $('#imagePreview').html('<i class="bx bx-image-add text-muted" style="font-size: 30px;"></i>');
                
                // Clear validation errors
                $('.is-invalid').removeClass('is-invalid');
                $('.invalid-feedback').remove();
                
                // Reset switch to default
                $('#is_active').prop('checked', true);
            }
        });
        
        // Prevent enter key from submitting form accidentally
        $('#categoryForm').find('input[type="text"], input[type="number"], textarea').keydown(function(e) {
            if (e.keyCode === 13) {
                e.preventDefault();
                return false;
            }
        });
    });
</script>
@endsection