@extends('layouts.app')

@section('style')
    <style>
        .brand-logo {
            width: 50px;
            height: 50px;
            object-fit: contain;
            border-radius: 8px;
            border: 1px solid #dee2e6;
            background: white;
        }
        .action-buttons .btn {
            padding: 0.25rem 0.5rem;
            font-size: 0.875rem;
        }
        .country-flag {
            font-size: 1.5em;
            margin-right: 5px;
        }
        .brand-card {
            transition: all 0.3s;
        }
        .brand-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
    </style>
@endsection

@section('content')
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">Brands</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="bx bx-home-alt"></i></a></li>
                        <li class="breadcrumb-item active" aria-current="page">Brands Management</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <a href="{{ route('brands.create') }}" class="btn btn-primary">
                    <i class="bx bx-plus"></i> Add New Brand
                </a>
            </div>
        </div>
        <!--end breadcrumb-->

        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-body">
                        <!-- Statistics -->
                        <div class="row mb-4">
                            <div class="col-md-3 col-sm-6">
                                <div class="card bg-primary bg-gradient text-white">
                                    <div class="card-body">
                                        <div class="d-flex align-items-center">
                                            <div class="flex-grow-1">
                                                <h5 class="mb-0">{{ $stats['total'] ?? 0 }}</h5>
                                                <p class="mb-0">Total Brands</p>
                                            </div>
                                            <div class="flex-shrink-0">
                                                <i class="bx bx-tag fs-1"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3 col-sm-6">
                                <div class="card bg-success bg-gradient text-white">
                                    <div class="card-body">
                                        <div class="d-flex align-items-center">
                                            <div class="flex-grow-1">
                                                <h5 class="mb-0">{{ $stats['active'] ?? 0 }}</h5>
                                                <p class="mb-0">Active Brands</p>
                                            </div>
                                            <div class="flex-shrink-0">
                                                <i class="bx bx-check-circle fs-1"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3 col-sm-6">
                                <div class="card bg-info bg-gradient text-white">
                                    <div class="card-body">
                                        <div class="d-flex align-items-center">
                                            <div class="flex-grow-1">
                                                <h5 class="mb-0">{{ $stats['featured'] ?? 0 }}</h5>
                                                <p class="mb-0">Featured Brands</p>
                                            </div>
                                            <div class="flex-shrink-0">
                                                <i class="bx bx-star fs-1"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-3 col-sm-6">
                                <div class="card bg-warning bg-gradient text-white">
                                    <div class="card-body">
                                        <div class="d-flex align-items-center">
                                            <div class="flex-grow-1">
                                                <h5 class="mb-0">{{ $stats['total_products'] ?? 0 }}</h5>
                                                <p class="mb-0">Total Products</p>
                                            </div>
                                            <div class="flex-shrink-0">
                                                <i class="bx bx-package fs-1"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Search and Filter -->
                        <div class="row mb-4">
                            <div class="col-md-8">
                                <form method="GET" action="{{ route('brands.index') }}" class="row g-3">
                                    <div class="col-md-4">
                                        <input type="text" name="search" class="form-control" 
                                               placeholder="Search brands..." value="{{ request('search') }}">
                                    </div>
                                    <div class="col-md-3">
                                        <select name="status" class="form-select">
                                            <option value="">All Status</option>
                                            <option value="active" {{ request('status') == 'active' ? 'selected' : '' }}>Active</option>
                                            <option value="inactive" {{ request('status') == 'inactive' ? 'selected' : '' }}>Inactive</option>
                                        </select>
                                    </div>
                                    <div class="col-md-3">
                                        <select name="country" class="form-select">
                                            <option value="">All Countries</option>
                                            @foreach($countries as $country)
                                                <option value="{{ $country }}" {{ request('country') == $country ? 'selected' : '' }}>
                                                    {{ $country }}
                                                </option>
                                            @endforeach
                                        </select>
                                    </div>
                                    <div class="col-md-2">
                                        <button type="submit" class="btn btn-primary w-100">
                                            <i class="bx bx-search"></i> Filter
                                        </button>
                                    </div>
                                </form>
                            </div>
                            <div class="col-md-4 text-end">
                                <div class="btn-group" role="group">
                                    <a href="{{ route('brands.export') }}" class="btn btn-outline-secondary">
                                        <i class="bx bx-export"></i> Export
                                    </a>
                                    <button type="button" class="btn btn-outline-secondary" onclick="bulkUpdateModal()">
                                        <i class="bx bx-edit"></i> Bulk Edit
                                    </button>
                                    <button type="button" class="btn btn-outline-danger" onclick="bulkDeleteModal()">
                                        <i class="bx bx-trash"></i> Bulk Delete
                                    </button>
                                </div>
                            </div>
                        </div>

                        <!-- Brands Table -->
                        <div class="table-responsive">
                            <table class="table table-hover table-striped">
                                <thead class="table-light">
                                    <tr>
                                        <th width="50">
                                            <input type="checkbox" id="selectAll">
                                        </th>
                                        <th>Logo</th>
                                        <th>Brand Name</th>
                                        <th>Description</th>
                                        <th>Country</th>
                                        <th>Products</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @forelse($brands as $brand)
                                        <tr>
                                            <td>
                                                <input type="checkbox" class="brand-checkbox" value="{{ $brand->id }}">
                                            </td>
                                            <td>
                                                <img src="{{ $brand->logo_url }}" alt="{{ $brand->name }}" 
                                                     class="brand-logo" onerror="this.src='{{ asset('images/default-brand.png') }}'">
                                            </td>
                                            <td>
                                                <strong>{{ $brand->name }}</strong>
                                                @if($brand->is_featured)
                                                    <br><span class="badge bg-warning">Featured</span>
                                                @endif
                                            </td>
                                            <td>
                                                @if($brand->description)
                                                    {{ Str::limit($brand->description, 50) }}
                                                @else
                                                    <span class="text-muted">No description</span>
                                                @endif
                                            </td>
                                            <td>
                                                @if($brand->country)
                                                    <span class="badge bg-light text-dark">{{ $brand->country }}</span>
                                                @else
                                                    <span class="text-muted">-</span>
                                                @endif
                                            </td>
                                            <td>
                                                <span class="badge bg-secondary">{{ $brand->products_count ?? 0 }}</span>
                                            </td>
                                            <td>
                                                <span class="badge bg-{{ $brand->status == 'active' ? 'success' : 'danger' }}">
                                                    {{ ucfirst($brand->status) }}
                                                </span>
                                            </td>
                                            <td>
                                                <div class="action-buttons">
                                                    <a href="{{ route('brands.show', $brand) }}" class="btn btn-sm btn-info" 
                                                       title="View">
                                                        <i class="bx bx-show"></i>
                                                    </a>
                                                    <a href="{{ route('brands.edit', $brand) }}" class="btn btn-sm btn-warning" 
                                                       title="Edit">
                                                        <i class="bx bx-edit"></i>
                                                    </a>
                                                    <button type="button" class="btn btn-sm btn-danger" 
                                                            onclick="deleteBrand({{ $brand->id }}, '{{ $brand->name }}')"
                                                            title="Delete">
                                                        <i class="bx bx-trash"></i>
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                    @empty
                                        <tr>
                                            <td colspan="8" class="text-center text-muted py-4">
                                                <i class="bx bx-tag fs-1"></i>
                                                <p class="mb-0">No brands found</p>
                                                <a href="{{ route('brands.create') }}" class="btn btn-primary mt-2">
                                                    <i class="bx bx-plus"></i> Add Your First Brand
                                                </a>
                                            </td>
                                        </tr>
                                    @endforelse
                                </tbody>
                            </table>
                        </div>

                        <!-- Pagination -->
                        <div class="row mt-3">
                            <div class="col-md-6">
                                <p class="text-muted">
                                    Showing {{ $brands->firstItem() }} to {{ $brands->lastItem() }} of {{ $brands->total() }} entries
                                </p>
                            </div>
                            <div class="col-md-6">
                                <div class="float-end">
                                    {{ $brands->withQueryString()->links() }}
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Bulk Update Modal -->
<div class="modal fade" id="bulkUpdateModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Bulk Update Brands</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <form id="bulkUpdateForm">
                    @csrf
                    <input type="hidden" name="brand_ids" id="bulkUpdateIds">
                    
                    <div class="mb-3">
                        <label class="form-label">Status</label>
                        <select name="data[status]" class="form-select">
                            <option value="">No Change</option>
                            <option value="active">Active</option>
                            <option value="inactive">Inactive</option>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="data[is_featured]" value="1" id="bulkFeatured">
                            <label class="form-check-label" for="bulkFeatured">
                                Mark as Featured
                            </label>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="submitBulkUpdate()">Update Brands</button>
            </div>
        </div>
    </div>
</div>

<!-- Bulk Delete Modal -->
<div class="modal fade" id="bulkDeleteModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Confirm Bulk Delete</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to delete the selected brands?</p>
                <div class="alert alert-warning">
                    <i class="bx bx-error"></i> This action cannot be undone. Brands that have products cannot be deleted.
                </div>
                <ul id="bulkDeleteList" class="list-group"></ul>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-danger" onclick="submitBulkDelete()">Delete Selected</button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script>
    // Select/Deselect all
    $('#selectAll').change(function() {
        $('.brand-checkbox').prop('checked', this.checked);
    });

    // Get selected brand IDs
    function getSelectedBrandIds() {
        return $('.brand-checkbox:checked').map(function() {
            return $(this).val();
        }).get();
    }

    // Get selected brand names
    function getSelectedBrandNames() {
        let names = [];
        $('.brand-checkbox:checked').each(function() {
            let brandName = $(this).closest('tr').find('td:nth-child(3) strong').text();
            names.push(brandName);
        });
        return names;
    }

    // Bulk update modal
    function bulkUpdateModal() {
        let selectedIds = getSelectedBrandIds();
        if (selectedIds.length === 0) {
            Swal.fire('Warning', 'Please select at least one brand.', 'warning');
            return;
        }

        $('#bulkUpdateIds').val(JSON.stringify(selectedIds));
        $('#bulkUpdateModal').modal('show');
    }

    // Submit bulk update
    function submitBulkUpdate() {
        let formData = $('#bulkUpdateForm').serialize();
        
        $.ajax({
            url: '{{ route("brands.bulk-update") }}',
            type: 'POST',
            data: formData,
            success: function(response) {
                Swal.fire('Success', response.message, 'success');
                $('#bulkUpdateModal').modal('hide');
                location.reload();
            },
            error: function(xhr) {
                Swal.fire('Error', xhr.responseJSON?.message || 'Failed to update brands', 'error');
            }
        });
    }

    // Bulk delete modal
    function bulkDeleteModal() {
        let selectedIds = getSelectedBrandIds();
        if (selectedIds.length === 0) {
            Swal.fire('Warning', 'Please select at least one brand.', 'warning');
            return;
        }

        let brandNames = getSelectedBrandNames();
        let listHtml = '';
        brandNames.forEach(function(name) {
            listHtml += `<li class="list-group-item">${name}</li>`;
        });

        $('#bulkDeleteList').html(listHtml);
        $('#bulkDeleteModal').modal('show');
    }

    // Submit bulk delete
    function submitBulkDelete() {
        let selectedIds = getSelectedBrandIds();
        
        $.ajax({
            url: '{{ route("brands.bulk-delete") }}',
            type: 'POST',
            data: {
                _token: '{{ csrf_token() }}',
                brand_ids: selectedIds
            },
            success: function(response) {
                Swal.fire('Success', response.message, 'success');
                if (response.errors && response.errors.length > 0) {
                    let errorHtml = '<ul class="text-start">';
                    response.errors.forEach(function(error) {
                        errorHtml += `<li>${error}</li>`;
                    });
                    errorHtml += '</ul>';
                    Swal.fire({
                        title: 'Some brands could not be deleted',
                        html: errorHtml,
                        icon: 'warning'
                    });
                }
                $('#bulkDeleteModal').modal('hide');
                location.reload();
            },
            error: function(xhr) {
                Swal.fire('Error', xhr.responseJSON?.message || 'Failed to delete brands', 'error');
            }
        });
    }

    // Delete single brand
    function deleteBrand(id, name) {
        Swal.fire({
            title: 'Delete Brand?',
            html: `Are you sure you want to delete <strong>${name}</strong>?`,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#d33',
            cancelButtonColor: '#3085d6',
            confirmButtonText: 'Yes, delete it!'
        }).then((result) => {
            if (result.isConfirmed) {
                $.ajax({
                    url: `/brands/${id}`,
                    type: 'DELETE',
                    data: {
                        _token: '{{ csrf_token() }}'
                    },
                    success: function(response) {
                        Swal.fire('Deleted!', response.message || 'Brand deleted successfully.', 'success');
                        location.reload();
                    },
                    error: function(xhr) {
                        Swal.fire('Error!', xhr.responseJSON?.message || 'Failed to delete brand.', 'error');
                    }
                });
            }
        });
    }
</script>
@endsection