@extends('layouts.app')

@section('title', 'Activity Logs Management')

@section('style')
<style>
.card-statistic .card-icon {
    width: 60px;
    height: 60px;
    border-radius: 8px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 24px;
    color: white;
}

.card-statistic h4 {
    font-size: 24px;
    font-weight: 700;
}

.badge-event {
    font-size: 11px;
    padding: 4px 8px;
    border-radius: 4px;
}

.user-avatar {
    width: 32px;
    height: 32px;
    border-radius: 50%;
    background-color: #f8f9fa;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #6c757d;
    font-size: 14px;
}

.activity-icon {
    width: 36px;
    height: 36px;
    border-radius: 8px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 16px;
}

.toast {
    background-color: white;
    border: 1px solid rgba(0,0,0,.1);
    box-shadow: 0 0.5rem 1rem rgba(0,0,0,.15);
}

#detailProperties {
    font-family: 'SFMono-Regular', Menlo, Monaco, Consolas, 'Liberation Mono', 'Courier New', monospace;
    background-color: #f8f9fa;
    border: 1px solid #dee2e6;
}

.form-control-plaintext {
    min-height: calc(1.5em + 0.75rem + 2px);
    padding: 0.375rem 0;
    margin-bottom: 0;
}

.table-hover tbody tr:hover {
    background-color: rgba(0, 0, 0, 0.075);
}
</style>
@endsection

@section('content')
<div class="page-wrapper">
    <div class="page-content">
        <!--breadcrumb-->
        <div class="page-breadcrumb d-none d-sm-flex align-items-center mb-3">
            <div class="breadcrumb-title pe-3">System</div>
            <div class="ps-3">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb mb-0 p-0">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="bx bx-home-alt"></i></a>
                        </li>
                        <li class="breadcrumb-item active" aria-current="page">Activity Logs</li>
                    </ol>
                </nav>
            </div>
            <div class="ms-auto">
                <div class="btn-group">
                    <button type="button" class="btn btn-primary" id="refreshBtn">
                        <i class="bx bx-refresh"></i> Refresh
                    </button>
                    <div class="btn-group" role="group">
                        <button type="button" class="btn btn-primary dropdown-toggle" data-bs-toggle="dropdown" aria-expanded="false">
                            <i class="bx bx-download"></i> Export
                        </button>
                        <ul class="dropdown-menu">
                            <li><a class="dropdown-item" href="#" onclick="exportLogs('csv')"><i class="bx bx-file"></i> CSV</a></li>
                            <li><a class="dropdown-item" href="#" onclick="exportLogs('json')"><i class="bx bx-code-alt"></i> JSON</a></li>
                            <li><a class="dropdown-item" href="#" onclick="exportLogs('excel')"><i class="bx bx-spreadsheet"></i> Excel</a></li>
                            <li><hr class="dropdown-divider"></li>
                            <li><a class="dropdown-item" href="#" onclick="showExportModal()"><i class="bx bx-cog"></i> Advanced Export</a></li>
                        </ul>
                    </div>
                    <button type="button" class="btn btn-danger" onclick="showCleanModal()">
                        <i class="bx bx-trash"></i> Clean Old
                    </button>
                </div>
            </div>
        </div>
        <!--end breadcrumb-->

        <!-- Statistics Cards -->
        <div class="row row-cols-1 row-cols-md-2 row-cols-xl-4" id="statsCards" style="display: none;">
            <div class="col">
                <div class="card radius-10 border-start border-0 border-3 border-primary">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-0 text-secondary">Total Logs</p>
                                <h4 class="my-1 text-primary" id="totalLogs">0</h4>
                            </div>
                            <div class="widgets-icons-2 rounded-circle bg-gradient-cosmic text-white ms-auto"><i class='bx bx-history'></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="card radius-10 border-start border-0 border-3 border-success">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-0 text-secondary">Today</p>
                                <h4 class="my-1 text-success" id="todayLogs">0</h4>
                            </div>
                            <div class="widgets-icons-2 rounded-circle bg-gradient-ibiza text-white ms-auto"><i class='bx bx-calendar'></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="card radius-10 border-start border-0 border-3 border-info">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-0 text-secondary">Active Users</p>
                                <h4 class="my-1 text-info" id="activeUsers">0</h4>
                            </div>
                            <div class="widgets-icons-2 rounded-circle bg-gradient-moonlit text-white ms-auto"><i class='bx bx-user'></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col">
                <div class="card radius-10 border-start border-0 border-3 border-warning">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div>
                                <p class="mb-0 text-secondary">Avg. Daily</p>
                                <h4 class="my-1 text-warning" id="avgDaily">0</h4>
                            </div>
                            <div class="widgets-icons-2 rounded-circle bg-gradient-blooker text-white ms-auto"><i class='bx bx-line-chart'></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Filters Card -->
        <div class="card radius-10 mb-4">
            <div class="card-body">
                <div class="d-flex align-items-center mb-3">
                    <div>
                        <h6 class="mb-0">Filters</h6>
                    </div>
                </div>
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Search</label>
                        <div class="input-group">
                            <input type="text" class="form-control" id="searchInput" placeholder="Search logs...">
                            <button class="btn btn-outline-secondary" type="button" id="searchBtn">
                                <i class="bx bx-search"></i>
                            </button>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Log Type</label>
                        <select class="form-select" id="logTypeFilter">
                            <option value="">All Types</option>
                            @foreach($logTypes as $type)
                                <option value="{{ $type }}">{{ ucfirst($type) }}</option>
                            @endforeach
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Event</label>
                        <select class="form-select" id="eventFilter">
                            <option value="">All Events</option>
                            @foreach($events as $event)
                                <option value="{{ $event }}">{{ ucfirst(str_replace('_', ' ', $event)) }}</option>
                            @endforeach
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">User</label>
                        <select class="form-select" id="userFilter">
                            <option value="">All Users</option>
                            @foreach($users as $user)
                                <option value="{{ $user->id }}">{{ $user->name }}</option>
                            @endforeach
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Date Range</label>
                        <div class="input-group">
                            <input type="date" class="form-control" id="dateFromFilter">
                            <span class="input-group-text">to</span>
                            <input type="date" class="form-control" id="dateToFilter">
                        </div>
                    </div>
                </div>
                <div class="row mt-3">
                    <div class="col-12">
                        <div class="d-flex justify-content-between">
                            <div>
                                <button class="btn btn-secondary" onclick="resetFilters()">
                                    <i class="bx bx-reset"></i> Reset Filters
                                </button>
                            </div>
                            <div>
                                <button class="btn btn-success" onclick="applyFilters()">
                                    <i class="bx bx-filter"></i> Apply Filters
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Activity Logs Table -->
        <div class="card radius-10">
            <div class="card-body">
                <div class="d-flex align-items-center mb-3">
                    <div>
                        <h6 class="mb-0">Activity Logs</h6>
                    </div>
                    <div class="dropdown ms-auto">
                        <div class="d-flex align-items-center">
                            <div class="form-check me-3">
                                <input class="form-check-input" type="checkbox" id="selectAllCheckbox">
                                <label class="form-check-label" for="selectAllCheckbox">Select All</label>
                            </div>
                            <button class="btn btn-sm btn-danger me-2" id="deleteSelectedBtn" disabled onclick="deleteSelected()">
                                <i class="bx bx-trash"></i> Delete Selected
                            </button>
                            <button class="btn btn-sm btn-light dropdown-toggle" type="button" data-bs-toggle="dropdown">
                                <i class="bx bx-cog"></i> Actions
                            </button>
                            <ul class="dropdown-menu">
                                <li><a class="dropdown-item" href="#" onclick="selectAll()"><i class="bx bx-check-square"></i> Select All</a></li>
                                <li><a class="dropdown-item" href="#" onclick="deselectAll()"><i class="bx bx-square"></i> Deselect All</a></li>
                                <li><hr class="dropdown-divider"></li>
                                <li><a class="dropdown-item" href="#" onclick="exportSelected()"><i class="bx bx-download"></i> Export Selected</a></li>
                                <li><a class="dropdown-item text-danger" href="#" onclick="showBulkDeleteModal()"><i class="bx bx-trash"></i> Bulk Delete</a></li>
                            </ul>
                        </div>
                    </div>
                </div>
                
                <div class="table-responsive">
                    <table class="table table-hover mb-0" id="activityLogsTable">
                        <thead class="table-light">
                            <tr>
                                <th width="50">
                                    <input type="checkbox" class="form-check-input" id="masterCheckbox">
                                </th>
                                <th width="180">Date & Time</th>
                                <th width="150">User</th>
                                <th>Activity</th>
                                <th width="120">Type</th>
                                <th width="100">Event</th>
                                <th width="120">Actions</th>
                            </tr>
                        </thead>
                        <tbody id="activitiesTableBody">
                            <tr>
                                <td colspan="7" class="text-center py-5">
                                    <div class="spinner-border text-primary" role="status">
                                        <span class="visually-hidden">Loading...</span>
                                    </div>
                                    <p class="mt-2">Loading activity logs...</p>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
                
                <!-- Pagination -->
                <div class="d-flex justify-content-between align-items-center mt-3">
                    <div>
                        <select class="form-select form-select-sm" id="perPageSelect" style="width: auto;">
                            <option value="10">10 per page</option>
                            <option value="25" selected>25 per page</option>
                            <option value="50">50 per page</option>
                            <option value="100">100 per page</option>
                        </select>
                    </div>
                    <nav aria-label="Page navigation">
                        <ul class="pagination justify-content-center mb-0" id="paginationContainer">
                            <li class="page-item disabled">
                                <a class="page-link" href="#" tabindex="-1">Previous</a>
                            </li>
                            <li class="page-item active"><a class="page-link" href="#">1</a></li>
                            <li class="page-item disabled">
                                <a class="page-link" href="#">Next</a>
                            </li>
                        </ul>
                    </nav>
                    <div class="text-muted">
                        Showing <span id="showingFrom">0</span> to <span id="showingTo">0</span> of <span id="totalItems">0</span> entries
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Details Modal -->
<div class="modal fade" id="detailsModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Activity Log Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label class="form-label fw-bold">Date & Time</label>
                            <p id="detailDateTime" class="form-control-plaintext"></p>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label class="form-label fw-bold">User</label>
                            <p id="detailUser" class="form-control-plaintext"></p>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label class="form-label fw-bold">Log Type</label>
                            <p id="detailLogType" class="form-control-plaintext"></p>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label class="form-label fw-bold">Event</label>
                            <p id="detailEvent" class="form-control-plaintext"></p>
                        </div>
                    </div>
                    <div class="col-12">
                        <div class="mb-3">
                            <label class="form-label fw-bold">Description</label>
                            <p id="detailDescription" class="form-control-plaintext border rounded p-2 bg-light"></p>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label class="form-label fw-bold">IP Address</label>
                            <p id="detailIpAddress" class="form-control-plaintext"></p>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label class="form-label fw-bold">User Agent</label>
                            <p id="detailUserAgent" class="form-control-plaintext text-truncate"></p>
                        </div>
                    </div>
                    <div class="col-12">
                        <div class="mb-3">
                            <label class="form-label fw-bold">Subject</label>
                            <p id="detailSubject" class="form-control-plaintext"></p>
                        </div>
                    </div>
                    <div class="col-12">
                        <div class="mb-3">
                            <label class="form-label fw-bold">Properties</label>
                            <pre class="bg-light p-3 rounded" id="detailProperties" style="max-height: 300px; overflow-y: auto; font-size: 12px;"></pre>
                        </div>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                <button type="button" class="btn btn-danger" onclick="deleteLog()" id="deleteLogBtn">
                    <i class="bx bx-trash"></i> Delete Log
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Clean Old Logs Modal -->
<div class="modal fade" id="cleanModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Clean Old Activity Logs</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="alert border-0 border-start border-5 border-warning alert-dismissible fade show py-2">
                    <div class="d-flex align-items-center">
                        <div class="font-35 text-warning"><i class="bx bx-info-circle"></i>
                        </div>
                        <div class="ms-3">
                            <h6 class="mb-0 text-warning">Warning</h6>
                            <div>This action will permanently delete activity logs older than the specified number of days.</div>
                        </div>
                    </div>
                </div>
                <div class="mb-3">
                    <label class="form-label">Delete logs older than (days)</label>
                    <input type="number" class="form-control" id="cleanDays" value="90" min="1" max="365">
                    <div class="form-text">Recommended: 90 days (3 months)</div>
                </div>
                <div class="mb-3">
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" id="confirmClean">
                        <label class="form-check-label" for="confirmClean">
                            I understand this action cannot be undone
                        </label>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-danger" onclick="cleanOldLogs()" id="cleanBtn" disabled>
                    <i class="bx bx-trash"></i> Clean Old Logs
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Bulk Delete Modal -->
<div class="modal fade" id="bulkDeleteModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Bulk Delete Activity Logs</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="alert border-0 border-start border-5 border-danger alert-dismissible fade show py-2">
                    <div class="d-flex align-items-center">
                        <div class="font-35 text-danger"><i class="bx bx-error-circle"></i>
                        </div>
                        <div class="ms-3">
                            <h6 class="mb-0 text-danger">Danger</h6>
                            <div>You are about to delete multiple activity logs. This action cannot be undone.</div>
                        </div>
                    </div>
                </div>
                <div class="mb-3">
                    <label class="form-label">Delete Method</label>
                    <select class="form-select" id="bulkDeleteMethod">
                        <option value="selected">Delete selected logs only</option>
                        <option value="filtered">Delete all logs matching current filters</option>
                        <option value="custom">Delete by custom criteria</option>
                    </select>
                </div>
                <div id="customCriteria" style="display: none;">
                    <div class="mb-3">
                        <label class="form-label">Log Type</label>
                        <select class="form-select" id="bulkLogType">
                            <option value="">All Types</option>
                            @foreach($logTypes as $type)
                                <option value="{{ $type }}">{{ ucfirst($type) }}</option>
                            @endforeach
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Event</label>
                        <select class="form-select" id="bulkEvent">
                            <option value="">All Events</option>
                            @foreach($events as $event)
                                <option value="{{ $event }}">{{ ucfirst(str_replace('_', ' ', $event)) }}</option>
                            @endforeach
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Older than (days)</label>
                        <input type="number" class="form-control" id="bulkOlderThan" value="30" min="1">
                    </div>
                </div>
                <div class="mb-3">
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" id="confirmBulkDelete">
                        <label class="form-check-label" for="confirmBulkDelete">
                            I understand this action cannot be undone
                        </label>
                    </div>
                </div>
                <div class="alert border-0 border-start border-5 border-info alert-dismissible fade show py-2" id="bulkDeleteInfo">
                    <div class="d-flex align-items-center">
                        <div class="font-35 text-info"><i class="bx bx-info-circle"></i>
                        </div>
                        <div class="ms-3">
                            <h6 class="mb-0 text-info">Info</h6>
                            <div>This will delete <span id="deleteCount">0</span> activity logs.</div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-danger" onclick="performBulkDelete()" id="bulkDeleteBtn" disabled>
                    <i class="bx bx-trash"></i> Delete Logs
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Export Modal -->
<div class="modal fade" id="exportModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Export Activity Logs</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="mb-3">
                    <label class="form-label">Export Format</label>
                    <select class="form-select" id="exportFormat">
                        <option value="csv">CSV (Comma Separated Values)</option>
                        <option value="json">JSON (JavaScript Object Notation)</option>
                        <option value="excel">Excel Spreadsheet</option>
                    </select>
                </div>
                <div class="mb-3">
                    <label class="form-label">Export Range</label>
                    <select class="form-select" id="exportRange">
                        <option value="filtered">Current filtered results</option>
                        <option value="selected">Selected logs only</option>
                        <option value="all">All activity logs</option>
                        <option value="custom">Custom date range</option>
                    </select>
                </div>
                <div id="customDateRange" style="display: none;">
                    <div class="row">
                        <div class="col-md-6">
                            <label class="form-label">From Date</label>
                            <input type="date" class="form-control" id="exportDateFrom">
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">To Date</label>
                            <input type="date" class="form-control" id="exportDateTo">
                        </div>
                    </div>
                </div>
                <div class="mb-3">
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" id="includeDetails">
                        <label class="form-check-label" for="includeDetails">
                            Include detailed properties
                        </label>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="performExport()">
                    <i class="bx bx-download"></i> Export
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Confirmation Modal -->
<div class="modal fade" id="confirmationModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Confirm Action</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p id="confirmationMessage"></p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" id="confirmActionBtn">Confirm</button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('script')
<script>
// Global variables
let currentPage = 1;
let perPage = 25;
let totalPages = 1;
let totalItems = 0;
let selectedLogs = new Set();
let currentLogId = null;
let currentFilters = {
    search: '',
    log_name: '',
    event: '',
    user_id: '',
    date_from: '',
    date_to: '',
    order_by: 'created_at',
    order_dir: 'desc'
};

// Initialize when page loads
document.addEventListener('DOMContentLoaded', function() {
    loadActivityLogs();
    loadStatistics();
    
    // Setup event listeners
    document.getElementById('perPageSelect').addEventListener('change', function() {
        perPage = parseInt(this.value);
        currentPage = 1;
        loadActivityLogs();
    });
    
    document.getElementById('refreshBtn').addEventListener('click', function() {
        loadActivityLogs();
        loadStatistics();
    });
    
    document.getElementById('searchBtn').addEventListener('click', applyFilters);
    document.getElementById('searchInput').addEventListener('keypress', function(e) {
        if (e.key === 'Enter') applyFilters();
    });
    
    // Clean modal listeners
    document.getElementById('cleanDays').addEventListener('input', function() {
        document.getElementById('cleanBtn').disabled = !document.getElementById('confirmClean').checked || !this.value;
    });
    
    document.getElementById('confirmClean').addEventListener('change', function() {
        document.getElementById('cleanBtn').disabled = !this.checked || !document.getElementById('cleanDays').value;
    });
    
    // Bulk delete modal listeners
    document.getElementById('bulkDeleteMethod').addEventListener('change', function() {
        const customDiv = document.getElementById('customCriteria');
        customDiv.style.display = this.value === 'custom' ? 'block' : 'none';
        updateBulkDeleteInfo();
    });
    
    document.getElementById('confirmBulkDelete').addEventListener('change', function() {
        document.getElementById('bulkDeleteBtn').disabled = !this.checked;
    });
    
    // Export modal listeners
    document.getElementById('exportRange').addEventListener('change', function() {
        const customDiv = document.getElementById('customDateRange');
        customDiv.style.display = this.value === 'custom' ? 'block' : 'none';
    });
    
    // Select all checkbox
    document.getElementById('masterCheckbox').addEventListener('change', function() {
        const checkboxes = document.querySelectorAll('.log-checkbox');
        checkboxes.forEach(checkbox => {
            checkbox.checked = this.checked;
            if (this.checked) {
                selectedLogs.add(checkbox.value);
            } else {
                selectedLogs.delete(checkbox.value);
            }
        });
        updateSelectedCount();
    });
});

// Load activity logs
function loadActivityLogs() {
    const tbody = document.getElementById('activitiesTableBody');
    tbody.innerHTML = `
        <tr>
            <td colspan="7" class="text-center py-5">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden">Loading...</span>
                </div>
                <p class="mt-2">Loading activity logs...</p>
            </td>
        </tr>
    `;
    
    const params = new URLSearchParams({
        page: currentPage,
        per_page: perPage,
        ...currentFilters
    });
    
    // Use correct route for AJAX requests
    fetch(`{{ route('activities.index') }}?${params.toString()}`, {
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
        },
    })
    .then(response => {
        if (!response.ok) {
            if (response.status === 403) {
                throw new Error('Unauthorized access. Please contact administrator.');
            }
            throw new Error('Network response was not ok');
        }
        return response.json();
    })
    .then(data => {
        if (data.success) {
            renderActivityLogs(data);
            updatePagination(data);
            updateSelectedCount();
        } else {
            throw new Error(data.message || 'Failed to load activity logs');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('error', 'Failed to load activity logs: ' + error.message);
        tbody.innerHTML = `
            <tr>
                <td colspan="7" class="text-center py-5 text-danger">
                    <i class="bx bx-error-circle bx-lg mb-3"></i>
                    <p>${error.message}</p>
                    <button class="btn btn-sm btn-primary" onclick="loadActivityLogs()">
                        <i class="bx bx-refresh"></i> Try Again
                    </button>
                </td>
            </tr>
        `;
    });
}

// Render activity logs table
function renderActivityLogs(data) {
    const tbody = document.getElementById('activitiesTableBody');
    
    if (!data.activities || data.activities.length === 0) {
        tbody.innerHTML = `
            <tr>
                <td colspan="7" class="text-center py-5 text-muted">
                    <i class="bx bx-inbox bx-lg mb-3"></i>
                    <h5>No activity logs found</h5>
                    <p class="mb-0">Try adjusting your filters or check back later</p>
                </td>
            </tr>
        `;
        return;
    }
    
    let html = '';
    data.activities.forEach(log => {
        const date = new Date(log.created_at);
        const isSelected = selectedLogs.has(log.id.toString());
        
        html += `
            <tr id="log-row-${log.id}" class="${isSelected ? 'table-active' : ''}">
                <td>
                    <input type="checkbox" class="form-check-input log-checkbox" 
                           value="${log.id}" 
                           ${isSelected ? 'checked' : ''}
                           onchange="toggleSelection(${log.id})">
                </td>
                <td>
                    <div class="d-flex flex-column">
                        <span class="fw-semibold">${date.toLocaleDateString()}</span>
                        <small class="text-muted">${date.toLocaleTimeString()}</small>
                    </div>
                </td>
                <td>
                    ${log.causer ? `
                    <div class="d-flex align-items-center">
                        <div class="user-avatar me-2">
                            ${getInitials(log.causer.name)}
                        </div>
                        <div>
                            <div class="fw-semibold">${log.causer.name}</div>
                            <small class="text-muted">${log.causer.email}</small>
                        </div>
                    </div>
                    ` : `
                    <div class="d-flex align-items-center">
                        <div class="user-avatar me-2 bg-secondary text-white">
                            <i class="bx bx-chip"></i>
                        </div>
                        <div class="fw-semibold">System</div>
                    </div>
                    `}
                </td>
                <td>
                    <div class="d-flex align-items-center">
                        <div class="activity-icon me-3 ${getBadgeColorClass(log.event)} text-white">
                            <i class="${getIconForLogType(log.log_name)}"></i>
                        </div>
                        <div>
                            <div class="fw-semibold">${log.description}</div>
                            ${log.subject ? `
                            <small class="text-muted">
                                ${log.subject.type}: ${log.subject.name}
                            </small>
                            ` : ''}
                        </div>
                    </div>
                </td>
                <td>
                    <span class="badge bg-light text-dark border">
                        ${log.log_name}
                    </span>
                </td>
                <td>
                    <span class="badge-event badge bg-${getBadgeColorClass(log.event)}">
                        ${log.event}
                    </span>
                </td>
                <td>
                    <div class="btn-group btn-group-sm" role="group">
                        <button type="button" class="btn btn-outline-primary" onclick="viewLogDetails(${log.id})" title="View Details">
                            <i class="bx bx-show"></i>
                        </button>
                        <button type="button" class="btn btn-outline-danger" onclick="deleteSingleLog(${log.id})" title="Delete">
                            <i class="bx bx-trash"></i>
                        </button>
                    </div>
                </td>
            </tr>
        `;
    });
    
    tbody.innerHTML = html;
}

// View log details
function viewLogDetails(id) {
    currentLogId = id;
    
    fetch(`{{ route('activities.show', '') }}/${id}`, {
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
        },
    })
    .then(response => {
        if (!response.ok) throw new Error('Network response was not ok');
        return response.json();
    })
    .then(data => {
        if (data.success) {
            const log = data.activity;
            const date = new Date(log.created_at);
            
            document.getElementById('detailDateTime').textContent = 
                `${date.toLocaleDateString()} ${date.toLocaleTimeString()} (${log.created_at_human})`;
            document.getElementById('detailUser').textContent = 
                log.causer ? `${log.causer.name} (${log.causer.email}) - ${log.causer.role}` : 'System';
            document.getElementById('detailLogType').textContent = log.log_name;
            document.getElementById('detailEvent').textContent = log.event_name;
            document.getElementById('detailDescription').textContent = log.description;
            document.getElementById('detailIpAddress').textContent = log.ip_address;
            document.getElementById('detailUserAgent').textContent = log.user_agent;
            document.getElementById('detailSubject').textContent = 
                log.subject ? `${log.subject.type} #${log.subject.id} (${log.subject.name})` : 'None';
            document.getElementById('detailProperties').textContent = log.formatted_properties;
            
            // Show modal
            const modal = new bootstrap.Modal(document.getElementById('detailsModal'));
            modal.show();
        } else {
            throw new Error(data.message || 'Failed to load log details');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('error', 'Failed to load log details: ' + error.message);
    });
}

// Delete single log
function deleteSingleLog(id) {
    currentLogId = id;
    
    showConfirmation(
        'Delete Activity Log',
        'Are you sure you want to delete this activity log? This action cannot be undone.',
        function() {
            performDelete(id);
        }
    );
}

// Delete log from details modal
function deleteLog() {
    if (!currentLogId) return;
    
    showConfirmation(
        'Delete Activity Log',
        'Are you sure you want to delete this activity log? This action cannot be undone.',
        function() {
            performDelete(currentLogId);
            const modal = bootstrap.Modal.getInstance(document.getElementById('detailsModal'));
            modal.hide();
        }
    );
}

// Perform delete operation
function performDelete(id) {
    fetch(`{{ route('activities.destroy', '') }}/${id}`, {
        method: 'DELETE',
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
            'Accept': 'application/json',
            'X-Requested-With': 'XMLHttpRequest',
        },
    })
    .then(response => {
        if (!response.ok) throw new Error('Network response was not ok');
        return response.json();
    })
    .then(data => {
        if (data.success) {
            showToast('success', data.message);
            // Remove from table
            const row = document.getElementById(`log-row-${id}`);
            if (row) row.remove();
            // Remove from selected logs
            selectedLogs.delete(id.toString());
            updateSelectedCount();
            // Update statistics
            loadStatistics();
        } else {
            throw new Error(data.message || 'Failed to delete log');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('error', 'Failed to delete log: ' + error.message);
    });
}

// Bulk delete
function showBulkDeleteModal() {
    const modal = new bootstrap.Modal(document.getElementById('bulkDeleteModal'));
    modal.show();
    updateBulkDeleteInfo();
}

function updateBulkDeleteInfo() {
    const method = document.getElementById('bulkDeleteMethod').value;
    let count = 0;
    
    if (method === 'selected') {
        count = selectedLogs.size;
    } else if (method === 'filtered') {
        // Get count from current filtered results
        count = totalItems;
    } else if (method === 'custom') {
        // Estimate based on criteria
        count = '?'; // You could make an API call to get count
    }
    
    document.getElementById('deleteCount').textContent = count;
    document.getElementById('bulkDeleteInfo').style.display = count > 0 ? 'block' : 'none';
}

function performBulkDelete() {
    const method = document.getElementById('bulkDeleteMethod').value;
    const filter = {};
    
    if (method === 'selected') {
        filter.ids = Array.from(selectedLogs);
    } else if (method === 'filtered') {
        filter.ids = [];
        filter.filter = currentFilters;
    } else if (method === 'custom') {
        filter.ids = [];
        filter.filter = {
            log_name: document.getElementById('bulkLogType').value || null,
            event: document.getElementById('bulkEvent').value || null,
            older_than_days: document.getElementById('bulkOlderThan').value || null
        };
    }
    
    fetch('{{ route("activities.bulk-destroy") }}', {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'X-Requested-With': 'XMLHttpRequest',
        },
        body: JSON.stringify(filter),
    })
    .then(response => {
        if (!response.ok) throw new Error('Network response was not ok');
        return response.json();
    })
    .then(data => {
        if (data.success) {
            showToast('success', data.message);
            // Close modal
            const modal = bootstrap.Modal.getInstance(document.getElementById('bulkDeleteModal'));
            modal.hide();
            // Reload data
            loadActivityLogs();
            loadStatistics();
            // Clear selection
            selectedLogs.clear();
            updateSelectedCount();
        } else {
            throw new Error(data.message || 'Failed to delete logs');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('error', 'Failed to delete logs: ' + error.message);
    });
}

// Clean old logs
function showCleanModal() {
    const modal = new bootstrap.Modal(document.getElementById('cleanModal'));
    modal.show();
}

function cleanOldLogs() {
    const days = document.getElementById('cleanDays').value;
    
    fetch('{{ route("activities.clean-old") }}', {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'X-Requested-With': 'XMLHttpRequest',
        },
        body: JSON.stringify({ days: days }),
    })
    .then(response => {
        if (!response.ok) throw new Error('Network response was not ok');
        return response.json();
    })
    .then(data => {
        if (data.success) {
            showToast('success', data.message);
            // Close modal
            const modal = bootstrap.Modal.getInstance(document.getElementById('cleanModal'));
            modal.hide();
            // Reload data
            loadActivityLogs();
            loadStatistics();
        } else {
            throw new Error(data.message || 'Failed to clean logs');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('error', 'Failed to clean logs: ' + error.message);
    });
}

// Export logs
function exportLogs(format) {
    const params = new URLSearchParams({
        format: format,
        ...currentFilters
    });
    
    window.location.href = `{{ route('activities.export') }}?${params.toString()}`;
}

function showExportModal() {
    const modal = new bootstrap.Modal(document.getElementById('exportModal'));
    modal.show();
}

function performExport() {
    const format = document.getElementById('exportFormat').value;
    const range = document.getElementById('exportRange').value;
    const includeDetails = document.getElementById('includeDetails').checked;
    
    let params = new URLSearchParams({
        format: format,
        export_type: range,
        include_details: includeDetails ? '1' : '0'
    });
    
    if (range === 'selected') {
        params.append('ids', Array.from(selectedLogs).join(','));
    } else if (range === 'custom') {
        params.append('start_date', document.getElementById('exportDateFrom').value);
        params.append('end_date', document.getElementById('exportDateTo').value);
    }
    
    window.location.href = `{{ route('activities.export') }}?${params.toString()}`;
    
    // Close modal
    const modal = bootstrap.Modal.getInstance(document.getElementById('exportModal'));
    modal.hide();
}

// Selection management
function toggleSelection(id) {
    const checkbox = document.querySelector(`.log-checkbox[value="${id}"]`);
    const row = document.getElementById(`log-row-${id}`);
    
    if (checkbox.checked) {
        selectedLogs.add(id.toString());
        row.classList.add('table-active');
    } else {
        selectedLogs.delete(id.toString());
        row.classList.remove('table-active');
    }
    
    // Update master checkbox
    const totalCheckboxes = document.querySelectorAll('.log-checkbox').length;
    const checkedCheckboxes = document.querySelectorAll('.log-checkbox:checked').length;
    document.getElementById('masterCheckbox').checked = totalCheckboxes > 0 && totalCheckboxes === checkedCheckboxes;
    
    updateSelectedCount();
}

function selectAll() {
    const checkboxes = document.querySelectorAll('.log-checkbox');
    const rows = document.querySelectorAll('#activitiesTableBody tr');
    
    checkboxes.forEach(checkbox => {
        checkbox.checked = true;
        selectedLogs.add(checkbox.value);
    });
    
    rows.forEach(row => {
        row.classList.add('table-active');
    });
    
    document.getElementById('masterCheckbox').checked = true;
    updateSelectedCount();
}

function deselectAll() {
    const checkboxes = document.querySelectorAll('.log-checkbox');
    const rows = document.querySelectorAll('#activitiesTableBody tr');
    
    checkboxes.forEach(checkbox => {
        checkbox.checked = false;
        selectedLogs.delete(checkbox.value);
    });
    
    rows.forEach(row => {
        row.classList.remove('table-active');
    });
    
    document.getElementById('masterCheckbox').checked = false;
    updateSelectedCount();
}

function deleteSelected() {
    if (selectedLogs.size === 0) {
        showToast('warning', 'No logs selected');
        return;
    }
    
    showConfirmation(
        'Delete Selected Logs',
        `Are you sure you want to delete ${selectedLogs.size} selected activity logs? This action cannot be undone.`,
        function() {
            const ids = Array.from(selectedLogs);
            
            fetch('{{ route("activities.bulk-destroy") }}', {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                    'Accept': 'application/json',
                    'Content-Type': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest',
                },
                body: JSON.stringify({ ids: ids }),
            })
            .then(response => {
                if (!response.ok) throw new Error('Network response was not ok');
                return response.json();
            })
            .then(data => {
                if (data.success) {
                    showToast('success', data.message);
                    // Reload data
                    loadActivityLogs();
                    loadStatistics();
                    // Clear selection
                    selectedLogs.clear();
                    updateSelectedCount();
                } else {
                    throw new Error(data.message || 'Failed to delete logs');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showToast('error', 'Failed to delete logs: ' + error.message);
            });
        }
    );
}

function exportSelected() {
    if (selectedLogs.size === 0) {
        showToast('warning', 'No logs selected for export');
        return;
    }
    
    const ids = Array.from(selectedLogs).join(',');
    window.location.href = `{{ route('activities.export') }}?format=csv&export_type=selected&ids=${ids}`;
}

function updateSelectedCount() {
    const deleteBtn = document.getElementById('deleteSelectedBtn');
    deleteBtn.disabled = selectedLogs.size === 0;
    
    if (selectedLogs.size > 0) {
        deleteBtn.innerHTML = `<i class="bx bx-trash"></i> Delete Selected (${selectedLogs.size})`;
    } else {
        deleteBtn.innerHTML = `<i class="bx bx-trash"></i> Delete Selected`;
    }
}

// Filter management
function applyFilters() {
    currentFilters = {
        search: document.getElementById('searchInput').value,
        log_name: document.getElementById('logTypeFilter').value,
        event: document.getElementById('eventFilter').value,
        user_id: document.getElementById('userFilter').value,
        date_from: document.getElementById('dateFromFilter').value,
        date_to: document.getElementById('dateToFilter').value,
        order_by: 'created_at',
        order_dir: 'desc'
    };
    
    currentPage = 1;
    loadActivityLogs();
}

function resetFilters() {
    document.getElementById('searchInput').value = '';
    document.getElementById('logTypeFilter').value = '';
    document.getElementById('eventFilter').value = '';
    document.getElementById('userFilter').value = '';
    document.getElementById('dateFromFilter').value = '';
    document.getElementById('dateToFilter').value = '';
    
    currentFilters = {
        search: '',
        log_name: '',
        event: '',
        user_id: '',
        date_from: '',
        date_to: '',
        order_by: 'created_at',
        order_dir: 'desc'
    };
    
    currentPage = 1;
    loadActivityLogs();
}

// Pagination
function updatePagination(data) {
    totalPages = data.last_page || 1;
    totalItems = data.total || 0;
    currentPage = data.current_page || 1;
    perPage = data.per_page || 25;
    
    // Update showing text
    const from = data.from || ((currentPage - 1) * perPage) + 1;
    const to = data.to || Math.min(currentPage * perPage, totalItems);
    
    document.getElementById('showingFrom').textContent = from;
    document.getElementById('showingTo').textContent = to;
    document.getElementById('totalItems').textContent = totalItems;
    
    // Update pagination controls
    const container = document.getElementById('paginationContainer');
    let html = '';
    
    // Previous button
    html += `
        <li class="page-item ${currentPage === 1 ? 'disabled' : ''}">
            <a class="page-link" href="#" onclick="changePage(${currentPage - 1})" tabindex="-1">Previous</a>
        </li>
    `;
    
    // Page numbers
    const maxPagesToShow = 5;
    let startPage = Math.max(1, currentPage - Math.floor(maxPagesToShow / 2));
    let endPage = Math.min(totalPages, startPage + maxPagesToShow - 1);
    
    if (endPage - startPage + 1 < maxPagesToShow) {
        startPage = Math.max(1, endPage - maxPagesToShow + 1);
    }
    
    if (startPage > 1) {
        html += `<li class="page-item"><a class="page-link" href="#" onclick="changePage(1)">1</a></li>`;
        if (startPage > 2) {
            html += `<li class="page-item disabled"><span class="page-link">...</span></li>`;
        }
    }
    
    for (let i = startPage; i <= endPage; i++) {
        html += `
            <li class="page-item ${i === currentPage ? 'active' : ''}">
                <a class="page-link" href="#" onclick="changePage(${i})">${i}</a>
            </li>
        `;
    }
    
    if (endPage < totalPages) {
        if (endPage < totalPages - 1) {
            html += `<li class="page-item disabled"><span class="page-link">...</span></li>`;
        }
        html += `<li class="page-item"><a class="page-link" href="#" onclick="changePage(${totalPages})">${totalPages}</a></li>`;
    }
    
    // Next button
    html += `
        <li class="page-item ${currentPage === totalPages ? 'disabled' : ''}">
            <a class="page-link" href="#" onclick="changePage(${currentPage + 1})">Next</a>
        </li>
    `;
    
    container.innerHTML = html;
}

function changePage(page) {
    if (page < 1 || page > totalPages || page === currentPage) return;
    currentPage = page;
    loadActivityLogs();
}

// Statistics
function loadStatistics() {
    fetch('{{ route("activities.statistics") }}', {
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
        },
    })
    .then(response => {
        if (!response.ok) {
            throw new Error('Failed to load statistics');
        }
        return response.json();
    })
    .then(data => {
        if (data.success) {
            const stats = data.statistics;
            document.getElementById('totalLogs').textContent = stats.total.toLocaleString();
            document.getElementById('todayLogs').textContent = stats.today.toLocaleString();
            document.getElementById('activeUsers').textContent = stats.active_users.toLocaleString();
            document.getElementById('avgDaily').textContent = stats.average_daily.toFixed(1);
            
            // Show stats cards
            document.getElementById('statsCards').style.display = 'flex';
        }
    })
    .catch(error => {
        console.error('Error loading statistics:', error);
        // Optionally hide stats cards or show error
        document.getElementById('statsCards').style.display = 'none';
    });
}

// Helper functions
function getInitials(name) {
    if (!name) return '?';
    return name.split(' ').map(n => n[0]).join('').toUpperCase().substring(0, 2);
}

function getIconForLogType(logName) {
    const icons = {
        'default': 'bx bx-history',
        'user': 'bx bx-user',
        'product': 'bx bx-box',
        'category': 'bx bx-tag',
        'customer': 'bx bx-group',
        'supplier': 'bx bx-truck',
        'sale': 'bx bx-cart',
        'purchase': 'bx bx-shopping-bag',
        'stock': 'bx bx-package',
        'payment': 'bx bx-credit-card',
        'mpesa': 'bx bx-mobile',
        'system': 'bx bx-cog',
        'auth': 'bx bx-shield',
        'error': 'bx bx-error-circle',
        'audit': 'bx bx-clipboard',
    };
    
    return icons[logName] || icons['default'];
}

function getBadgeColorClass(event) {
    const colors = {
        'created': 'success',
        'updated': 'warning',
        'deleted': 'danger',
        'login': 'info',
        'logout': 'secondary',
        'error': 'danger',
        'warning': 'warning',
        'info': 'info',
        'success': 'success',
        'default': 'primary',
    };
    
    return colors[event] || colors['default'];
}

// Toast notifications
function showToast(type, message) {
    // Create toast container if not exists
    let container = document.getElementById('toastContainer');
    if (!container) {
        container = document.createElement('div');
        container.id = 'toastContainer';
        container.className = 'position-fixed top-0 end-0 p-3';
        container.style.zIndex = '1055';
        document.body.appendChild(container);
    }
    
    const toastId = 'toast-' + Date.now();
    const toastHtml = `
        <div id="${toastId}" class="toast align-items-center text-white bg-${type} border-0" role="alert" aria-live="assertive" aria-atomic="true">
            <div class="d-flex">
                <div class="toast-body">
                    ${message}
                </div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Close"></button>
            </div>
        </div>
    `;
    
    container.insertAdjacentHTML('beforeend', toastHtml);
    
    const toastElement = document.getElementById(toastId);
    const toast = new bootstrap.Toast(toastElement, { delay: 5000 });
    toast.show();
    
    // Remove toast after it's hidden
    toastElement.addEventListener('hidden.bs.toast', function() {
        toastElement.remove();
    });
}

// Confirmation modal
function showConfirmation(title, message, confirmCallback) {
    document.getElementById('confirmationMessage').textContent = message;
    
    const modal = new bootstrap.Modal(document.getElementById('confirmationModal'));
    modal.show();
    
    // Set up confirm button
    const confirmBtn = document.getElementById('confirmActionBtn');
    confirmBtn.onclick = function() {
        modal.hide();
        if (confirmCallback) confirmCallback();
    };
}
</script>
@endsection