// =========================================================
// POS System Main Entry Point - Production Grade v3.0.0
// =========================================================

// Add this at the top of your pos-main.js file
window.POSPayments = window.POSPayments || {
    showModal: function() {
        console.log('POSPayments.showModal called');
        const modal = document.getElementById('paymentModal');
        if (modal) {
            try {
                const bsModal = new bootstrap.Modal(modal);
                bsModal.show();
            } catch (e) {
                modal.style.display = 'block';
                modal.classList.add('show');
                modal.setAttribute('aria-modal', 'true');
                modal.setAttribute('style', 'display: block; padding-right: 17px;');
                document.body.classList.add('modal-open');
            }
        }
    },
    selectMethod: function(method) {
        console.log('Selecting payment method:', method);
        // Select the payment method card
        document.querySelectorAll('.payment-method-card').forEach(card => {
            card.classList.remove('selected');
        });
        
        const selectedCard = document.querySelector(`[data-method="${method}"]`);
        if (selectedCard) {
            selectedCard.classList.add('selected');
            
            // Show the form for this method
            document.querySelectorAll('.payment-method-form').forEach(form => {
                form.classList.add('hidden');
            });
            
            const form = document.getElementById(`${method}Form`);
            if (form) {
                form.classList.remove('hidden');
            }
        }
    }
};

class POSSystem {
    constructor() {
        this.version = '3.0.0';
        this.modules = new Map();
        this.initialized = false;
        this.initializationPhase = 'not_started';
        this.startTime = Date.now();
        
        // Enhanced module dependencies with priorities
        this.moduleDependencies = {
            // Core utilities (priority 1)
            'POSUtils': { deps: [], priority: 1, critical: true },
            
            // State management (priority 2)
            'POSState': { deps: ['POSUtils'], priority: 2, critical: true },
            
            // Core functionality (priority 3)
            'POSCart': { deps: ['POSUtils', 'POSState'], priority: 3, critical: true },
            'POSProducts': { deps: ['POSUtils', 'POSState'], priority: 3, critical: true },
            
            // Payment system (priority 4)
            'PaymentManager': { deps: ['POSUtils', 'POSState'], priority: 4, critical: true },
            'POSPayments': { deps: ['POSUtils', 'POSState', 'PaymentManager'], priority: 4, critical: false },
            
            // Additional modules (priority 5)
            'POSCustomers': { deps: ['POSUtils', 'POSState'], priority: 5, critical: false },
            'POSScanner': { deps: ['POSUtils'], priority: 5, critical: false },
            'POSDiscounts': { deps: ['POSUtils', 'POSState', 'POSCart'], priority: 5, critical: false },
            'POSHeldSales': { deps: ['POSUtils', 'POSState'], priority: 5, critical: false },
            'POSStats': { deps: ['POSUtils', 'POSState'], priority: 5, critical: false },
            'POSReceipts': { deps: ['POSUtils', 'POSState'], priority: 5, critical: false },
            'ETIMSIntegration': { deps: ['POSUtils'], priority: 6, critical: false }
        };
        
        // Enhanced configuration
        this.config = {
            // System
            debug: window.location.hostname === 'localhost' || window.location.hostname === '127.0.0.1',
            version: this.version,
            environment: window.location.hostname.includes('localhost') ? 'development' : 'production',
            
            // Performance
            cacheEnabled: true,
            apiTimeout: 15000,
            retryAttempts: 3,
            retryDelay: 1000,
            autoSaveInterval: 30000,
            syncInterval: 60000,
            
            // UI
            animationDuration: 300,
            toastDuration: 5000,
            loadingMinDuration: 500,
            
            // Features
            enableOfflineMode: true,
            enableAutoSave: true,
            enableAnalytics: true,
            enableErrorReporting: true,
            
            // Payment
            paymentMethods: ['cash', 'mpesa', 'card', 'credit', 'split', 'bank'],
            defaultPaymentMethod: 'cash',
            requirePaymentConfirmation: true,
            enableReceiptPrinting: true,
            enableEtimsIntegration: true
        };
        
        // State tracking
        this.status = {
            modules: {},
            errors: [],
            warnings: [],
            performance: {},
            network: navigator.onLine ? 'online' : 'offline'
        };
        
        // Initialize core utilities immediately
        this.setupImmediateUtilities();
    }

    // ========== INITIALIZATION ==========
    async init() {
        if (this.initialized) {
            console.log('🔄 POS System already initialized');
            return this;
        }
        
        this.initializationPhase = 'starting';
        console.log(`🚀 Starting POS System v${this.version}...`);
        
        // Show loading overlay
        this.showLoading(true, 'Initializing POS System...');
        
        try {
            // Record start time
            this.startTime = Date.now();
            
            // Phase 1: Setup error handling and core systems
            this.initializationPhase = 'setup';
            await this.setupCoreSystems();
            
            // Phase 2: Load modules
            this.initializationPhase = 'loading_modules';
            await this.loadModules();
            
            // Phase 3: Initialize modules
            this.initializationPhase = 'initializing_modules';
            await this.initializeModules();
            
            // Phase 4: Load initial data
            this.initializationPhase = 'loading_data';
            await this.loadInitialData();
            
            // Phase 5: Setup background services
            this.initializationPhase = 'setup_services';
            await this.setupBackgroundServices();
            
            // Mark as initialized
            this.initialized = true;
            this.initializationPhase = 'complete';
            
            // Calculate initialization time
            const initTime = Date.now() - this.startTime;
            this.status.performance.initializationTime = initTime;
            
            console.log(`✅ POS System v${this.version} fully initialized in ${initTime}ms`);
            
            // Show success message
            this.showLoading(false);
            this.showToast('POS System Ready!', 'success', 2000);
            
            // Track initialization
            this.trackEvent('system_initialized', {
                version: this.version,
                initTime: initTime,
                modules: Object.keys(this.status.modules).filter(m => this.status.modules[m].loaded),
                environment: this.config.environment
            });
            
            // Start performance monitoring
            this.startPerformanceMonitoring();
            
        } catch (error) {
            this.initializationPhase = 'failed';
            console.error('❌ POS System initialization failed:', error);
            
            // Show error to user
            this.showLoading(false);
            this.showFallback(`System Error: ${error.message}`);
            
            // Try to recover with minimal functionality
            this.setupFallbackMode();
            
            throw error;
        }
        
        return this;
    }

    // ========== CORE SYSTEM SETUP ==========
    async setupCoreSystems() {
        console.log('⚙️ Setting up core systems...');
        
        // 1. Error handling
        this.setupErrorHandling();
        
        // 2. Global event listeners
        this.setupGlobalEvents();
        
        // 3. Network monitoring
        this.setupNetworkMonitoring();
        
        // 4. Performance monitoring
        this.setupPerformanceMonitoring();
        
        // 5. Service Worker registration (if available)
        await this.registerServiceWorker();
        
        console.log('✅ Core systems setup complete');
    }

    setupImmediateUtilities() {
        // Setup immediate utilities that don't require async loading
        window.POS = this;
        window.posSystem = this;
        
        // Safe global functions
        window.safeShowPaymentModal = () => this.openPayment();
        window.safeNewSale = () => this.newSale();
        window.safeOpenScanner = () => this.openScanner();
        window.safeShowHeldSales = () => window.POSHeldSales?.showModal?.() || this.showToast('Held sales not available', 'warning');
        
        // Emergency functions
        window.emergencyReset = () => this.emergencyReset();
        window.forceReload = () => location.reload();
        
        console.log('⚡ Immediate utilities setup');
    }

    setupErrorHandling() {
        console.log('🛡️ Setting up error handling...');
        
        // Global error handler
        const originalOnError = window.onerror;
        window.onerror = (msg, url, lineNo, columnNo, error) => {
            const errorDetails = {
                message: msg,
                url: url,
                line: lineNo,
                column: columnNo,
                error: error,
                timestamp: new Date().toISOString(),
                phase: this.initializationPhase,
                modules: this.status.modules
            };
            
            console.error('🚨 Global Error:', errorDetails);
            this.trackError(error || new Error(msg), errorDetails);
            
            // Show user-friendly error if critical
            if (this.isCriticalError(error || msg)) {
                this.showErrorToast('A system error occurred. Some features may not work properly.');
            }
            
            // Call original handler if exists
            if (originalOnError) {
                return originalOnError(msg, url, lineNo, columnNo, error);
            }
            
            return false;
        };

        // Unhandled promise rejection
        window.addEventListener('unhandledrejection', (event) => {
            console.error('🚨 Unhandled Promise Rejection:', event.reason);
            this.trackError(event.reason, { type: 'unhandled_rejection' });
            
            // Show warning toast for user-facing promises
            if (this.isUserFacingError(event.reason)) {
                this.showWarningToast('An operation failed. Please try again.');
            }
        });

        // Console error interception
        if (this.config.debug) {
            const originalConsoleError = console.error;
            console.error = (...args) => {
                this.trackError(new Error(args[0]), { consoleArgs: args });
                originalConsoleError.apply(console, args);
            };
        }
    }

    setupGlobalEvents() {
        console.log('🎮 Setting up global event listeners...');
        
        // Keyboard shortcuts
        document.addEventListener('keydown', this.handleKeyboardShortcuts.bind(this));
        
        // Before unload warning
        window.addEventListener('beforeunload', (e) => {
            if (window.POSState?.cart?.length > 0) {
                e.preventDefault();
                e.returnValue = 'You have unsaved items in your cart. Are you sure you want to leave?';
                
                // Auto-save before leaving
                this.saveState();
                return e.returnValue;
            }
        });
        
        // Visibility change (save state when tab hidden)
        document.addEventListener('visibilitychange', () => {
            if (document.hidden) {
                this.saveState();
                this.trackEvent('app_backgrounded');
            } else {
                this.trackEvent('app_foregrounded');
            }
        });
        
        // Online/offline detection
        window.addEventListener('online', () => {
            this.status.network = 'online';
            this.showToast('Back online', 'success');
            this.trackEvent('network_online');
            
            // Trigger sync if needed
            this.triggerNetworkSync();
        });
        
        window.addEventListener('offline', () => {
            this.status.network = 'offline';
            this.showToast('Working offline', 'warning');
            this.trackEvent('network_offline');
        });
        
        // Resize events for responsive adjustments
        let resizeTimeout;
        window.addEventListener('resize', () => {
            clearTimeout(resizeTimeout);
            resizeTimeout = setTimeout(() => {
                this.handleResize();
            }, 250);
        });
        
        // Click outside modal handler
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('modal-backdrop')) {
                this.closeAllModals();
            }
        });
        
        console.log('✅ Global event listeners setup complete');
    }

    setupNetworkMonitoring() {
        console.log('📡 Setting up network monitoring...');
        
        // Network status indicator
        this.createNetworkStatusIndicator();
        
        // Periodic network check
        setInterval(() => {
            const wasOnline = this.status.network === 'online';
            const isNowOnline = navigator.onLine;
            
            if (wasOnline !== isNowOnline) {
                this.status.network = isNowOnline ? 'online' : 'offline';
                this.trackEvent(`network_${isNowOnline ? 'online' : 'offline'}`);
            }
        }, 5000);
    }

    setupPerformanceMonitoring() {
        console.log('📊 Setting up performance monitoring...');
        
        // Performance observer for long tasks
        if ('PerformanceObserver' in window) {
            try {
                const perfObserver = new PerformanceObserver((list) => {
                    for (const entry of list.getEntries()) {
                        if (entry.duration > 50) { // Long task threshold
                            this.trackEvent('long_task', {
                                duration: entry.duration,
                                name: entry.name
                            });
                        }
                    }
                });
                
                perfObserver.observe({ entryTypes: ['longtask'] });
            } catch (e) {
                console.warn('PerformanceObserver not supported:', e);
            }
        }
        
        // Memory monitoring (if available)
        if (performance.memory) {
            setInterval(() => {
                const memory = performance.memory;
                const usedMB = Math.round(memory.usedJSHeapSize / 1048576);
                const totalMB = Math.round(memory.totalJSHeapSize / 1048576);
                
                if (usedMB > 100) { // 100MB threshold
                    this.trackEvent('high_memory', { usedMB, totalMB });
                }
            }, 30000);
        }
    }

    async registerServiceWorker() {
        if ('serviceWorker' in navigator && this.config.environment === 'production') {
            try {
                const registration = await navigator.serviceWorker.register('/service-worker.js');
                console.log('✅ Service Worker registered:', registration);
                
                registration.addEventListener('updatefound', () => {
                    const newWorker = registration.installing;
                    console.log('🔄 Service Worker update found');
                    
                    newWorker.addEventListener('statechange', () => {
                        if (newWorker.state === 'installed' && navigator.serviceWorker.controller) {
                            this.showToast('New version available. Refresh to update.', 'info');
                        }
                    });
                });
            } catch (error) {
                console.warn('Service Worker registration failed:', error);
            }
        }
    }

    // ========== MODULE MANAGEMENT ==========
 async loadModules() {
    console.log('📦 Loading modules...');
    
    const loadedModules = new Set();
    const failedModules = new Map(); // Map to track failures with reasons
    const loadingQueue = [];
    const moduleGraph = this.buildDependencyGraph();
    
    // Track module states
    const moduleStates = new Map();
    for (const [moduleName, config] of Object.entries(this.moduleDependencies)) {
        moduleStates.set(moduleName, {
            name: moduleName,
            config,
            dependencies: new Set(config.deps || []),
            dependents: new Set(),
            loaded: false,
            failed: false,
            error: null,
            priority: config.priority || 99,
            critical: config.critical || false
        });
    }
    
    // Build dependency graph
    for (const [moduleName, state] of moduleStates) {
        for (const dep of state.dependencies) {
            const depState = moduleStates.get(dep);
            if (depState) {
                depState.dependents.add(moduleName);
            }
        }
    }
    
    // Sort modules by priority and dependencies
    const sortedModules = this.topologicalSort(moduleStates);
    
    console.log(`📊 Loading ${sortedModules.length} modules in optimal order`);
    
    // Load modules in batches (parallel where possible)
    const batchSize = 3; // Load 3 modules at a time
    for (let i = 0; i < sortedModules.length; i += batchSize) {
        const batch = sortedModules.slice(i, i + batchSize);
        console.log(`🔄 Loading batch ${Math.floor(i/batchSize) + 1}: ${batch.map(m => m.name).join(', ')}`);
        
        const batchPromises = batch.map(moduleState => 
            this.loadModuleWithState(moduleState, loadedModules, failedModules)
        );
        
        // Wait for batch to complete before next batch
        const batchResults = await Promise.allSettled(batchPromises);
        
        // Update loaded modules set
        batchResults.forEach((result, index) => {
            const moduleState = batch[index];
            if (result.status === 'fulfilled' && result.value) {
                loadedModules.add(moduleState.name);
                moduleState.loaded = true;
            } else {
                moduleState.failed = true;
                moduleState.error = result.reason?.message || 'Unknown error';
                failedModules.set(moduleState.name, {
                    error: moduleState.error,
                    critical: moduleState.critical,
                    dependencies: Array.from(moduleState.dependencies)
                });
            }
        });
        
        // Check if critical failures in this batch prevent further loading
        const criticalFailures = batch.filter(m => m.failed && m.critical);
        if (criticalFailures.length > 0) {
            console.error('❌ Critical module failures detected:', 
                criticalFailures.map(m => m.name));
            
            // Try to load fallbacks for critical modules
            await this.loadCriticalFallbacks(criticalFailures, loadedModules);
            
            // If still failed, stop loading dependent modules
            const stillCriticalFailed = criticalFailures.filter(m => m.failed);
            if (stillCriticalFailed.length > 0) {
                this.handleCriticalFailure(stillCriticalFailed, moduleStates);
                break;
            }
        }
    }
    
    // Report final loading results
    this.reportModuleLoadingResults(loadedModules, failedModules, moduleStates);
    
    return {
        loaded: Array.from(loadedModules),
        failed: Array.from(failedModules.keys()),
        details: {
            total: sortedModules.length,
            successful: loadedModules.size,
            failed: failedModules.size,
            criticalFailures: Array.from(failedModules.entries())
                .filter(([_, info]) => info.critical)
                .map(([name]) => name)
        }
    };

    
}

/**
 * Build dependency graph for modules
 */
buildDependencyGraph() {
    const graph = new Map();
    
    for (const [moduleName, config] of Object.entries(this.moduleDependencies)) {
        graph.set(moduleName, {
            name: moduleName,
            dependencies: new Set(config.deps || []),
            dependents: new Set(),
            config: config
        });
    }
    
    // Populate dependents
    for (const [moduleName, node] of graph) {
        for (const dep of node.dependencies) {
            const depNode = graph.get(dep);
            if (depNode) {
                depNode.dependents.add(moduleName);
            }
        }
    }
    
    return graph;
}

/**
 * Topological sort for module loading order
 */
topologicalSort(moduleStates) {
    const sorted = [];
    const visited = new Set();
    const temp = new Set();
    
    const visit = (moduleName) => {
        if (temp.has(moduleName)) {
            throw new Error(`Circular dependency detected involving ${moduleName}`);
        }
        
        if (!visited.has(moduleName)) {
            temp.add(moduleName);
            const state = moduleStates.get(moduleName);
            
            // Visit dependencies first
            for (const dep of state.dependencies) {
                visit(dep);
            }
            
            temp.delete(moduleName);
            visited.add(moduleName);
            sorted.push(state);
        }
    };
    
    // Sort modules by priority first, then visit
    const moduleNames = Array.from(moduleStates.keys())
        .sort((a, b) => {
            const stateA = moduleStates.get(a);
            const stateB = moduleStates.get(b);
            return stateA.priority - stateB.priority;
        });
    
    for (const moduleName of moduleNames) {
        visit(moduleName);
    }
    
    return sorted;
}

/**
 * Load a single module with its state
 */
async loadModuleWithState(moduleState, loadedModules, failedModules) {
    const { name, config } = moduleState;
    
    try {
        // Verify all dependencies are loaded
        for (const dep of moduleState.dependencies) {
            if (!loadedModules.has(dep)) {
                throw new Error(`Dependency "${dep}" not loaded. Required for ${name}`);
            }
        }
        
        console.log(`📦 Loading ${name} (priority ${config.priority})...`);
        
        // Load the module
        const module = await this.loadSingleModule(name, config);
        
        if (!module) {
            throw new Error(`Module "${name}" failed to load (null result)`);
        }
        
        // Store the module
        this.modules.set(name, module);
        
        // Update status
        this.status.modules[name] = {
            loaded: true,
            initialized: false,
            priority: config.priority,
            critical: config.critical,
            timestamp: Date.now(),
            dependencies: Array.from(moduleState.dependencies)
        };
        
        console.log(`✅ Module loaded: ${name}`);
        
        // Attempt to initialize if it has an init method
        if (typeof module.init === 'function') {
            try {
                console.log(`⚡ Initializing ${name}...`);
                await module.init();
                this.status.modules[name].initialized = true;
                this.status.modules[name].initTime = Date.now();
                console.log(`✅ Module initialized: ${name}`);
            } catch (initError) {
                console.warn(`⚠️ Module ${name} loaded but initialization failed:`, initError.message);
                this.status.modules[name].initialized = false;
                this.status.modules[name].initError = initError.message;
                // Don't fail the module load - it might still be usable
            }
        }
        
        return module;
        
    } catch (error) {
        console.error(`❌ Failed to load module ${name}:`, error.message);
        
        // Store failure information
        this.status.modules[name] = {
            loaded: false,
            error: error.message,
            critical: config.critical,
            timestamp: Date.now(),
            dependencies: Array.from(moduleState.dependencies),
            stack: error.stack
        };
        
        // If critical, try immediate fallback
        if (config.critical) {
            try {
                console.warn(`🔄 Attempting immediate fallback for critical module: ${name}`);
                const fallback = await this.loadModuleFallback(name);
                if (fallback) {
                    this.modules.set(name, fallback);
                    this.status.modules[name].loaded = true;
                    this.status.modules[name].fallback = true;
                    console.log(`✅ Critical module ${name} loaded with fallback`);
                    return fallback;
                }
            } catch (fallbackError) {
                console.error(`❌ Fallback also failed for ${name}:`, fallbackError.message);
            }
        }
        
        throw error;
    }
}

/**
 * Load critical module fallbacks
 */
async loadCriticalFallbacks(criticalFailures, loadedModules) {
    console.warn('🔄 Loading fallbacks for critical modules...');
    
    const fallbackPromises = criticalFailures.map(async (moduleState) => {
        if (!moduleState.failed) return null;
        
        try {
            const fallback = await this.loadModuleFallback(moduleState.name);
            if (fallback) {
                this.modules.set(moduleState.name, fallback);
                loadedModules.add(moduleState.name);
                moduleState.loaded = true;
                moduleState.failed = false;
                moduleState.fallback = true;
                
                this.status.modules[moduleState.name] = {
                    loaded: true,
                    initialized: false,
                    fallback: true,
                    critical: true,
                    timestamp: Date.now(),
                    note: 'Using fallback implementation'
                };
                
                console.log(`✅ Critical fallback loaded: ${moduleState.name}`);
                return fallback;
            }
        } catch (error) {
            console.error(`❌ Fallback failed for ${moduleState.name}:`, error.message);
        }
        return null;
    });
    
    await Promise.allSettled(fallbackPromises);
}

/**
 * Handle critical failure scenario
 */
handleCriticalFailure(failedModules, moduleStates) {
    const criticalModuleNames = failedModules.map(m => m.name);
    
    console.error('🚨 CRITICAL SYSTEM FAILURE');
    console.error('Failed critical modules:', criticalModuleNames);
    
    // Mark dependent modules as blocked
    const blockedModules = new Set();
    for (const failedModule of failedModules) {
        this.markDependentsAsBlocked(failedModule.name, moduleStates, blockedModules);
    }
    
    console.warn(`⚠️ Blocked ${blockedModules.size} dependent modules`);
    
    // Show user-facing error
    this.showSystemAlert(`
        <div class="alert alert-danger">
            <h4><i class="bx bx-error-circle"></i> System Configuration Error</h4>
            <p>Critical system components failed to load:</p>
            <ul>
                ${criticalModuleNames.map(name => `<li><strong>${name}</strong></li>`).join('')}
            </ul>
            <p class="mb-0">Some features may be unavailable. Please refresh the page or contact support.</p>
            <div class="mt-3">
                <button class="btn btn-sm btn-outline-light" onclick="location.reload()">
                    <i class="bx bx-refresh"></i> Refresh Page
                </button>
                <button class="btn btn-sm btn-outline-light ms-2" onclick="this.closest('.alert').remove()">
                    <i class="bx bx-x"></i> Dismiss
                </button>
            </div>
        </div>
    `);
    
    // Try to load minimal functionality
    this.loadMinimalFunctionality();
}

/**
 * Mark dependent modules as blocked
 */
markDependentsAsBlocked(failedModuleName, moduleStates, blockedModules) {
    const state = moduleStates.get(failedModuleName);
    if (!state) return;
    
    for (const dependent of state.dependents) {
        if (!blockedModules.has(dependent)) {
            blockedModules.add(dependent);
            this.status.modules[dependent] = {
                loaded: false,
                blocked: true,
                blockedBy: failedModuleName,
                timestamp: Date.now()
            };
            // Recursively mark dependents
            this.markDependentsAsBlocked(dependent, moduleStates, blockedModules);
        }
    }
}

/**
 * Load minimal functionality when critical modules fail
 */
loadMinimalFunctionality() {
    console.warn('🔧 Loading minimal system functionality...');
    
    // Ensure basic utilities exist
    if (!window.POSUtils) {
        window.POSUtils = this.createUtilsFallback();
    }
    
    // Ensure basic state exists
    if (!window.posState) {
        window.posState = this.createStateFallback();
    }
    
    // Ensure basic cart functionality
    if (!window.POSCart) {
        window.POSCart = this.createCartFallback();
    }
    
    // Create emergency payment handler
    if (!window.showPaymentModal) {
        window.showPaymentModal = () => {
            const modal = document.getElementById('paymentModal');
            if (modal) {
                new bootstrap.Modal(modal).show();
            } else {
                alert('Payment system unavailable. Please contact support.');
            }
        };
    }
    
    this.showToast('System loaded with limited functionality', 'warning', 5000);
}

/**
 * Report module loading results
 */
reportModuleLoadingResults(loadedModules, failedModules, moduleStates) {
    const total = moduleStates.size;
    const successful = loadedModules.size;
    const failed = failedModules.size;
    const successRate = ((successful / total) * 100).toFixed(1);
    
    console.log('📊 ===== MODULE LOADING REPORT =====');
    console.log(`✅ Successfully loaded: ${successful}/${total} (${successRate}%)`);
    
    if (failed > 0) {
        console.log(`❌ Failed modules: ${failed}`);
        for (const [name, info] of failedModules) {
            const state = moduleStates.get(name);
            console.log(`   • ${name} (${state.critical ? 'CRITICAL' : 'non-critical'}): ${info.error}`);
            if (state.dependencies.size > 0) {
                console.log(`     Dependencies: ${Array.from(state.dependencies).join(', ')}`);
            }
        }
    }
    
    // Log module dependencies graph
    console.log('🕸️ Module dependency graph:');
    for (const state of moduleStates.values()) {
        if (state.dependencies.size > 0) {
            console.log(`   ${state.name} → ${Array.from(state.dependencies).join(', ')}`);
        }
    }
    
    console.log('====================================');
    
    // Show summary to user
    if (failed > 0) {
        const criticalFailed = Array.from(failedModules.entries())
            .filter(([_, info]) => info.critical)
            .map(([name]) => name);
        
        if (criticalFailed.length > 0) {
            this.showWarningToast(`${criticalFailed.length} critical modules failed. Some features disabled.`);
        } else {
            this.showInfoToast(`${failed} non-critical modules failed. Most features available.`);
        }
    }
}

/**
 * Show system alert
 */
showSystemAlert(html) {
    // Remove existing alerts
    const existing = document.getElementById('systemAlert');
    if (existing) existing.remove();
    
    // Create alert container
    const alertDiv = document.createElement('div');
    alertDiv.id = 'systemAlert';
    alertDiv.style.cssText = `
        position: fixed;
        top: 20px;
        left: 50%;
        transform: translateX(-50%);
        z-index: 99999;
        min-width: 400px;
        max-width: 600px;
    `;
    alertDiv.innerHTML = html;
    
    document.body.appendChild(alertDiv);
    
    // Auto-remove after 30 seconds
    setTimeout(() => {
        if (alertDiv.parentNode) {
            alertDiv.remove();
        }
    }, 30000);
}

    async loadSingleModule(moduleName, config) {
        // Check if module already exists in window
        let moduleClass = window[moduleName];
        
        if (!moduleClass) {
            // Try to load via dynamic import for ES6 modules
            if (moduleName.startsWith('Payment') || moduleName === 'POSPayments') {
                try {
                    // Payment modules use ES6 imports
                    const modulePath = this.getModulePath(moduleName);
                    if (modulePath) {
                        const module = await import(modulePath);
                        moduleClass = module.default || module;
                        window[moduleName] = moduleClass;
                    } else {
                        throw new Error(`No path found for module ${moduleName}`);
                    }
                } catch (importError) {
                    console.warn(`Dynamic import failed for ${moduleName}:`, importError);
                    // Fall back to script tag loading
                    moduleClass = window[moduleName];
                }
            }
            
            // If still not found, wait for script tag
            if (!moduleClass) {
                moduleClass = await this.waitForModule(moduleName, 5000);
            }
        }
        
        if (!moduleClass) {
            throw new Error(`Module ${moduleName} not found`);
        }
        
        // Create instance
        let instance;
        if (typeof moduleClass === 'function') {
            // Check if it's a class
            if (moduleClass.prototype && moduleClass.prototype.constructor) {
                instance = new moduleClass();
            } else {
                // Regular function, call it
                instance = moduleClass();
            }
        } else if (typeof moduleClass === 'object' && moduleClass !== null) {
            // Already an instance
            instance = moduleClass;
        } else {
            throw new Error(`Invalid module type for ${moduleName}`);
        }
        
        return instance;
    }

    async waitForModule(moduleName, timeout = 5000) {
        return new Promise((resolve) => {
            let elapsed = 0;
            const interval = 100;
            
            const check = () => {
                if (window[moduleName]) {
                    resolve(window[moduleName]);
                } else if (elapsed >= timeout) {
                    resolve(null);
                } else {
                    elapsed += interval;
                    setTimeout(check, interval);
                }
            };
            
            check();
        });
    }

    getModulePath(moduleName) {
        const paths = {
            'PaymentManager': '/js/pos/modules/payment/payment-manager.js',
            'POSPayments': '/js/pos/modules/payment/payments.js',
            'CashProcessor': '/js/pos/modules/payment/payment-processors/cash-processor.js',
            'MpesaProcessor': '/js/pos/modules/payment/payment-processors/mpesa-processor.js'
        };
        
        return paths[moduleName] || null;
    }

    async loadModuleFallback(moduleName) {
        console.warn(`🔄 Loading fallback for ${moduleName}`);
        
        switch(moduleName) {
            case 'POSUtils':
                window.POSUtils = this.createUtilsFallback();
                break;
                
            case 'POSState':
                window.POSState = this.createStateFallback();
                break;
                
            case 'POSCart':
                window.POSCart = this.createCartFallback();
                break;
                
            case 'PaymentManager':
                window.PaymentManager = this.createPaymentManagerFallback();
                break;
                
            case 'POSPayments':
                window.POSPayments = this.createPaymentsFallback();
                break;
                
            case 'POSProducts':
                window.POSProducts = this.createProductsFallback();
                break;
                
            default:
                console.warn(`No fallback available for ${moduleName}`);
                return null;
        }
        
        return window[moduleName];
    }

    async initializeModules() {
        console.log('⚡ Initializing modules...');
        
        const initializationOrder = Array.from(this.modules.entries())
            .sort((a, b) => {
                const prioA = this.moduleDependencies[a[0]]?.priority || 99;
                const prioB = this.moduleDependencies[b[0]]?.priority || 99;
                return prioA - prioB;
            });
        
        for (const [name, module] of initializationOrder) {
            try {
                if (typeof module.init === 'function') {
                    console.log(`⚡ Initializing: ${name}`);
                    
                    // Add timeout for initialization
                    const initPromise = module.init();
                    const timeoutPromise = new Promise((_, reject) => {
                        setTimeout(() => reject(new Error(`Initialization timeout for ${name}`)), 10000);
                    });
                    
                    await Promise.race([initPromise, timeoutPromise]);
                    
                    this.status.modules[name].initialized = true;
                    this.status.modules[name].initTime = Date.now();
                    
                    console.log(`✅ Initialized: ${name}`);
                } else {
                    console.log(`⏭️  No init required for: ${name}`);
                    this.status.modules[name].initialized = true;
                }
            } catch (error) {
                console.warn(`⚠️ Failed to initialize ${name}:`, error.message);
                this.status.modules[name].initialized = false;
                this.status.modules[name].initError = error.message;
                
                // If critical module fails, try to load minimal functionality
                if (this.moduleDependencies[name]?.critical) {
                    this.showWarningToast(`${name} initialization failed. Some features disabled.`);
                }
            }
        }
        
        console.log('✅ Module initialization complete');
    }

    // ========== DATA LOADING ==========
    async loadInitialData() {
        console.log('📊 Loading initial data...');
        
        const dataLoaders = [
            // Products
            () => this.modules.get('POSProducts')?.loadAll?.()
                .then(() => console.log('✅ Products loaded'))
                .catch(e => console.warn('⚠️ Products load failed:', e.message)),
            
            // Customers
            () => this.modules.get('POSCustomers')?.loadCustomers?.()
                .then(() => console.log('✅ Customers loaded'))
                .catch(e => console.warn('⚠️ Customers load failed:', e.message)),
            
            // Today's stats
            () => this.modules.get('POSStats')?.loadToday?.()
                .then(() => console.log('✅ Stats loaded'))
                .catch(e => console.warn('⚠️ Stats load failed:', e.message)),
            
            // Saved cart
            () => this.modules.get('POSCart')?.loadSavedCart?.()
                .then(() => console.log('✅ Saved cart loaded'))
                .catch(e => console.warn('⚠️ Saved cart load failed:', e.message)),
            
            // Payment methods
            () => this.modules.get('PaymentManager')?.loadPaymentMethods?.()
                .then(() => console.log('✅ Payment methods loaded'))
                .catch(e => console.warn('⚠️ Payment methods load failed:', e.message))
        ];
        
        // Load in parallel with error isolation
        await Promise.allSettled(dataLoaders.map(loader => loader()));
        
        console.log('✅ Initial data loading complete');
    }

    // ========== BACKGROUND SERVICES ==========
    async setupBackgroundServices() {
        console.log('🔄 Setting up background services...');
        
        // Auto-save
        if (this.config.enableAutoSave) {
            this.setupAutoSave();
        }
        
        // Auto-sync
        if (this.config.enableOfflineMode) {
            this.setupAutoSync();
        }
        
        // Health checks
        this.setupHealthChecks();
        
        // Cleanup old data
        this.setupDataCleanup();
        
        console.log('✅ Background services setup complete');
    }

    setupAutoSave() {
        console.log('💾 Setting up auto-save...');
        
        setInterval(() => {
            if (window.POSState?.cart?.length > 0) {
                this.saveState();
            }
        }, this.config.autoSaveInterval);
        
        // Also save on specific events
        if (window.POSState) {
            const originalAddItem = window.POSState.addItem;
            if (originalAddItem) {
                window.POSState.addItem = function(...args) {
                    const result = originalAddItem.apply(this, args);
                    setTimeout(() => window.posSystem?.saveState(), 1000);
                    return result;
                };
            }
            
            const originalRemoveItem = window.POSState.removeItem;
            if (originalRemoveItem) {
                window.POSState.removeItem = function(...args) {
                    const result = originalRemoveItem.apply(this, args);
                    setTimeout(() => window.posSystem?.saveState(), 1000);
                    return result;
                };
            }
        }
    }

    setupAutoSync() {
        console.log('🔄 Setting up auto-sync...');
        
        setInterval(() => {
            if (this.status.network === 'online') {
                this.syncPendingData();
            }
        }, this.config.syncInterval);
    }

    setupHealthChecks() {
        console.log('🏥 Setting up health checks...');
        
        // Periodic health check
        setInterval(() => {
            this.performHealthCheck();
        }, 60000); // Every minute
    }

    setupDataCleanup() {
        console.log('🧹 Setting up data cleanup...');
        
        // Clean old localStorage data weekly
        const lastCleanup = localStorage.getItem('pos_last_cleanup');
        const now = Date.now();
        const weekInMs = 7 * 24 * 60 * 60 * 1000;
        
        if (!lastCleanup || (now - parseInt(lastCleanup)) > weekInMs) {
            this.cleanupOldData();
            localStorage.setItem('pos_last_cleanup', now.toString());
        }
    }

    // ========== UTILITY METHODS ==========
    showLoading(show, message = 'Loading...') {
        const overlay = document.getElementById('loadingOverlay');
        if (!overlay) {
            // Create loading overlay if it doesn't exist
            const newOverlay = document.createElement('div');
            newOverlay.id = 'loadingOverlay';
            newOverlay.className = 'loading-overlay';
            newOverlay.innerHTML = `
                <div class="loading-content">
                    <div class="loading-spinner"></div>
                    <div class="loading-message">${message}</div>
                </div>
            `;
            document.body.appendChild(newOverlay);
        } else {
            overlay.style.display = show ? 'flex' : 'none';
            
            if (show && message) {
                const messageEl = overlay.querySelector('.loading-message');
                if (messageEl) {
                    messageEl.textContent = message;
                }
            }
        }
    }

    showToast(message, type = 'info', duration = null) {
        // Use POSUtils if available
        if (window.POSUtils?.showToast) {
            window.POSUtils.showToast(message, type, duration || this.config.toastDuration);
            return;
        }
        
        // Fallback toast implementation
        console.log(`💬 ${type.toUpperCase()}: ${message}`);
        
        const toastContainer = document.getElementById('toastContainer');
        if (!toastContainer) {
            // Create toast container
            const container = document.createElement('div');
            container.id = 'toastContainer';
            container.className = 'position-fixed top-0 end-0 p-3';
            container.style.zIndex = '9999';
            document.body.appendChild(container);
        }
        
        const toastId = 'toast-' + Date.now();
        const typeClass = {
            success: 'bg-success',
            error: 'bg-danger',
            warning: 'bg-warning',
            info: 'bg-info'
        }[type] || 'bg-info';
        
        const icon = {
            success: 'bx-check-circle',
            error: 'bx-error-circle',
            warning: 'bx-error',
            info: 'bx-info-circle'
        }[type] || 'bx-info-circle';
        
        const toastHtml = `
            <div id="${toastId}" class="toast show align-items-center text-white ${typeClass} border-0" role="alert">
                <div class="d-flex">
                    <div class="toast-body d-flex align-items-center">
                        <i class="bx ${icon} me-2 fs-4"></i>
                        <span>${message}</span>
                    </div>
                    <button type="button" class="btn-close btn-close-white me-2 m-auto" onclick="document.getElementById('${toastId}').remove()"></button>
                </div>
            </div>
        `;
        
        toastContainer.insertAdjacentHTML('beforeend', toastHtml);
        
        // Auto-remove
        setTimeout(() => {
            const toast = document.getElementById(toastId);
            if (toast) {
                toast.classList.remove('show');
                setTimeout(() => toast.remove(), 300);
            }
        }, duration || this.config.toastDuration);
    }

    showErrorToast(message) {
        this.showToast(message, 'error', 5000);
    }

    showWarningToast(message) {
        this.showToast(message, 'warning', 4000);
    }

    showSuccessToast(message) {
        this.showToast(message, 'success', 3000);
    }

    showFallback(message) {
        const fallback = document.getElementById('safeFallbackContainer');
        if (fallback) {
            fallback.querySelector('p').textContent = message;
            fallback.classList.remove('d-none');
        } else {
            // Create fallback if it doesn't exist
            const fallbackHtml = `
                <div id="safeFallbackContainer" class="safe-fallback">
                    <h5><i class="bx bx-error-circle me-2"></i>POS System Error</h5>
                    <p class="mb-3">${message}</p>
                    <button class="btn btn-sm btn-danger" onclick="location.reload()">
                        <i class="bx bx-refresh me-1"></i> Reload Page
                    </button>
                    <button class="btn btn-sm btn-outline-secondary" onclick="document.getElementById('safeFallbackContainer').classList.add('d-none')">
                        <i class="bx bx-x me-1"></i> Dismiss
                    </button>
                </div>
            `;
            document.body.insertAdjacentHTML('afterbegin', fallbackHtml);
        }
    }

    saveState() {
        try {
            if (window.POSState) {
                const state = {
                    cart: window.POSState.cart || [],
                    customer: window.POSState.currentCustomer,
                    discount: window.POSState.currentDiscount || 0,
                    timestamp: Date.now(),
                    version: this.version
                };
                
                localStorage.setItem('pos_state_v3', JSON.stringify(state));
                this.trackEvent('state_saved', { items: state.cart.length });
            }
        } catch (e) {
            console.warn('Failed to save state:', e);
        }
    }

    loadState() {
        try {
            const saved = localStorage.getItem('pos_state_v3');
            if (saved) {
                const state = JSON.parse(saved);
                
                if (window.POSState) {
                    window.POSState.cart = state.cart || [];
                    window.POSState.currentCustomer = state.customer;
                    window.POSState.currentDiscount = state.discount || 0;
                    
                    // Update cart display
                    if (window.POSCart?.updateDisplay) {
                        window.POSCart.updateDisplay();
                    }
                    
                    return true;
                }
            }
        } catch (e) {
            console.warn('Failed to load state:', e);
        }
        return false;
    }

    // ========== EVENT TRACKING ==========
    trackEvent(event, data = {}) {
        if (!this.config.enableAnalytics) return;
        
        const eventData = {
            event,
            data,
            timestamp: new Date().toISOString(),
            version: this.version,
            environment: this.config.environment,
            phase: this.initializationPhase,
            network: this.status.network
        };
        
        console.log(`📊 Event: ${event}`, eventData);
        
        // Send to server if online
        if (this.status.network === 'online' && navigator.sendBeacon) {
            try {
                const blob = new Blob([JSON.stringify(eventData)], { type: 'application/json' });
                navigator.sendBeacon('/api/analytics/track', blob);
            } catch (e) {
                console.warn('Failed to send analytics:', e);
            }
        } else {
            // Queue for later
            this.queueEvent(eventData);
        }
    }

    trackError(error, context = {}) {
        const errorData = {
            message: error.message,
            stack: error.stack,
            name: error.name,
            code: error.code,
            ...context,
            timestamp: new Date().toISOString(),
            version: this.version
        };
        
        this.status.errors.push(errorData);
        
        // Keep only last 100 errors
        if (this.status.errors.length > 100) {
            this.status.errors.shift();
        }
        
        console.error('📊 Error tracked:', errorData);
        
        // Send to error reporting service if enabled
        if (this.config.enableErrorReporting && this.status.network === 'online') {
            this.reportError(errorData);
        }
    }

    queueEvent(eventData) {
        const queue = JSON.parse(localStorage.getItem('pos_event_queue') || '[]');
        queue.push(eventData);
        
        // Keep only last 100 events
        if (queue.length > 100) {
            queue.shift();
        }
        
        localStorage.setItem('pos_event_queue', JSON.stringify(queue));
    }

    reportError(errorData) {
        // Use fetch with beacon fallback
        if (navigator.sendBeacon) {
            const blob = new Blob([JSON.stringify(errorData)], { type: 'application/json' });
            navigator.sendBeacon('/api/errors/report', blob);
        } else {
            fetch('/api/errors/report', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(errorData),
                keepalive: true
            }).catch(() => {
                // Silently fail
            });
        }
    }

    // ========== KEYBOARD SHORTCUTS ==========
    handleKeyboardShortcuts(e) {
        if (e.target.tagName === 'INPUT' || e.target.tagName === 'TEXTAREA' || e.target.isContentEditable) {
            return;
        }
        
        try {
            switch(e.key) {
                case 'F1':
                    e.preventDefault();
                    this.focusSearch();
                    break;
                    
                case 'F2':
                    e.preventDefault();
                    this.selectCustomer();
                    break;
                    
                case 'F3':
                    e.preventDefault();
                    this.newSale();
                    break;
                    
                case 'F8':
                    e.preventDefault();
                    this.openPayment();
                    break;
                    
                case 'F9':
                    e.preventDefault();
                    this.openMpesaManager();
                    break;
                    
                case 'F12':
                    if (this.config.debug) {
                        e.preventDefault();
                        this.openDebugPanel();
                    }
                    break;
                    
                case 'Escape':
                    e.preventDefault();
                    this.closeAllModals();
                    break;
                    
                case 'b':
                    if (e.ctrlKey) {
                        e.preventDefault();
                        this.openScanner();
                    }
                    break;
                    
                case 'r':
                    if (e.ctrlKey && e.shiftKey) {
                        e.preventDefault();
                        this.emergencyReset();
                    }
                    break;
            }
        } catch (error) {
            console.error('Keyboard shortcut error:', error);
        }
    }

    // ========== GLOBAL ACTION METHODS ==========
    focusSearch() {
        const searchInput = document.getElementById('productSearch') || document.getElementById('globalSearch');
        if (searchInput) {
            searchInput.focus();
            searchInput.select();
            this.trackEvent('search_focused');
        }
    }

    async selectCustomer() {
        const customers = this.modules.get('POSCustomers');
        if (customers?.showModal) {
            await customers.showModal();
            this.trackEvent('customer_selected');
        } else {
            this.showToast('Customer selection not available', 'warning');
        }
    }

    newSale() {
        if (window.POSState?.cart?.length > 0) {
            if (!confirm('Start new sale? Current cart will be cleared.')) return;
        }
        
        window.POSState?.clearCart?.();
        this.modules.get('POSCart')?.updateDisplay?.();
        this.showToast('New sale started', 'success');
        this.trackEvent('new_sale_started');
    }

    openPayment() {
        // Check if cart has items
        const cart = window.POSState?.cart || [];
        if (cart.length === 0) {
            this.showToast('Add items to cart first', 'warning');
            return;
        }
        
        const paymentManager = this.modules.get('PaymentManager');
        if (paymentManager?.showModal) {
            paymentManager.showModal();
            this.trackEvent('payment_modal_opened');
        } else {
            this.showToast('Payment system not ready', 'warning');
        }
    }

    openMpesaManager() {
        const modal = document.getElementById('manualMpesaModal');
        if (modal) {
            try {
                new bootstrap.Modal(modal).show();
                this.trackEvent('mpesa_manager_opened');
            } catch (error) {
                this.showToast('Cannot open M-Pesa manager', 'error');
            }
        }
    }

    closeAllModals() {
        document.querySelectorAll('.modal.show').forEach(modal => {
            const bsModal = bootstrap.Modal.getInstance(modal);
            if (bsModal) {
                bsModal.hide();
                this.trackEvent('modal_closed', { modalId: modal.id });
            }
        });
    }

    openScanner() {
        const scanner = this.modules.get('POSScanner');
        if (scanner?.open) {
            scanner.open();
            this.trackEvent('scanner_opened');
        } else {
            this.showToast('Scanner not available', 'warning');
        }
    }

    openDebugPanel() {
        console.log('🔧 Opening debug panel...');
        this.showSystemStatus();
    }

    // ========== NETWORK & SYNC ==========
    createNetworkStatusIndicator() {
        // Create network status indicator in UI
        const existingIndicator = document.getElementById('networkStatusIndicator');
        if (existingIndicator) return;
        
        const indicator = document.createElement('div');
        indicator.id = 'networkStatusIndicator';
        indicator.className = 'network-status';
        indicator.innerHTML = `
            <div class="network-status-content">
                <i class="network-icon bx bx-wifi"></i>
                <span class="network-text">Online</span>
            </div>
        `;
        
        document.body.appendChild(indicator);
        
        // Update based on network status
        this.updateNetworkIndicator();
    }

    updateNetworkIndicator() {
        const indicator = document.getElementById('networkStatusIndicator');
        if (!indicator) return;
        
        const isOnline = this.status.network === 'online';
        const icon = indicator.querySelector('.network-icon');
        const text = indicator.querySelector('.network-text');
        
        if (isOnline) {
            indicator.className = 'network-status online';
            icon.className = 'network-icon bx bx-wifi';
            text.textContent = 'Online';
        } else {
            indicator.className = 'network-status offline';
            icon.className = 'network-icon bx bx-wifi-off';
            text.textContent = 'Offline';
        }
    }

    async triggerNetworkSync() {
        if (this.status.network === 'online') {
            console.log('🔄 Triggering network sync...');
            await this.syncPendingData();
        }
    }

    async syncPendingData() {
        try {
            // Sync pending events
            const eventQueue = JSON.parse(localStorage.getItem('pos_event_queue') || '[]');
            if (eventQueue.length > 0) {
                console.log(`🔄 Syncing ${eventQueue.length} pending events...`);
                
                for (const event of eventQueue) {
                    await this.sendEventToServer(event);
                }
                
                localStorage.removeItem('pos_event_queue');
                console.log('✅ Events synced');
            }
            
            // Sync pending transactions
            const pendingTransactions = JSON.parse(localStorage.getItem('pos_pending_transactions') || '[]');
            if (pendingTransactions.length > 0) {
                console.log(`🔄 Syncing ${pendingTransactions.length} pending transactions...`);
                
                // This would be implemented based on your backend API
                // await this.syncTransactions(pendingTransactions);
                
                console.log('✅ Transactions synced');
            }
            
        } catch (error) {
            console.warn('Sync failed:', error);
        }
    }

    async sendEventToServer(eventData) {
        try {
            const response = await fetch('/api/analytics/track', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(eventData),
                keepalive: true
            });
            
            return response.ok;
        } catch (error) {
            throw error;
        }
    }

    // ========== PERFORMANCE MONITORING ==========
    startPerformanceMonitoring() {
        console.log('📈 Starting performance monitoring...');
        
        // Track page load performance
        if (window.performance) {
            const perfData = performance.getEntriesByType('navigation')[0];
            if (perfData) {
                this.status.performance.pageLoad = {
                    domComplete: perfData.domComplete,
                    loadEventEnd: perfData.loadEventEnd,
                    domInteractive: perfData.domInteractive
                };
                
                this.trackEvent('performance_page_load', this.status.performance.pageLoad);
            }
        }
        
        // Monitor FPS
        this.startFPSMonitoring();
        
        // Monitor memory
        this.startMemoryMonitoring();
    }

    startFPSMonitoring() {
        let frameCount = 0;
        let lastTime = Date.now();
        
        const checkFPS = () => {
            frameCount++;
            const currentTime = Date.now();
            
            if (currentTime - lastTime >= 1000) {
                const fps = Math.round((frameCount * 1000) / (currentTime - lastTime));
                
                if (fps < 30) {
                    this.trackEvent('low_fps', { fps });
                }
                
                frameCount = 0;
                lastTime = currentTime;
            }
            
            requestAnimationFrame(checkFPS);
        };
        
        requestAnimationFrame(checkFPS);
    }

    startMemoryMonitoring() {
        if (performance.memory) {
            setInterval(() => {
                const memory = performance.memory;
                const usedMB = Math.round(memory.usedJSHeapSize / 1048576);
                const totalMB = Math.round(memory.totalJSHeapSize / 1048576);
                
                if (usedMB > 200) { // 200MB threshold
                    this.trackEvent('high_memory_usage', { usedMB, totalMB });
                    this.showWarningToast('High memory usage detected');
                }
            }, 30000);
        }
    }

    performHealthCheck() {
        const health = {
            timestamp: Date.now(),
            modules: {},
            memory: null,
            network: this.status.network,
            uptime: Date.now() - this.startTime
        };
        
        // Check module health
        for (const [name, module] of this.modules) {
            health.modules[name] = {
                loaded: this.status.modules[name]?.loaded || false,
                initialized: this.status.modules[name]?.initialized || false,
                alive: this.checkModuleHealth(module)
            };
        }
        
        // Check memory
        if (performance.memory) {
            health.memory = {
                usedMB: Math.round(performance.memory.usedJSHeapSize / 1048576),
                totalMB: Math.round(performance.memory.totalJSHeapSize / 1048576)
            };
        }
        
        // Log health status
        if (this.config.debug) {
            console.log('🏥 Health check:', health);
        }
        
        // Track if any critical issues
        const criticalModules = Object.entries(this.moduleDependencies)
            .filter(([_, config]) => config.critical)
            .map(([name]) => name);
        
        const failedCritical = criticalModules.filter(name => 
            !health.modules[name]?.loaded || !health.modules[name]?.initialized
        );
        
        if (failedCritical.length > 0) {
            console.error('🚨 Critical modules failed:', failedCritical);
            this.trackEvent('health_check_critical', { failedModules: failedCritical });
        }
    }

    checkModuleHealth(module) {
        try {
            // Check if module has a health check method
            if (typeof module.healthCheck === 'function') {
                return module.healthCheck();
            }
            
            // Default health check
            return module !== null && module !== undefined;
        } catch (error) {
            return false;
        }
    }

    // ========== DATA MANAGEMENT ==========
    cleanupOldData() {
        console.log('🧹 Cleaning up old data...');
        
        try {
            // Clean old localStorage items
            const now = Date.now();
            const oneMonthAgo = now - (30 * 24 * 60 * 60 * 1000);
            
            // Check all localStorage items
            for (let i = 0; i < localStorage.length; i++) {
                const key = localStorage.key(i);
                
                if (key.startsWith('pos_') && !key.includes('config') && !key.includes('state')) {
                    try {
                        const item = JSON.parse(localStorage.getItem(key));
                        if (item.timestamp && item.timestamp < oneMonthAgo) {
                            localStorage.removeItem(key);
                            console.log(`🗑️  Cleaned: ${key}`);
                        }
                    } catch (e) {
                        // Skip non-JSON items
                    }
                }
            }
            
            console.log('✅ Data cleanup complete');
        } catch (error) {
            console.warn('Data cleanup failed:', error);
        }
    }

    // ========== FALLBACK CREATORS ==========
    createUtilsFallback() {
        return {
            showToast: (msg, type, duration = 5000) => {
                this.showToast(msg, type, duration);
            },
            formatCurrency: (amount) => {
                return `KES ${parseFloat(amount || 0).toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ',')}`;
            },
            showLoading: (show, message) => {
                this.showLoading(show, message);
            },
            formatDate: (date) => {
                return new Date(date).toLocaleString('en-KE');
            },
            generateId: () => {
                return Date.now().toString(36) + Math.random().toString(36).substr(2);
            }
        };
    }

    createStateFallback() {
        const state = {
            cart: [],
            currentCustomer: null,
            currentDiscount: 0,
            
            addItem: function(product, quantity = 1) {
                const existing = this.cart.find(item => item.id === product.id);
                if (existing) {
                    existing.quantity += quantity;
                } else {
                    this.cart.push({
                        ...product,
                        quantity: quantity,
                        addedAt: Date.now()
                    });
                }
                return this.cart;
            },
            
            removeItem: function(productId) {
                this.cart = this.cart.filter(item => item.id !== productId);
                return this.cart;
            },
            
            updateQuantity: function(productId, quantity) {
                const item = this.cart.find(item => item.id === productId);
                if (item) {
                    item.quantity = quantity;
                }
                return this.cart;
            },
            
            getCartSubtotal: function() {
                return this.cart.reduce((sum, item) => sum + (item.price * item.quantity), 0);
            },
            
            getCartTotal: function() {
                return this.getCartSubtotal() - this.currentDiscount;
            },
            
            clearCart: function() {
                this.cart = [];
                this.currentDiscount = 0;
                this.currentCustomer = null;
            },
            
            setCustomer: function(customer) {
                this.currentCustomer = customer;
            },
            
            setDiscount: function(amount) {
                this.currentDiscount = amount;
            }
        };
        
        // Load saved state
        const saved = localStorage.getItem('pos_state_v3');
        if (saved) {
            try {
                const parsed = JSON.parse(saved);
                state.cart = parsed.cart || [];
                state.currentCustomer = parsed.customer;
                state.currentDiscount = parsed.discount || 0;
            } catch (e) {
                console.warn('Failed to load saved state:', e);
            }
        }
        
        return state;
    }

    createCartFallback() {
        return {
            updateDisplay: function() {
                const cartContainer = document.querySelector('.cart-items');
                const totalElement = document.querySelector('.cart-total-value');
                
                if (!cartContainer || !window.posState) return;
                
                const state = window.posState;
                const subtotal = state.getCartSubtotal();
                const total = state.getCartTotal();
                
                // Update cart items
                cartContainer.innerHTML = state.cart.map(item => `
                    <div class="cart-item" data-id="${item.id}">
                        <div class="cart-item-name">${item.name}</div>
                        <div class="cart-item-details">
                            <span class="cart-item-quantity">${item.quantity} x</span>
                            <span class="cart-item-price">KES ${item.price.toFixed(2)}</span>
                            <span class="cart-item-total">KES ${(item.price * item.quantity).toFixed(2)}</span>
                        </div>
                    </div>
                `).join('');
                
                // Update totals
                if (totalElement) {
                    totalElement.textContent = `KES ${total.toFixed(2)}`;
                }
                
                // Update item count
                const countElement = document.querySelector('.cart-item-count');
                if (countElement) {
                    countElement.textContent = state.cart.length;
                }
            },
            
            loadSavedCart: function() {
                this.updateDisplay();
            }
        };
    }

    createPaymentManagerFallback() {
        return {
            showModal: function() {
                const modal = document.getElementById('paymentModal');
                if (modal) {
                    // Update summary
                    const total = window.posState?.getCartTotal?.() || 0;
                    const totalElement = document.getElementById('paymentTotalAmount');
                    if (totalElement) {
                        totalElement.textContent = `KES ${total.toFixed(2)}`;
                    }
                    
                    // Show modal
                    const bsModal = new bootstrap.Modal(modal);
                    bsModal.show();
                } else {
                    alert('Payment system error');
                }
            },
            
            loadPaymentMethods: async function() {
                return Promise.resolve();
            }
        };
    }

    createPaymentsFallback() {
        return {
            init: function() {
                console.log('💰 Payments fallback initialized');
                return Promise.resolve();
            }
        };
    }

    createProductsFallback() {
        return {
            loadAll: async function() {
                console.log('📦 Products fallback loaded');
                return Promise.resolve();
            }
        };
    }

    // ========== EMERGENCY & RECOVERY ==========
    emergencyReset() {
        if (confirm('Emergency reset will clear all local data and reload. Are you sure?')) {
            // Clear all POS-related localStorage
            Object.keys(localStorage).forEach(key => {
                if (key.startsWith('pos_')) {
                    localStorage.removeItem(key);
                }
            });
            
            // Clear sessionStorage
            sessionStorage.clear();
            
            // Reload page
            location.reload();
        }
    }

    setupFallbackMode() {
        console.warn('⚠️ Setting up fallback mode...');
        
        // Ensure critical functions exist
        if (!window.posState) {
            window.posState = this.createStateFallback();
        }
        
        if (!window.POSUtils) {
            window.POSUtils = this.createUtilsFallback();
        }
        
        if (!window.POSCart) {
            window.POSCart = this.createCartFallback();
        }
        
        // Update cart display
        if (window.POSCart.updateDisplay) {
            window.POSCart.updateDisplay();
        }
        
        // Show warning
        this.showWarningToast('System loaded in fallback mode. Some features may be limited.');
        
        // Track fallback activation
        this.trackEvent('fallback_mode_activated', {
            phase: this.initializationPhase,
            errors: this.status.errors.length
        });
    }

    // ========== UI HELPERS ==========
    handleResize() {
        // Handle responsive adjustments
        const width = window.innerWidth;
        
        if (width < 768) {
            // Mobile adjustments
            document.body.classList.add('mobile-view');
        } else {
            document.body.classList.remove('mobile-view');
        }
        
        this.trackEvent('window_resized', { width });
    }

    showSystemStatus() {
        const status = {
            version: this.version,
            initialized: this.initialized,
            initializationPhase: this.initializationPhase,
            network: this.status.network,
            modules: this.status.modules,
            errors: this.status.errors.length,
            warnings: this.status.warnings.length,
            performance: this.status.performance,
            config: this.config
        };
        
        console.log('🔧 System Status:', status);
        
        // Show in a modal if in debug mode
        if (this.config.debug) {
            const statusStr = JSON.stringify(status, null, 2);
            alert(`System Status:\n\n${statusStr}`);
        }
    }

    isCriticalError(error) {
        if (!error) return false;
        
        const criticalPatterns = [
            /payment/i,
            /cart/i,
            /state/i,
            /initialization/i,
            /module.*fail/i
        ];
        
        const errorStr = error.message || error.toString();
        return criticalPatterns.some(pattern => pattern.test(errorStr));
    }

    isUserFacingError(error) {
        if (!error) return false;
        
        const userFacingPatterns = [
            /network/i,
            /timeout/i,
            /server/i,
            /api/i,
            /fetch/i
        ];
        
        const errorStr = error.message || error.toString();
        return userFacingPatterns.some(pattern => pattern.test(errorStr));
    }

    // ========== PUBLIC API ==========
    getModule(name) {
        return this.modules.get(name);
    }

    isModuleLoaded(name) {
        return this.modules.has(name) && this.status.modules[name]?.loaded;
    }

    isModuleInitialized(name) {
        return this.status.modules[name]?.initialized || false;
    }

    getSystemStatus() {
        return {
            version: this.version,
            initialized: this.initialized,
            modules: { ...this.status.modules },
            network: this.status.network,
            uptime: Date.now() - this.startTime
        };
    }

    reloadModule(name) {
        console.log(`🔄 Reloading module: ${name}`);
        
        // Remove from modules
        this.modules.delete(name);
        delete this.status.modules[name];
        
        // Try to reload
        return this.loadSingleModule(name, this.moduleDependencies[name])
            .then(module => {
                this.modules.set(name, module);
                this.status.modules[name] = {
                    loaded: true,
                    initialized: false,
                    reloaded: true,
                    timestamp: Date.now()
                };
                
                // Initialize if needed
                if (typeof module.init === 'function') {
                    return module.init().then(() => {
                        this.status.modules[name].initialized = true;
                        console.log(`✅ Module reloaded: ${name}`);
                        return module;
                    });
                }
                
                return module;
            })
            .catch(error => {
                console.error(`❌ Failed to reload module ${name}:`, error);
                throw error;
            });
    }

    // ========== DESTRUCTOR ==========
    destroy() {
        console.log('🔄 Destroying POS System...');
        
        // Save state
        this.saveState();
        
        // Clear intervals
        if (this.autoSaveInterval) {
            clearInterval(this.autoSaveInterval);
        }
        
        if (this.syncInterval) {
            clearInterval(this.syncInterval);
        }
        
        if (this.healthCheckInterval) {
            clearInterval(this.healthCheckInterval);
        }
        
        // Destroy modules
        for (const [name, module] of this.modules) {
            if (typeof module.destroy === 'function') {
                try {
                    module.destroy();
                    console.log(`✅ Module destroyed: ${name}`);
                } catch (error) {
                    console.warn(`⚠️ Failed to destroy module ${name}:`, error);
                }
            }
        }
        
        // Clear references
        this.modules.clear();
        this.status = {};
        this.initialized = false;
        
        // Remove global references
        delete window.POS;
        delete window.posSystem;
        
        console.log('✅ POS System destroyed');
    }
}

// =========================================================
// GLOBAL INITIALIZATION
// =========================================================

// Create global instance
const POS = new POSSystem();

// Initialize when DOM is ready with enhanced timing
function initializePOS() {
    // Check if already initialized
    if (window.POS && window.POS.initialized) {
        console.log('🔄 POS already initialized, skipping');
        return;
    }
    
    // Set a timeout for initialization
    const initTimeout = setTimeout(() => {
        console.warn('⚠️ POS initialization taking too long, forcing completion');
        if (window.POS && !window.POS.initialized) {
            window.POS.setupFallbackMode();
        }
    }, 30000); // 30 second timeout
    
    // Start initialization
    POS.init()
        .then(() => {
            clearTimeout(initTimeout);
            console.log('🎉 POS System ready for use');
        })
        .catch(error => {
            clearTimeout(initTimeout);
            console.error('❌ POS initialization failed:', error);
            
            // Try to recover
            setTimeout(() => {
                if (!POS.initialized) {
                    POS.setupFallbackMode();
                }
            }, 1000);
        });
}

// DOM ready check with multiple strategies
if (document.readyState === 'loading') {
    // Use DOMContentLoaded
    document.addEventListener('DOMContentLoaded', () => {
        console.log('📄 DOM Content Loaded');
        setTimeout(initializePOS, 100);
    });
    
    // Fallback: Also use window.load
    window.addEventListener('load', () => {
        console.log('🖼️ Window Loaded');
        // If not already initialized, try again
        if (!window.POS?.initialized) {
            setTimeout(initializePOS, 500);
        }
    });
} else {
    // DOM already loaded
    console.log('⚡ DOM already loaded');
    setTimeout(initializePOS, 100);
}

// Export globally
window.POS = POS;
window.posSystem = POS;

// Emergency recovery
window.addEventListener('error', (event) => {
    // Capture any uncaught errors after initialization
    if (window.POS && window.POS.initialized) {
        window.POS.trackError(event.error, { type: 'global_error' });
    }
});

// Performance monitoring
window.addEventListener('pageshow', (event) => {
    if (event.persisted && window.POS) {
        console.log('📄 Page restored from bfcache');
        window.POS.trackEvent('page_restored_from_bfcache');
    }
});

console.log('✅ POS System main loaded v3.0.0');