// public/js/pos/modules/scanner.js
// POS Barcode Scanner Module
// =========================================================

class POSScanner {
    constructor() {
        this.scannedProduct = null;
        this.stream = null;
        this.recentScans = JSON.parse(localStorage.getItem('pos_recent_scans') || '[]');
        this.MAX_RECENT_SCANS = 5;
    }
    
    init() {
        this.setupEventListeners();
        this.updateRecentScans();
    }
    
    setupEventListeners() {
        const barcodeModal = document.getElementById('barcodeModal');
        if (barcodeModal) {
            barcodeModal.addEventListener('shown.bs.modal', () => {
                this.initialize();
                this.updateRecentScans();
                const manualInput = document.getElementById('manualBarcodeInput');
                if (manualInput) manualInput.focus();
            });
            
            barcodeModal.addEventListener('hidden.bs.modal', () => {
                this.stop();
                this.resetUI();
            });
        }
        
        const scanBarcodeBtn = document.getElementById('scanBarcodeBtn');
        if (scanBarcodeBtn) {
            scanBarcodeBtn.addEventListener('click', () => {
                const barcode = document.getElementById('manualBarcodeInput')?.value.trim();
                if (barcode) {
                    this.handleScanned(barcode);
                }
            });
        }
        
        const addScannedProductBtn = document.getElementById('addScannedProduct');
        if (addScannedProductBtn) {
            addScannedProductBtn.addEventListener('click', () => this.addToCart());
        }
        
        const scanAnotherBtn = document.getElementById('scanAnother');
        if (scanAnotherBtn) {
            scanAnotherBtn.addEventListener('click', () => this.resetUI());
        }
        
        // Manual input enter key
        const manualInput = document.getElementById('manualBarcodeInput');
        if (manualInput) {
            manualInput.addEventListener('keypress', (e) => {
                if (e.key === 'Enter') {
                    const barcode = manualInput.value.trim();
                    if (barcode) {
                        this.handleScanned(barcode);
                    }
                }
            });
        }
    }
    
    open() {
        const modalElement = document.getElementById('barcodeModal');
        if (modalElement) {
            const modal = new bootstrap.Modal(modalElement);
            modal.show();
            return true;
        }
        console.error('Barcode modal not found');
        return false;
    }
    
    async initialize() {
        try {
            const constraints = {
                video: {
                    facingMode: 'environment',
                    width: { ideal: 1280 },
                    height: { ideal: 720 }
                }
            };
            
            this.stream = await navigator.mediaDevices.getUserMedia(constraints);
            const video = document.getElementById('scanner-video');
            if (video) {
                video.srcObject = this.stream;
            }
            
            this.updateStatus('Scanner ready - point at barcode');
            
        } catch (error) {
            console.error('Scanner error:', error);
            this.updateStatus('Camera access denied. Use manual input.');
            
            const scannerContainer = document.getElementById('scanner-container');
            if (scannerContainer) {
                scannerContainer.innerHTML = `
                    <div class="text-center py-5">
                        <i class="bx bx-video-off fs-1 text-muted mb-3"></i>
                        <p class="text-muted">Camera not available</p>
                        <p class="small text-muted">Use manual barcode input</p>
                    </div>
                `;
            }
        }
    }
    
    stop() {
        const video = document.getElementById('scanner-video');
        if (video && video.srcObject) {
            const tracks = video.srcObject.getTracks();
            tracks.forEach(track => track.stop());
        }
        this.stream = null;
    }
    
    updateStatus(message) {
        const statusElement = document.getElementById('scannerStatus');
        if (statusElement) {
            statusElement.textContent = message;
        }
    }
    
    handleScanned(barcode) {
        if (!barcode || barcode.trim() === '') return;
        
        const manualInput = document.getElementById('manualBarcodeInput');
        if (manualInput) {
            manualInput.value = barcode;
        }
        
        this.searchByBarcode(barcode);
        this.addToRecentScans(barcode);
    }
    
    async searchByBarcode(barcode) {
        if (window.POSUtils && window.POSUtils.showLoading) {
            window.POSUtils.showLoading(true);
        }
        
        try {
            const response = await fetch(`/api/products/barcode/${encodeURIComponent(barcode)}`);
            
            if (!response.ok) throw new Error('Product not found');
            
            const data = await response.json();
            
            if (window.POSUtils && window.POSUtils.showLoading) {
                window.POSUtils.showLoading(false);
            }
            
            if (data.success && data.product) {
                this.scannedProduct = data.product;
                this.showPreview(data.product);
                
                // Auto-add to cart if enabled
                const autoAddCheckbox = document.getElementById('autoAddToCart');
                if (autoAddCheckbox && autoAddCheckbox.checked) {
                    this.addToCart();
                }
                
            } else {
                this.showNotFound(barcode);
            }
        } catch (error) {
            if (window.POSUtils && window.POSUtils.showLoading) {
                window.POSUtils.showLoading(false);
            }
            console.error('Barcode search error:', error);
            this.showNotFound(barcode);
        }
    }
    
    showPreview(product) {
        const previewDiv = document.getElementById('productPreview');
        if (!previewDiv) return;
        
        // Update product info
        const elements = {
            'productName': product.name,
            'productPrice': `KES ${parseFloat(product.final_price || product.sale_price || product.price || 0).toFixed(2)}`,
            'productStock': product.available_stock || product.stock || 0,
            'productSku': product.sku || 'N/A',
            'productBarcode': product.barcode || 'N/A'
        };
        
        Object.keys(elements).forEach(id => {
            const element = document.getElementById(id);
            if (element) element.textContent = elements[id];
        });
        
        // Show preview
        previewDiv.style.display = 'block';
        const scannerContainer = document.getElementById('scanner-container');
        const recentScansDiv = document.getElementById('recentScans');
        if (scannerContainer) scannerContainer.style.display = 'none';
        if (recentScansDiv) recentScansDiv.style.display = 'none';
        
        this.updateStatus('Product found!');
    }
    
    showNotFound(barcode) {
        const previewDiv = document.getElementById('productPreview');
        if (!previewDiv) return;
        
        previewDiv.innerHTML = `
            <div class="alert alert-warning">
                <div class="d-flex align-items-center">
                    <i class="bx bx-error-circle fs-4 me-2"></i>
                    <div>
                        <h6 class="mb-1">Product Not Found</h6>
                        <p class="mb-0 small">No product found for barcode: <strong>${barcode}</strong></p>
                        <div class="mt-2">
                            <button class="btn btn-sm btn-outline-primary" onclick="POSProducts.showAddProductModal('${barcode}')">
                                <i class="bx bx-plus me-1"></i> Add New Product
                            </button>
                            <button class="btn btn-sm btn-outline-secondary" onclick="window.POSScanner.resetUI()">
                                <i class="bx bx-redo me-1"></i> Try Again
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        `;
        previewDiv.style.display = 'block';
        
        const scannerContainer = document.getElementById('scanner-container');
        if (scannerContainer) scannerContainer.style.display = 'none';
    }
    
    addToCart() {
        if (!this.scannedProduct) return;
        
        // Check stock
        if ((this.scannedProduct.available_stock <= 0 && this.scannedProduct.available_stock !== undefined) || 
            (this.scannedProduct.stock <= 0 && this.scannedProduct.stock !== undefined)) {
            if (window.POSUtils && window.POSUtils.showToast) {
                window.POSUtils.showToast('Product is out of stock', 'error');
            }
            return;
        }
        
        // Add to cart
        if (window.POSCart && window.POSCart.add) {
            const added = window.POSCart.add(this.scannedProduct, 1);
            if (added) {
                if (window.POSUtils && window.POSUtils.showToast) {
                    window.POSUtils.showToast(`${this.scannedProduct.name} added to cart`, 'success');
                }
                this.resetUI();
            }
        } else {
            console.error('POSCart module not available');
            if (window.POSUtils && window.POSUtils.showToast) {
                window.POSUtils.showToast('Failed to add to cart', 'error');
            }
        }
    }
    
    resetUI() {
        const previewDiv = document.getElementById('productPreview');
        const scannerContainer = document.getElementById('scanner-container');
        const recentScansDiv = document.getElementById('recentScans');
        const manualInput = document.getElementById('manualBarcodeInput');
        
        if (previewDiv) previewDiv.style.display = 'none';
        if (scannerContainer) scannerContainer.style.display = 'block';
        if (recentScansDiv) recentScansDiv.style.display = 'block';
        if (manualInput) {
            manualInput.value = '';
            manualInput.focus();
        }
        
        this.scannedProduct = null;
        this.updateStatus('Scanner ready - point at barcode');
    }
    
    addToRecentScans(barcode) {
        // Remove if already exists
        this.recentScans = this.recentScans.filter(scan => scan.barcode !== barcode);
        
        // Add to beginning
        this.recentScans.unshift({
            barcode: barcode,
            timestamp: new Date().toISOString(),
            time: new Date().toLocaleTimeString()
        });
        
        // Keep only last N scans
        if (this.recentScans.length > this.MAX_RECENT_SCANS) {
            this.recentScans.pop();
        }
        
        // Save to localStorage
        localStorage.setItem('pos_recent_scans', JSON.stringify(this.recentScans));
        
        // Update UI
        this.updateRecentScans();
    }
    
    updateRecentScans() {
        const container = document.getElementById('recentScansList');
        const recentScansDiv = document.getElementById('recentScans');
        
        if (!container || !recentScansDiv) return;
        
        if (this.recentScans.length === 0) {
            recentScansDiv.style.display = 'none';
            return;
        }
        
        recentScansDiv.style.display = 'block';
        
        container.innerHTML = this.recentScans.map(scan => `
            <div class="list-group-item" onclick="window.POSScanner.searchByBarcode('${scan.barcode}')">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <div class="fw-bold">${scan.barcode}</div>
                        <small class="text-muted">${scan.time}</small>
                    </div>
                    <i class="bx bx-chevron-right text-muted"></i>
                </div>
            </div>
        `).join('');
    }
}

// Create instance and export
window.POSScanner = new POSScanner();

// Global functions for inline onclick handlers
window.openBarcodeScanner = () => window.POSScanner.open();
window.handleBarcodeScanned = (barcode) => window.POSScanner.handleScanned(barcode);
window.searchProductByBarcode = (barcode) => window.POSScanner.searchByBarcode(barcode);
window.resetScannerUI = () => window.POSScanner.resetUI();
window.addScannedProductToCart = () => window.POSScanner.addToCart();