// =========================================================
// POS Receipts - Fixed Version
// =========================================================

class POSReceipts {
    constructor() {
        this.version = '3.2.0';
        this.lastReceipt = null;
        this.printerAvailable = false;
        this.thermalPrinter = null;
        
        console.log(`🧾 POS Receipts v${this.version} initializing...`);
    }
    
    async init() {
        try {
            await this.checkPrinter();
            this.setupEventListeners();
            
            console.log('✅ POS Receipts initialized');
            return this;
        } catch (error) {
            console.error('❌ POS Receipts initialization failed:', error);
            return this;
        }
    }
    
    async checkPrinter() {
        // Check if we can print
        if (window.print) {
            this.printerAvailable = true;
        }
        
        // Check for thermal printer
        try {
            // Try to detect USB printer
            if (navigator.usb) {
                const devices = await navigator.usb.getDevices();
                this.thermalPrinter = devices.find(device => 
                    device.productName?.includes('Printer') || 
                    device.manufacturerName?.includes('Printer')
                );
            }
        } catch (error) {
            // USB API not available or permission denied
            console.warn('USB printer check failed:', error);
        }
    }
    
    setupEventListeners() {
        // Listen for payment completion to print receipt
        document.addEventListener('payment:completed', (event) => {
            const receiptData = event.detail;
            if (receiptData && receiptData.printReceipt !== false) {
                this.print(receiptData);
            }
        });
        
        // Keyboard shortcut for reprint (Ctrl+P)
        document.addEventListener('keydown', (e) => {
            if (e.ctrlKey && e.key === 'p' && !e.altKey) {
                e.preventDefault();
                window.printLastReceipt();
            }
        });
    }
    
    async print(receiptData) {
        try {
            console.log('🖨️ Printing receipt:', receiptData);
            
            // Store as last receipt
            this.lastReceipt = receiptData;
            localStorage.setItem('pos_last_receipt_data', JSON.stringify(receiptData));
            
            // Show print dialog
            if (this.printerAvailable) {
                await this.openPrintDialog(receiptData);
            } else {
                await this.showReceiptPreview(receiptData);
            }
            
            return { success: true, message: 'Receipt printed' };
            
        } catch (error) {
            console.error('❌ Print failed:', error);
            return { 
                success: false, 
                message: 'Print failed: ' + error.message 
            };
        }
    }
    
    async openPrintDialog(receiptData) {
        return new Promise((resolve, reject) => {
            try {
                // Create print window
                const printWindow = window.open('', '_blank');
                
                if (!printWindow) {
                    reject(new Error('Popup blocked. Please allow popups to print.'));
                    return;
                }
                
                // Generate receipt HTML
                const html = this.generateReceiptHTML(receiptData);
                
                printWindow.document.open();
                printWindow.document.write(html);
                printWindow.document.close();
                
                // Wait for content to load
                printWindow.onload = () => {
                    try {
                        printWindow.focus();
                        
                        // Print after a short delay
                        setTimeout(() => {
                            printWindow.print();
                            printWindow.onafterprint = () => {
                                printWindow.close();
                                resolve();
                            };
                        }, 500);
                    } catch (error) {
                        printWindow.close();
                        reject(error);
                    }
                };
                
            } catch (error) {
                reject(error);
            }
        });
    }
    
    generateReceiptHTML(receiptData) {
        const store = receiptData.store || {
            name: 'Kenyan Supermarket',
            address: 'Nairobi CBD',
            phone: '0700 000 000',
            vat_pin: 'P051XXXXXXXX'
        };
        
        const items = receiptData.items || [];
        const totals = receiptData.totals || {};
        const customer = receiptData.customer || {};
        const payment = receiptData.payment || {};
        
        return `
            <!DOCTYPE html>
            <html>
            <head>
                <title>Receipt - ${receiptData.invoice_no || 'POS Receipt'}</title>
                <style>
                    body { font-family: 'Courier New', monospace; font-size: 12px; margin: 0; padding: 10px; }
                    .receipt { max-width: 300px; margin: 0 auto; }
                    .header { text-align: center; margin-bottom: 15px; border-bottom: 1px dashed #000; padding-bottom: 10px; }
                    .store-name { font-weight: bold; font-size: 14px; }
                    .receipt-info { margin: 10px 0; }
                    .items-table { width: 100%; border-collapse: collapse; margin: 10px 0; }
                    .items-table th { text-align: left; border-bottom: 1px solid #000; padding: 5px 0; }
                    .items-table td { padding: 5px 0; border-bottom: 1px dashed #ccc; }
                    .items-table .total { font-weight: bold; }
                    .totals { margin-top: 15px; border-top: 1px dashed #000; padding-top: 10px; }
                    .totals-row { display: flex; justify-content: space-between; margin: 3px 0; }
                    .grand-total { font-weight: bold; font-size: 14px; border-top: 2px solid #000; padding-top: 5px; }
                    .footer { margin-top: 20px; text-align: center; font-size: 10px; }
                    @media print {
                        body { margin: 0; }
                        .no-print { display: none; }
                    }
                </style>
            </head>
            <body>
                <div class="receipt">
                    <div class="header">
                        <div class="store-name">${store.name}</div>
                        <div>${store.address}</div>
                        <div>Tel: ${store.phone}</div>
                        ${store.vat_pin ? `<div>VAT PIN: ${store.vat_pin}</div>` : ''}
                    </div>
                    
                    <div class="receipt-info">
                        <div>Date: ${receiptData.date || new Date().toLocaleDateString()}</div>
                        <div>Time: ${receiptData.time || new Date().toLocaleTimeString()}</div>
                        <div>Invoice: ${receiptData.invoice_no || 'N/A'}</div>
                        <div>Receipt: ${receiptData.receipt_no || 'N/A'}</div>
                        ${customer.name ? `<div>Customer: ${customer.name}</div>` : ''}
                        ${customer.phone ? `<div>Phone: ${customer.phone}</div>` : ''}
                    </div>
                    
                    <table class="items-table">
                        <thead>
                            <tr>
                                <th>Item</th>
                                <th>Qty</th>
                                <th>Price</th>
                                <th>Total</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${items.map(item => `
                                <tr>
                                    <td>${item.name || 'Item'}</td>
                                    <td>${item.quantity || 1}</td>
                                    <td>KES ${(item.unit_price || 0).toFixed(2)}</td>
                                    <td>KES ${((item.unit_price || 0) * (item.quantity || 1)).toFixed(2)}</td>
                                </tr>
                            `).join('')}
                        </tbody>
                    </table>
                    
                    <div class="totals">
                        <div class="totals-row">
                            <span>Subtotal:</span>
                            <span>KES ${(totals.subtotal || 0).toFixed(2)}</span>
                        </div>
                        ${totals.discount ? `
                            <div class="totals-row">
                                <span>Discount:</span>
                                <span>-KES ${totals.discount.toFixed(2)}</span>
                            </div>
                        ` : ''}
                        ${totals.vat_amount ? `
                            <div class="totals-row">
                                <span>VAT (${totals.vat_rate || 16}%):</span>
                                <span>KES ${totals.vat_amount.toFixed(2)}</span>
                            </div>
                        ` : ''}
                        <div class="totals-row grand-total">
                            <span>Total:</span>
                            <span>KES ${(totals.grand_total || 0).toFixed(2)}</span>
                        </div>
                    </div>
                    
                    <div class="payment-info">
                        <div>Payment Method: ${payment.method || 'Cash'}</div>
                        <div>Amount Paid: KES ${(payment.amount_paid || 0).toFixed(2)}</div>
                        ${payment.change ? `<div>Change: KES ${payment.change.toFixed(2)}</div>` : ''}
                    </div>
                    
                    <div class="footer">
                        <div>Cashier: ${receiptData.cashier || 'System'}</div>
                        <div>Thank you for your business!</div>
                        <div>Please keep this receipt for returns/exchanges</div>
                        <div>Valid for 7 days from date of purchase</div>
                    </div>
                    
                    <div class="no-print" style="margin-top: 20px; text-align: center;">
                        <button onclick="window.print()">Print Receipt</button>
                        <button onclick="window.close()">Close</button>
                    </div>
                </div>
                
                <script>
                    window.onload = function() {
                        // Auto-print if requested
                        if (window.location.search.includes('autoprint=true')) {
                            setTimeout(() => {
                                window.print();
                            }, 1000);
                        }
                    };
                </script>
            </body>
            </html>
        `;
    }
    
    async showReceiptPreview(receiptData) {
        const previewUrl = `/pos/receipt/preview/${receiptData.invoice_no || receiptData.receipt_no}`;
        window.open(previewUrl, '_blank', 'width=800,height=600');
    }
    
    // Backward compatibility
    printReceipt(receiptData) {
        return this.print(receiptData);
    }
    
    printThermal(receiptData) {
        // Implement thermal printing if available
        return this.print(receiptData);
    }
}

// Initialize and export
window.POSReceipts = new POSReceipts();

// Auto-initialize
setTimeout(() => {
    window.POSReceipts.init().then(() => {
        console.log('🧾 POS Receipts ready');
    }).catch(error => {
        console.error('Failed to initialize receipts:', error);
    });
}, 1000);

// Export for module systems
if (typeof module !== 'undefined' && module.exports) {
    module.exports = POSReceipts;
}