/**
 * POS Products Management Module - PRODUCTION GRADE
 * Version: 2.0.0
 * Author: POS System
 * Description: Handles product loading, searching, and cart operations
 */

class POSProducts {
    constructor() {
        this.config = {
            apiEndpoint: '/api/products',
            searchEndpoint: '/api/products/search',
            cacheKey: 'pos_products_cache',
            cacheDuration: 5 * 60 * 1000, // 5 minutes
            retryAttempts: 3,
            retryDelay: 1000,
            requestTimeout: 15000, // 15 seconds
            itemsPerPage: 50,
            debug: false
        };
        
        this.state = {
            products: [],
            filteredProducts: [],
            currentPage: 1,
            isLoading: false,
            searchQuery: '',
            currentCategory: 'all',
            hasMore: true,
            error: null,
            lastUpdate: null
        };
        
        this.initialize();
    }
    
    initialize() {
        console.log('🔄 Initializing POS Products module...');
        
        // Set up event listeners
        this.setupEventListeners();
        
        // Load initial products
        this.loadInitialProducts();
        
        // Set up error monitoring
        this.setupErrorMonitoring();
        
        // Set up periodic refresh
        this.setupPeriodicRefresh();
        
        console.log('✅ POS Products module initialized');
    }
    
    setupEventListeners() {
        // Search input
        const searchInput = document.getElementById('productSearch');
        if (searchInput) {
            let searchTimeout;
            
            searchInput.addEventListener('input', (e) => {
                clearTimeout(searchTimeout);
                const query = e.target.value.trim();
                
                if (query.length >= 2) {
                    searchTimeout = setTimeout(() => {
                        this.performSearch(query);
                    }, 350);
                } else if (query.length === 0) {
                    this.clearSearch();
                }
            });
            
            searchInput.addEventListener('keydown', (e) => {
                if (e.key === 'Escape') {
                    searchInput.value = '';
                    this.clearSearch();
                } else if (e.key === 'Enter') {
                    this.performSearch(searchInput.value);
                }
            });
            
            // Auto-focus with delay
            setTimeout(() => {
                if (!document.activeElement || document.activeElement.tagName === 'BODY') {
                    searchInput.focus();
                }
            }, 1000);
        }
        
        // Category filter
        const categoryFilter = document.getElementById('categoryFilter');
        if (categoryFilter) {
            categoryFilter.addEventListener('change', (e) => {
                this.filterByCategory(e.target.value);
            });
        }
        
        // View mode toggle
        const viewModeBtn = document.querySelector('[onclick*="toggleViewMode"]');
        if (viewModeBtn) {
            viewModeBtn.addEventListener('click', () => this.toggleViewMode());
        }
        
        // Low stock filter
        const lowStockBtn = document.querySelector('[onclick*="showLowStock"]');
        if (lowStockBtn) {
            lowStockBtn.addEventListener('click', () => this.showLowStock());
        }
        
        // Load more products on scroll
        window.addEventListener('scroll', () => this.handleScroll());
        
        // Network status monitoring
        window.addEventListener('online', () => this.handleNetworkOnline());
        window.addEventListener('offline', () => this.handleNetworkOffline());
    }
    
    setupErrorMonitoring() {
        window.addEventListener('error', (e) => {
            if (e.filename && e.filename.includes('products.js')) {
                console.error('Products module error:', e.error);
                this.logError('JavaScript Error', e.error);
            }
        });
        
        window.addEventListener('unhandledrejection', (e) => {
            console.error('Unhandled promise rejection:', e.reason);
            this.logError('Promise Rejection', e.reason);
        });
    }
    
    setupPeriodicRefresh() {
        // Refresh products every 5 minutes
        setInterval(() => {
            if (document.visibilityState === 'visible' && !this.state.isLoading) {
                this.refreshProducts();
            }
        }, 5 * 60 * 1000);
    }
    
    async loadInitialProducts() {
        console.log('📦 Loading initial products...');
        
        // Show loading state
        this.showLoading(true);
        
        try {
            // Try to load from cache first for instant display
            const cachedProducts = this.getCachedProducts();
            if (cachedProducts && cachedProducts.length > 0) {
                console.log('📦 Found cached products:', cachedProducts.length);
                this.state.products = cachedProducts;
                this.state.filteredProducts = [...cachedProducts];
                this.renderProducts();
                
                // Show toast about using cached data
                this.showToast('Using cached products. Loading latest...', 'info');
            }
            
            // Load fresh data from API
            await this.loadProductsFromAPI();
            
            this.showLoading(false);
            
        } catch (error) {
            console.error('Failed to load initial products:', error);
            this.handleLoadError(error);
        }
    }
    
 // Change this in products.js
async loadProductsFromAPI(options = {}) {
    const startTime = performance.now();
    
    this.state.isLoading = true;
    this.state.error = null;
    
    try {
        // Build query parameters
        const params = new URLSearchParams();
        
        if (options.search) {
            params.append('q', options.search);
        }
        
        if (options.categoryId && options.categoryId !== 'all') {
            params.append('category_id', options.categoryId);
        }
        
        if (options.page) {
            params.append('page', options.page);
        }
        
        // Add authentication token
        const token = document.querySelector('meta[name="csrf-token"]')?.content;
        
        // Use the correct API endpoint
        let url = '/api/v1/products';
        if (params.toString()) {
            url += '?' + params.toString();
        }
        
        console.log('🌐 Fetching products from:', url);
        
        const response = await fetch(url, {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
                'X-Requested-With': 'XMLHttpRequest',
                'X-CSRF-TOKEN': token || ''
            },
            credentials: 'same-origin'
        });
        
        if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        
        const data = await response.json();
        
        const loadTime = performance.now() - startTime;
        console.log(`✅ Products loaded in ${loadTime.toFixed(0)}ms`, data);
        
        if (data.success) {
            // Process the response
            this.processAPIResponse(data, options);
            return true;
        } else {
            throw new Error(data.error || data.message || 'API returned failure');
        }
        
    } catch (error) {
        console.error('API load error:', error);
        this.state.error = error.message;
        
        // Fallback to direct route
        await this.tryFallbackAPI(options);
        
        return false;
    } finally {
        this.state.isLoading = false;
        this.state.lastUpdate = new Date();
        this.updateLastUpdateDisplay();
    }
}

// Add fallback method
async tryFallbackAPI(options = {}) {
    try {
        console.log('🔄 Trying fallback API...');
        
        // Try the web route instead
        const params = new URLSearchParams();
        if (options.search) {
            params.append('search', options.search);
        }
        if (options.categoryId && options.categoryId !== 'all') {
            params.append('category_id', options.categoryId);
        }
        
        const url = '/products/search' + (params.toString() ? '?' + params.toString() : '');
        
        const response = await fetch(url, {
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        });
        
        const html = await response.text();
        console.log('Fallback response length:', html.length);
        
        // Parse HTML for product data if needed
        // This is a temporary fallback
        
    } catch (fallbackError) {
        console.error('Fallback also failed:', fallbackError);
    }
}
    
    async makeRequest(url, options = {}) {
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), this.config.requestTimeout);
        
        const requestOptions = {
            method: options.method || 'GET',
            headers: {
                'Accept': 'application/json',
                'X-Requested-With': 'XMLHttpRequest',
                'X-Request-ID': this.generateRequestId(),
                ...options.headers
            },
            signal: controller.signal,
            credentials: 'same-origin'
        };
        
        // Add CSRF token for non-GET requests
        if (requestOptions.method !== 'GET') {
            const csrfToken = document.querySelector('meta[name="csrf-token"]')?.content;
            if (csrfToken) {
                requestOptions.headers['X-CSRF-TOKEN'] = csrfToken;
            }
        }
        
        // Retry logic
        let lastError;
        for (let attempt = 1; attempt <= this.config.retryAttempts; attempt++) {
            try {
                const response = await fetch(url, requestOptions);
                clearTimeout(timeoutId);
                
                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                }
                
                return response;
                
            } catch (error) {
                lastError = error;
                
                if (attempt < this.config.retryAttempts) {
                    console.log(`🔄 Retry attempt ${attempt}/${this.config.retryAttempts}`);
                    await this.sleep(this.config.retryDelay * attempt);
                }
            }
        }
        
        clearTimeout(timeoutId);
        throw lastError;
    }
    
    processAPIResponse(data, options) {
        const newProducts = data.products || [];
        
        if (options.page && options.page > 1) {
            // Append to existing products for pagination
            this.state.products = [...this.state.products, ...newProducts];
        } else {
            // Replace products
            this.state.products = newProducts;
        }
        
        this.state.filteredProducts = [...this.state.products];
        this.state.hasMore = data.pagination?.has_more_pages || false;
        
        // Update cache
        this.cacheProducts(this.state.products);
        
        // Update stats
        this.updateProductStats(data.stats);
        
        // Render products
        this.renderProducts();
        
        // Show success message for first load
        if (!options.page || options.page === 1) {
            const message = `Loaded ${newProducts.length} products`;
            if (this.config.debug) {
                this.showToast(message, 'success');
            }
        }
    }
    
    async performSearch(query) {
        if (this.state.searchQuery === query) return;
        
        this.state.searchQuery = query;
        this.state.currentPage = 1;
        this.showLoading(true);
        
        console.log(`🔍 Searching for: "${query}"`);
        
        // Show search indicator
        this.showSearchIndicator(query);
        
        try {
            await this.loadProductsFromAPI({
                search: query,
                page: 1
            });
            
        } catch (error) {
            console.error('Search error:', error);
            
            // Fallback: Filter cached products
            const filtered = this.state.products.filter(product => 
                product.name.toLowerCase().includes(query.toLowerCase()) ||
                (product.sku && product.sku.toLowerCase().includes(query.toLowerCase())) ||
                (product.barcode && product.barcode.toLowerCase().includes(query.toLowerCase()))
            );
            
            this.state.filteredProducts = filtered;
            this.renderProducts();
            this.showSearchResults(filtered.length, true);
        }
        
        this.showLoading(false);
    }
    
    clearSearch() {
        const searchInput = document.getElementById('productSearch');
        if (searchInput) searchInput.value = '';
        
        this.state.searchQuery = '';
        this.state.filteredProducts = [...this.state.products];
        this.renderProducts();
        
        // Hide search indicator
        this.hideSearchIndicator();
    }
    
    filterByCategory(categoryId) {
        console.log(`🏷️ Filtering by category: ${categoryId}`);
        
        this.state.currentCategory = categoryId;
        this.state.currentPage = 1;
        
        if (categoryId === 'all') {
            this.state.filteredProducts = [...this.state.products];
        } else {
            this.state.filteredProducts = this.state.products.filter(
                product => product.category_id == categoryId
            );
        }
        
        this.renderProducts();
        
        // Show category filter status
        if (categoryId !== 'all') {
            const categoryName = document.querySelector(`#categoryFilter option[value="${categoryId}"]`)?.textContent;
            this.showToast(`Showing ${this.state.filteredProducts.length} products in "${categoryName}"`, 'info', 2000);
        }
    }
    
    renderProducts() {
        const container = document.getElementById('productGrid');
        if (!container) {
            console.error('Product grid container not found');
            return;
        }
        
        // Clear container
        container.innerHTML = '';
        
        if (!this.state.filteredProducts || this.state.filteredProducts.length === 0) {
            this.renderNoProducts();
            return;
        }
        
        // Create product cards
        this.state.filteredProducts.forEach(product => {
            const productCard = this.createProductCard(product);
            container.appendChild(productCard);
        });
        
        // Add load more button if there are more products
        if (this.state.hasMore && !this.state.searchQuery) {
            this.renderLoadMoreButton();
        }
        
        // Update product count
        this.updateProductCount();
    }
    
    createProductCard(product) {
        const template = document.getElementById('productCardTemplate');
        if (template) {
            return this.createProductCardFromTemplate(product, template);
        }
        
        // Fallback: Create card manually
        return this.createProductCardManually(product);
    }
    
    createProductCardFromTemplate(product, template) {
        const card = template.content.cloneNode(true);
        const cardElement = card.querySelector('.product-card');
        
        // Store product data
        cardElement.dataset.productId = product.id;
        cardElement.dataset.productName = product.name;
        cardElement.dataset.productPrice = product.sale_price || product.price || 0;
        cardElement.dataset.productStock = product.stock || 0;
        
        // Fill in data
        card.querySelector('[data-field="name"]').textContent = product.name || 'Unknown Product';
        card.querySelector('[data-field="sku"]').textContent = product.sku || 'N/A';
        card.querySelector('[data-field="barcode"]').textContent = product.barcode || 'N/A';
        card.querySelector('[data-field="price"]').textContent = this.formatCurrency(product.sale_price || product.price || 0);
        card.querySelector('[data-field="cost"]').textContent = `Cost: ${this.formatCurrency(product.cost_price || 0)}`;
        card.querySelector('[data-field="stock"]').textContent = `${product.stock || 0} in stock`;
        
        // Update badges
        const stockBadge = card.querySelector('[data-badge-type="stock"]');
        const vatBadge = card.querySelector('[data-badge-type="vat"]');
        const lowStockBadge = card.querySelector('[data-badge-type="low-stock"]');
        
        if (!product.stock || product.stock <= 0) {
            stockBadge.textContent = 'Out of Stock';
            stockBadge.className = 'badge bg-danger';
            cardElement.classList.add('out-of-stock');
        } else if (product.stock <= 5) {
            stockBadge.textContent = 'Low Stock';
            stockBadge.className = 'badge bg-warning';
            if (lowStockBadge) lowStockBadge.classList.remove('d-none');
        } else {
            stockBadge.textContent = 'In Stock';
            stockBadge.className = 'badge bg-success';
        }
        
        if (product.has_vat || product.is_vatable) {
            if (vatBadge) vatBadge.classList.remove('d-none');
        }
        
        // Add event listeners
        this.setupProductCardEvents(cardElement, product);
        
        return card;
    }
    
    createProductCardManually(product) {
        const card = document.createElement('div');
        card.className = 'product-card';
        
        // Determine stock status
        let stockClass = 'success';
        let stockText = 'In Stock';
        let isOutOfStock = false;
        
        if (!product.stock || product.stock <= 0) {
            stockClass = 'danger';
            stockText = 'Out of Stock';
            isOutOfStock = true;
            card.classList.add('out-of-stock');
        } else if (product.stock <= 5) {
            stockClass = 'warning';
            stockText = 'Low Stock';
        }
        
        card.innerHTML = `
            <div class="product-card-inner">
                <div class="product-image">
                    <div class="image-placeholder">
                        <i class="bx bx-package"></i>
                    </div>
                    <div class="product-badges">
                        <span class="badge bg-${stockClass}">${stockText}</span>
                        ${product.has_vat ? '<span class="badge bg-info">VAT</span>' : ''}
                    </div>
                </div>
                <div class="product-info">
                    <div class="product-name">${product.name || 'Unknown Product'}</div>
                    <div class="product-meta">
                        <span class="sku">${product.sku || 'N/A'}</span>
                        <span class="barcode">${product.barcode || 'N/A'}</span>
                    </div>
                    <div class="product-pricing">
                        <div class="price">${this.formatCurrency(product.sale_price || product.price || 0)}</div>
                        <div class="cost text-muted small">Cost: ${this.formatCurrency(product.cost_price || 0)}</div>
                    </div>
                    <div class="product-stock">${product.stock || 0} in stock</div>
                    <div class="product-actions">
                        <button class="btn btn-sm btn-outline-primary add-to-cart-btn" ${isOutOfStock ? 'disabled' : ''}>
                            <i class="bx bx-plus"></i> Add
                        </button>
                        <div class="btn-group btn-group-sm">
                            <button class="btn btn-outline-secondary view-details-btn">
                                <i class="bx bx-info-circle"></i>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        // Store product data
        card.dataset.productId = product.id;
        card.dataset.productName = product.name;
        card.dataset.productPrice = product.sale_price || product.price || 0;
        card.dataset.productStock = product.stock || 0;
        
        // Add event listeners
        this.setupProductCardEvents(card, product);
        
        return card;
    }
    
    setupProductCardEvents(cardElement, product) {
        // Add to cart button
        const addBtn = cardElement.querySelector('.add-to-cart-btn');
        if (addBtn && (!product.stock || product.stock > 0)) {
            addBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                this.addToCart(product, 1);
            });
        }
        
        // View details button
        const detailsBtn = cardElement.querySelector('.view-details-btn');
        if (detailsBtn) {
            detailsBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                this.viewProductDetails(product.id);
            });
        }
        
        // Card click (add to cart)
        if (!product.stock || product.stock > 0) {
            cardElement.addEventListener('click', (e) => {
                if (!e.target.closest('.btn') && !e.target.closest('.btn-group')) {
                    this.addToCart(product, 1);
                }
            });
            
            // Keyboard support
            cardElement.addEventListener('keypress', (e) => {
                if ((e.key === 'Enter' || e.key === ' ') && !cardElement.classList.contains('out-of-stock')) {
                    this.addToCart(product, 1);
                }
            });
        }
        
        // Make card focusable for accessibility
        cardElement.tabIndex = 0;
        cardElement.setAttribute('role', 'button');
        cardElement.setAttribute('aria-label', `Add ${product.name} to cart. Price: ${this.formatCurrency(product.sale_price || product.price || 0)}`);
    }
    
    addToCart(product, quantity = 1) {
        if (!product || !product.id) {
            console.error('Invalid product data');
            return false;
        }
        
        // Check stock
        if (product.stock !== undefined && product.stock < quantity) {
            this.showToast(`Insufficient stock. Only ${product.stock} available`, 'warning');
            return false;
        }
        
        // Use global cart function if available
        if (window.addToCart) {
            return window.addToCart(product, quantity);
        }
        
        // Fallback: Direct cart manipulation
        if (window.POSCart && window.POSCart.add) {
            window.POSCart.add(product, quantity);
            return true;
        }
        
        // Last resort: Show message
        this.showToast(`${product.name} added to cart`, 'success');
        return true;
    }
    
    viewProductDetails(productId) {
        console.log('View product details:', productId);
        // Implement product details modal
        this.showToast('Product details feature coming soon', 'info');
    }
    
    renderNoProducts() {
        const container = document.getElementById('productGrid');
        if (!container) return;
        
        let message = 'No products found';
        let icon = 'bx-package';
        
        if (this.state.searchQuery) {
            message = `No products found for "${this.state.searchQuery}"`;
            icon = 'bx-search';
        } else if (this.state.currentCategory !== 'all') {
            message = 'No products in this category';
            icon = 'bx-category';
        }
        
        container.innerHTML = `
            <div class="col-12 text-center py-5">
                <i class="bx ${icon} bx-lg text-muted mb-3"></i>
                <p class="text-muted mb-3">${message}</p>
                ${this.state.searchQuery ? 
                    '<button class="btn btn-sm btn-outline-primary me-2" onclick="POSProducts.clearSearch()">Clear Search</button>' : ''}
                <button class="btn btn-sm btn-outline-primary" onclick="POSProducts.refreshProducts()">
                    <i class="bx bx-refresh me-1"></i> Refresh
                </button>
            </div>
        `;
    }
    
    renderLoadMoreButton() {
        const container = document.getElementById('productGrid');
        if (!container) return;
        
        const loadMoreBtn = document.createElement('div');
        loadMoreBtn.className = 'col-12 text-center mt-3';
        loadMoreBtn.innerHTML = `
            <button class="btn btn-outline-primary" onclick="POSProducts.loadMore()" ${this.state.isLoading ? 'disabled' : ''}>
                ${this.state.isLoading ? 
                    '<span class="spinner-border spinner-border-sm me-2"></span> Loading...' : 
                    '<i class="bx bx-plus me-1"></i> Load More'}
            </button>
        `;
        
        container.appendChild(loadMoreBtn);
    }
    
    async loadMore() {
        if (this.state.isLoading || !this.state.hasMore) return;
        
        this.state.currentPage++;
        this.state.isLoading = true;
        
        try {
            await this.loadProductsFromAPI({
                page: this.state.currentPage,
                categoryId: this.state.currentCategory !== 'all' ? this.state.currentCategory : null
            });
        } catch (error) {
            console.error('Load more error:', error);
            this.state.currentPage--; // Roll back page on error
        }
        
        this.state.isLoading = false;
    }
    
    handleScroll() {
        if (this.state.isLoading || !this.state.hasMore || this.state.searchQuery) return;
        
        const container = document.getElementById('productGrid');
        if (!container) return;
        
        const scrollPosition = window.innerHeight + window.scrollY;
        const containerBottom = container.offsetTop + container.offsetHeight;
        
        if (scrollPosition >= containerBottom - 100) {
            this.loadMore();
        }
    }
    
    async refreshProducts() {
        console.log('🔄 Refreshing products...');
        
        this.state.currentPage = 1;
        await this.loadProductsFromAPI({
            page: 1,
            categoryId: this.state.currentCategory !== 'all' ? this.state.currentCategory : null
        });
        
        this.showToast('Products refreshed', 'success');
    }
    
    showLowStock() {
        const lowStockProducts = this.state.products.filter(
            product => product.stock > 0 && product.stock <= (product.minimum_stock || 5)
        );
        
        this.state.filteredProducts = lowStockProducts;
        this.renderProducts();
        
        this.showToast(`Showing ${lowStockProducts.length} low stock products`, 'warning');
    }
    
    toggleViewMode() {
        const currentView = document.querySelector('.product-grid') ? 'grid' : 'table';
        const newView = currentView === 'grid' ? 'table' : 'grid';
        
        // Update UI
        const icon = document.getElementById('viewModeIcon');
        const text = document.getElementById('viewModeText');
        
        if (icon && text) {
            if (newView === 'grid') {
                icon.className = 'bx bx-grid';
                text.textContent = 'Grid View';
            } else {
                icon.className = 'bx bx-list-ul';
                text.textContent = 'List View';
            }
        }
        
        // Switch view (you would need to implement table view)
        this.showToast('View mode toggle coming soon', 'info');
    }
    
    // Caching methods
    cacheProducts(products) {
        try {
            const cacheData = {
                products: products,
                timestamp: Date.now(),
                count: products.length
            };
            
            localStorage.setItem(this.config.cacheKey, JSON.stringify(cacheData));
            
            if (this.config.debug) {
                console.log(`💾 Cached ${products.length} products`);
            }
        } catch (error) {
            console.warn('Failed to cache products:', error);
        }
    }
    
    getCachedProducts() {
        try {
            const cached = localStorage.getItem(this.config.cacheKey);
            if (!cached) return null;
            
            const cacheData = JSON.parse(cached);
            const cacheAge = Date.now() - cacheData.timestamp;
            
            if (cacheAge > this.config.cacheDuration) {
                console.log('💾 Cache expired, clearing...');
                localStorage.removeItem(this.config.cacheKey);
                return null;
            }
            
            return cacheData.products || [];
        } catch (error) {
            console.warn('Failed to read cache:', error);
            return null;
        }
    }
    
    clearCache() {
        localStorage.removeItem(this.config.cacheKey);
        console.log('💾 Cache cleared');
    }
    
    // UI Helper methods
    showLoading(show) {
        const container = document.getElementById('productGrid');
        if (!container) return;
        
        if (show) {
            const loadingHtml = `
                <div class="col-12 text-center py-5">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Loading products...</span>
                    </div>
                    <p class="mt-2 text-muted">Loading products...</p>
                </div>
            `;
            
            // Only show loading if container is empty
            if (container.innerHTML.trim() === '') {
                container.innerHTML = loadingHtml;
            }
        }
        
        // Update global loading state if available
        if (window.POSUtils && window.POSUtils.showLoading) {
            window.POSUtils.showLoading(show, 'products');
        }
    }
    
    showError(message) {
        console.error('Product error:', message);
        
        const container = document.getElementById('productGrid');
        if (container) {
            container.innerHTML = `
                <div class="col-12 text-center py-5">
                    <i class="bx bx-error-circle bx-lg text-danger mb-3"></i>
                    <p class="text-danger mb-2">${message}</p>
                    <div class="mt-3">
                        <button class="btn btn-sm btn-outline-primary me-2" onclick="POSProducts.refreshProducts()">
                            <i class="bx bx-refresh me-1"></i> Try Again
                        </button>
                        <button class="btn btn-sm btn-outline-secondary" onclick="POSProducts.clearCache()">
                            <i class="bx bx-trash me-1"></i> Clear Cache
                        </button>
                    </div>
                </div>
            `;
        }
        
        this.showToast(message, 'error');
    }
    
    showToast(message, type = 'info', duration = 3000) {
        if (window.POSUtils && window.POSUtils.showToast) {
            window.POSUtils.showToast(message, type);
            return;
        }
        
        // Fallback toast
        const toastId = 'product-toast-' + Date.now();
        const toastHtml = `
            <div id="${toastId}" class="toast align-items-center border-0 bg-${type === 'error' ? 'danger' : type === 'success' ? 'success' : 'info'} text-white" role="alert">
                <div class="d-flex">
                    <div class="toast-body">
                        <i class="bx bx-${type === 'success' ? 'check-circle' : type === 'error' ? 'error-circle' : 'info-circle'} me-2"></i>
                        ${message}
                    </div>
                    <button type="button" class="btn-close btn-close-white me-2 m-auto" onclick="document.getElementById('${toastId}').remove()"></button>
                </div>
            </div>
        `;
        
        const container = document.getElementById('toastContainer') || (() => {
            const div = document.createElement('div');
            div.id = 'toastContainer';
            div.className = 'position-fixed top-0 end-0 p-3';
            div.style.zIndex = '9999';
            document.body.appendChild(div);
            return div;
        })();
        
        container.insertAdjacentHTML('beforeend', toastHtml);
        
        // Auto-remove
        setTimeout(() => {
            const toast = document.getElementById(toastId);
            if (toast) toast.remove();
        }, duration);
    }
    
    showSearchIndicator(query) {
        let indicator = document.getElementById('searchIndicator');
        if (!indicator) {
            indicator = document.createElement('div');
            indicator.id = 'searchIndicator';
            indicator.className = 'position-fixed top-0 start-50 translate-middle-x mt-3';
            indicator.style.zIndex = '999';
            document.body.appendChild(indicator);
        }
        
        indicator.innerHTML = `
            <div class="badge bg-info">
                <i class="bx bx-search me-1"></i> Searching: "${query}"
            </div>
        `;
    }
    
    showSearchResults(count, fromCache = false) {
        const indicator = document.getElementById('searchIndicator');
        if (indicator) {
            indicator.innerHTML = `
                <div class="badge ${fromCache ? 'bg-warning' : 'bg-success'}">
                    <i class="bx bx-check me-1"></i> Found ${count} products
                    ${fromCache ? ' (from cache)' : ''}
                </div>
            `;
            
            setTimeout(() => {
                if (indicator) indicator.innerHTML = '';
            }, 3000);
        }
    }
    
    hideSearchIndicator() {
        const indicator = document.getElementById('searchIndicator');
        if (indicator) {
            indicator.innerHTML = '';
        }
    }
    
    updateProductCount() {
        const countElement = document.getElementById('productCount');
        if (countElement) {
            countElement.textContent = this.state.filteredProducts.length;
        }
    }
    
    updateProductStats(stats) {
        if (!stats) return;
        
        // Update UI elements with stats
        const statsElement = document.getElementById('productStats');
        if (statsElement) {
            statsElement.innerHTML = `
                <small class="text-muted">
                    ${stats.total_products} total | ${stats.in_stock_count} in stock
                </small>
            `;
        }
    }
    
    updateLastUpdateDisplay() {
        const updateElement = document.getElementById('lastUpdate');
        if (updateElement && this.state.lastUpdate) {
            const timeAgo = this.formatTimeAgo(this.state.lastUpdate);
            updateElement.innerHTML = `<small class="text-muted">Updated ${timeAgo}</small>`;
        }
    }
    
    // Network event handlers
    handleNetworkOnline() {
        console.log('🌐 Network is back online');
        this.showToast('Back online. Syncing products...', 'success');
        
        // Refresh products after coming online
        setTimeout(() => this.refreshProducts(), 2000);
    }
    
    handleNetworkOffline() {
        console.log('🌐 Network is offline');
        this.showToast('You are offline. Using cached products.', 'warning');
    }
    
    // Error handling
    handleLoadError(error) {
        this.showLoading(false);
        
        // Check error type
        if (error.name === 'AbortError') {
            this.showError('Request timeout. Please try again.');
        } else if (error.message.includes('Network')) {
            this.showError('Network error. Please check your connection.');
        } else {
            this.showError('Failed to load products. Please try again.');
        }
        
        this.logError('Product Load Error', error);
    }
    
    logError(type, error) {
        const errorData = {
            type: type,
            message: error.message,
            stack: error.stack,
            timestamp: new Date().toISOString(),
            url: window.location.href,
            userAgent: navigator.userAgent,
            productsCount: this.state.products.length
        };
        
        // Log to console
        console.error('📝 Error Log:', errorData);
        
        // Send to server if online
        if (navigator.onLine && window.fetch) {
            fetch('/api/log-error', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || ''
                },
                body: JSON.stringify({
                    error: errorData,
                    module: 'POSProducts'
                })
            }).catch(() => {
                // Silent fail for error logging
            });
        }
    }
    
    // Utility methods
    formatCurrency(amount) {
        if (window.POSUtils && window.POSUtils.formatCurrency) {
            return window.POSUtils.formatCurrency(amount);
        }
        
        const formatter = new Intl.NumberFormat('en-KE', {
            style: 'currency',
            currency: 'KES',
            minimumFractionDigits: 2,
            maximumFractionDigits: 2
        });
        
        return formatter.format(amount || 0);
    }
    
    formatTimeAgo(date) {
        const seconds = Math.floor((new Date() - date) / 1000);
        
        if (seconds < 60) return 'just now';
        if (seconds < 3600) return `${Math.floor(seconds / 60)} minutes ago`;
        if (seconds < 86400) return `${Math.floor(seconds / 3600)} hours ago`;
        return `${Math.floor(seconds / 86400)} days ago`;
    }
    
    generateRequestId() {
        return 'req_' + Date.now() + '_' + Math.random().toString(36).substr(2, 9);
    }
    
    sleep(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }
    
    // Public API methods
    static getInstance() {
        if (!window._posProductsInstance) {
            window._posProductsInstance = new POSProducts();
        }
        return window._posProductsInstance;
    }
}

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    // Give other scripts time to load
    setTimeout(() => {
        try {
            window.POSProducts = POSProducts.getInstance();
            console.log('🎯 POS Products module ready');
        } catch (error) {
            console.error('Failed to initialize POS Products:', error);
            
            // Emergency fallback
            window.POSProducts = {
                loadAll: () => console.log('Products module failed to load'),
                refreshProducts: () => location.reload()
            };
        }
    }, 500);
});

// Export for global use
if (typeof module !== 'undefined' && module.exports) {
    module.exports = POSProducts;
}