// =========================================================
// POS Payments Module v2.3 (Fixed)
// =========================================================

(function() {
    'use strict';
    
    class POSPayments {
        constructor() {
            this.version = '2.3';
            this.initialized = false;
            this.paymentManager = null;
            
            console.log(`💰 POS Payments v${this.version} initializing...`);
        }
        
        async init() {
            if (this.initialized) return this;
            
            try {
                // Setup global reference immediately
                window.POSPayments = this;
                window.showPaymentModal = this.showModal.bind(this);
                window.selectPaymentMethod = this.selectPaymentMethod.bind(this);
                
                // Wait for PaymentManager
                this.paymentManager = await this.waitForPaymentManager();
                
                console.log('✅ POS Payments initialized');
                this.initialized = true;
                
                return this;
            } catch (error) {
                console.error('❌ POS Payments initialization failed:', error);
                // Setup minimal functionality anyway
                this.setupMinimalFunctionality();
                return this;
            }
        }
        
        async waitForPaymentManager() {
            return new Promise((resolve, reject) => {
                let attempts = 0;
                const maxAttempts = 30;
                
                const check = () => {
                    attempts++;
                    
                    if (window.PaymentManager && window.PaymentManager.initialized) {
                        console.log(`💰 Payment Manager found after ${attempts} attempts`);
                        resolve(window.PaymentManager);
                    } else if (attempts >= maxAttempts) {
                        console.warn('PaymentManager not found, using fallback');
                        resolve(this.createFallbackManager());
                    } else {
                        setTimeout(check, 100);
                    }
                };
                
                check();
            });
        }
        
        createFallbackManager() {
            console.warn('Creating fallback Payment Manager');
            
            return {
                showModal: function() {
                    const modal = document.getElementById('paymentModal');
                    if (modal) {
                        try {
                            const bsModal = new bootstrap.Modal(modal);
                            bsModal.show();
                            return true;
                        } catch (e) {
                            console.error('Cannot show payment modal:', e);
                            return false;
                        }
                    }
                    return false;
                },
                
                selectPaymentMethod: function(method) {
                    document.querySelectorAll('.payment-method-card').forEach(card => {
                        card.classList.remove('selected');
                    });
                    
                    const selectedCard = document.querySelector(`[data-method="${method}"]`);
                    if (selectedCard) {
                        selectedCard.classList.add('selected');
                    }
                    
                    // Load form template
                    const container = document.getElementById('paymentFormContainer');
                    const template = document.getElementById(`${method}FormTemplate`);
                    
                    if (container && template) {
                        container.innerHTML = template.innerHTML;
                    }
                },
                
                initialized: true
            };
        }
        
        setupMinimalFunctionality() {
            console.log('⚠️ Setting up minimal payment functionality');
            
            // Make sure showModal is always available
            this.showModal = function() {
                const modal = document.getElementById('paymentModal');
                if (modal) {
                    try {
                        const bsModal = new bootstrap.Modal(modal);
                        bsModal.show();
                        return true;
                    } catch (e) {
                        console.error('Cannot show payment modal:', e);
                        alert('Payment system not ready. Please try again.');
                        return false;
                    }
                }
                return false;
            };
            
            this.selectPaymentMethod = function(method) {
                document.querySelectorAll('.payment-method-card').forEach(card => {
                    card.classList.remove('selected');
                });
                
                const selectedCard = document.querySelector(`[data-method="${method}"]`);
                if (selectedCard) {
                    selectedCard.classList.add('selected');
                }
            };
            
            this.initialized = true;
        }
        
        showModal() {
            console.log('💰 POSPayments.showModal called');
            
            if (!this.initialized) {
                this.init().catch(() => {
                    // Silent fail
                });
            }
            
            if (this.paymentManager && this.paymentManager.showModal) {
                return this.paymentManager.showModal();
            } else if (window.PaymentManager && window.PaymentManager.showModal) {
                return window.PaymentManager.showModal();
            } else {
                // Fallback to direct modal show
                const modal = document.getElementById('paymentModal');
                if (modal) {
                    try {
                        const bsModal = new bootstrap.Modal(modal);
                        bsModal.show();
                        return true;
                    } catch (e) {
                        console.error('Cannot show payment modal:', e);
                        return false;
                    }
                }
                return false;
            }
        }
        
        selectPaymentMethod(method) {
            if (this.paymentManager && this.paymentManager.selectPaymentMethod) {
                return this.paymentManager.selectPaymentMethod(method);
            } else if (window.PaymentManager && window.PaymentManager.selectPaymentMethod) {
                return window.PaymentManager.selectPaymentMethod(method);
            } else {
                // Fallback
                document.querySelectorAll('.payment-method-card').forEach(card => {
                    card.classList.remove('selected');
                });
                
                const selectedCard = document.querySelector(`[data-method="${method}"]`);
                if (selectedCard) {
                    selectedCard.classList.add('selected');
                }
            }
        }
        
        // Add other payment methods as needed...
        generateInvoiceNumber() {
            if (this.paymentManager && this.paymentManager.generateInvoiceNumber) {
                return this.paymentManager.generateInvoiceNumber();
            } else if (window.PaymentManager && window.PaymentManager.generateInvoiceNumber) {
                return window.PaymentManager.generateInvoiceNumber();
            }
            return `INV-${Date.now()}`;
        }
        
        generateReceiptNumber() {
            if (this.paymentManager && this.paymentManager.generateReceiptNumber) {
                return this.paymentManager.generateReceiptNumber();
            } else if (window.PaymentManager && window.PaymentManager.generateReceiptNumber) {
                return window.PaymentManager.generateReceiptNumber();
            }
            return `RCP-${Date.now()}`;
        }
    }
    
    // Create and export instance
    const posPaymentsInstance = new POSPayments();
    
    // Initialize immediately
    setTimeout(() => {
        posPaymentsInstance.init().then(() => {
            console.log('💰 POS Payments ready v2.3');
        }).catch(error => {
            console.error('Failed to initialize payments:', error);
            // Even if init fails, ensure basic functionality
            posPaymentsInstance.setupMinimalFunctionality();
        });
    }, 500);
    
    // Make available globally
    window.POSPayments = posPaymentsInstance;
    window.POSPayments.showModal = posPaymentsInstance.showModal.bind(posPaymentsInstance);
    window.POSPayments.selectPaymentMethod = posPaymentsInstance.selectPaymentMethod.bind(posPaymentsInstance);
    
    // Export for module systems
    if (typeof module !== 'undefined' && module.exports) {
        module.exports = POSPayments;
    }
    
})();