// public/js/pos/modules/payment/payment-vat-helper.js
// Payment VAT Helper - Manages VAT display in payment modal

class PaymentVATHelper {
    constructor() {
        this.version = '1.0.0';
        this.vatData = null;
    }
    
    /**
     * Calculate and update VAT display in payment modal
     */
    updatePaymentModalVAT() {
        console.log('🔄 Updating payment modal VAT display...');
        
        try {
            // Get cart data
            const cartData = this.getCurrentCartData();
            if (!cartData || !cartData.items || cartData.items.length === 0) {
                this.resetVATDisplay();
                return;
            }
            
            // Calculate VAT using VATService
            let vatData;
            if (window.VATService && window.VATService.calculateCartVAT) {
                vatData = window.VATService.calculateCartVAT(
                    cartData.items,
                    cartData.customer,
                    cartData.discount || 0
                );
            } else {
                // Fallback calculation
                vatData = this.calculateVATFallback(cartData);
            }
            
            this.vatData = vatData;
            
            // Update all VAT displays
            this.updateAllVATElements(vatData);
            
            console.log('✅ Payment modal VAT updated:', {
                hasVAT: vatData.hasVAT,
                totalVAT: vatData.totalVAT,
                total: vatData.totalWithVAT
            });
            
        } catch (error) {
            console.error('❌ Error updating payment modal VAT:', error);
            this.resetVATDisplay();
        }
    }
    
    /**
     * Get current cart data
     */
    getCurrentCartData() {
        try {
            if (window.posState) {
                const items = window.posState.cart || [];
                const subtotal = window.posState.getCartSubtotal ? 
                    window.posState.getCartSubtotal() : 
                    items.reduce((sum, item) => sum + (item.price * (item.quantity || 1)), 0);
                const discount = window.posState.currentDiscount || 0;
                const customer = window.posState.currentCustomer || null;
                
                return {
                    items: items,
                    subtotal: subtotal,
                    discount: discount,
                    customer: customer
                };
            }
        } catch (error) {
            console.error('Error getting cart data:', error);
        }
        
        return {
            items: [],
            subtotal: 0,
            discount: 0,
            customer: null
        };
    }
    
    /**
     * Fallback VAT calculation
     */
    calculateVATFallback(cartData) {
        let subtotal = 0;
        let totalVAT = 0;
        let taxableAmount = 0;
        let nonTaxableAmount = 0;
        const vatByRate = {};
        
        const customer = cartData.customer;
        const customerStatus = customer?.vat_status || 'vatable';
        const isExempted = customerStatus === 'exempted';
        const isZeroRated = customerStatus === 'zero_rated';
        
        cartData.items.forEach(item => {
            const itemTotal = item.price * (item.quantity || 1);
            subtotal += itemTotal;
            
            // Check if item is VATable - CORRECTED LOGIC
            const isProductVatable = item.has_vat === true || item.is_vatable === true || false;
            const vatRate = parseFloat(item.tax_rate) || 0;
            
            if (isExempted) {
                // No VAT for exempted customers
                nonTaxableAmount += itemTotal;
            } else if (isZeroRated && isProductVatable) {
                // Zero-rated only for VATable items
                taxableAmount += itemTotal;
                // Zero-rated (0% VAT)
            } else if (isProductVatable && vatRate > 0) {
                // VATable item with tax rate
                const itemVAT = (itemTotal * vatRate) / 100;
                totalVAT += itemVAT;
                taxableAmount += itemTotal;
                
                // Group by VAT rate
                if (!vatByRate[vatRate]) {
                    vatByRate[vatRate] = {
                        amount: 0,
                        taxableAmount: 0
                    };
                }
                vatByRate[vatRate].amount += itemVAT;
                vatByRate[vatRate].taxableAmount += itemTotal;
            } else {
                // Non-VATable item
                nonTaxableAmount += itemTotal;
            }
        });
        
        const discount = cartData.discount || 0;
        const discountedSubtotal = Math.max(0, subtotal - discount);
        
        // Apply discount proportionally to VAT
        let discountedVAT = totalVAT;
        if (discount > 0 && taxableAmount > 0) {
            const discountRatio = discount / subtotal;
            discountedVAT = totalVAT * (1 - discountRatio);
        }
        
        const totalWithVAT = discountedSubtotal + discountedVAT;
        
        return {
            subtotal: subtotal,
            discount: discount,
            discountedSubtotal: discountedSubtotal,
            taxableAmount: taxableAmount,
            nonTaxableAmount: nonTaxableAmount,
            totalVAT: discountedVAT,
            totalWithVAT: totalWithVAT,
            vatByRate: vatByRate,
            customerVatStatus: customerStatus,
            isExempted: isExempted,
            isZeroRated: isZeroRated,
            hasVAT: discountedVAT > 0
        };
    }
    
    /**
     * Update all VAT elements in the modal
     */
    updateAllVATElements(vatData) {
        // 1. Update main totals
        this.updateMainTotals(vatData);
        
        // 2. Update VAT summary section
        this.updateVATSummarySection(vatData);
        
        // 3. Update confirmation section
        this.updateConfirmationSection(vatData);
        
        // 4. Show/hide VAT section based on whether VAT exists
        this.toggleVATSectionDisplay(vatData);
    }
    
    /**
     * Update main totals in the banner
     */
    updateMainTotals(vatData) {
        // Update total amount in banner
        const totalElement = document.getElementById('paymentTotalAmount');
        if (totalElement) {
            totalElement.textContent = `KES ${vatData.totalWithVAT.toFixed(2)}`;
        }
        
        // Update total in summary
        const totalSummaryElement = document.getElementById('paymentTotalAmountSummary');
        if (totalSummaryElement) {
            totalSummaryElement.textContent = `KES ${vatData.totalWithVAT.toFixed(2)}`;
        }
        
        // Update item count
        const countElement = document.getElementById('paymentItemCount');
        if (countElement && window.posState) {
            countElement.textContent = window.posState.cart?.length || 0;
        }
        
        // Update customer name
        const customerElement = document.getElementById('paymentCustomer');
        if (customerElement && window.posState?.currentCustomer) {
            customerElement.textContent = window.posState.currentCustomer.name || 'Walk-in';
        }
    }
    
    /**
     * Update VAT summary section
     */
    updateVATSummarySection(vatData) {
        // Update subtotal
        const subtotalElement = document.getElementById('paymentSubtotalAmount');
        if (subtotalElement) {
            subtotalElement.textContent = `KES ${vatData.subtotal.toFixed(2)}`;
        }
        
        // Update discount
        const discountElement = document.getElementById('paymentDiscountAmount');
        if (discountElement) {
            if (vatData.discount > 0) {
                discountElement.textContent = `-KES ${vatData.discount.toFixed(2)}`;
                discountElement.classList.remove('d-none');
            } else {
                discountElement.classList.add('d-none');
            }
        }
        
        // Update VAT amount - ONLY SHOW IF VAT EXISTS
        const vatElement = document.getElementById('paymentVATAmount');
        const vatLabel = vatElement?.closest('.d-flex');
        
        if (vatElement && vatLabel) {
            if (vatData.hasVAT) {
                vatElement.textContent = `KES ${vatData.totalVAT.toFixed(2)}`;
                vatLabel.classList.remove('d-none');
            } else {
                vatLabel.classList.add('d-none');
            }
        }
        
        // Update VAT breakdown
        this.updateVATBreakdown(vatData);
    }
    
    /**
     * Update VAT breakdown details
     */
    updateVATBreakdown(vatData) {
        const vatBreakdownElement = document.getElementById('paymentVATBreakdown');
        if (!vatBreakdownElement) return;
        
        // Clear existing content
        vatBreakdownElement.innerHTML = '';
        
        // Only show breakdown if there's VAT
        if (vatData.hasVAT && Object.keys(vatData.vatByRate).length > 0) {
            let breakdownHTML = '';
            
            // Show taxable/non-taxable summary
            if (vatData.taxableAmount > 0) {
                breakdownHTML += `
                    <div class="d-flex justify-content-between small mb-1">
                        <span>Taxable Amount:</span>
                        <span class="fw-semibold">KES ${vatData.taxableAmount.toFixed(2)}</span>
                    </div>
                `;
            }
            
            if (vatData.nonTaxableAmount > 0) {
                breakdownHTML += `
                    <div class="d-flex justify-content-between small mb-1">
                        <span>Non-taxable:</span>
                        <span>KES ${vatData.nonTaxableAmount.toFixed(2)}</span>
                    </div>
                `;
            }
            
            // Show VAT by rate
            if (Object.keys(vatData.vatByRate).length > 0) {
                breakdownHTML += '<div class="vat-rates mt-2 pt-2 border-top">';
                
                Object.entries(vatData.vatByRate).forEach(([rate, data]) => {
                    if (parseFloat(rate) > 0 && data.amount > 0) {
                        breakdownHTML += `
                            <div class="vat-rate-line d-flex justify-content-between mb-1">
                                <span>VAT ${rate}% on KES ${data.taxableAmount.toFixed(2)}:</span>
                                <span class="text-success fw-semibold">KES ${data.amount.toFixed(2)}</span>
                            </div>
                        `;
                    }
                });
                
                breakdownHTML += '</div>';
            }
            
            vatBreakdownElement.innerHTML = breakdownHTML;
            vatBreakdownElement.classList.remove('d-none');
        } else {
            // Show "No VAT applicable" message
            vatBreakdownElement.innerHTML = `
                <div class="alert alert-info small mb-0 mt-2">
                    <i class="bx bx-info-circle me-1"></i>
                    No VAT applicable to items in cart
                </div>
            `;
            vatBreakdownElement.classList.remove('d-none');
        }
    }
    
    /**
     * Update confirmation section
     */
    updateConfirmationSection(vatData) {
        // Update subtotal in confirmation
        const confirmSubtotal = document.getElementById('confirmSubtotal');
        if (confirmSubtotal) {
            confirmSubtotal.textContent = `KES ${vatData.subtotal.toFixed(2)}`;
        }
        
        // Update VAT amount in confirmation - ONLY SHOW IF VAT EXISTS
        const confirmVAT = document.getElementById('confirmVAT');
        const confirmVATRow = confirmVAT?.closest('.row')?.querySelector('.col-md-6:last-child');
        
        if (confirmVAT && confirmVATRow) {
            if (vatData.hasVAT) {
                confirmVAT.textContent = `KES ${vatData.totalVAT.toFixed(2)}`;
                confirmVATRow.classList.remove('d-none');
            } else {
                confirmVATRow.classList.add('d-none');
            }
        }
        
        // Update VAT breakdown in confirmation
        const confirmVATBreakdown = document.getElementById('confirmVATBreakdown');
        if (confirmVATBreakdown) {
            let breakdownText = '';
            
            if (vatData.hasVAT) {
                const rates = Object.keys(vatData.vatByRate).filter(rate => parseFloat(rate) > 0);
                if (rates.length === 1) {
                    breakdownText = `VAT ${rates[0]}% applied`;
                } else if (rates.length > 1) {
                    breakdownText = `Multiple VAT rates applied`;
                } else {
                    breakdownText = 'Zero-rated VAT';
                }
            } else {
                breakdownText = 'No VAT applied';
            }
            
            confirmVATBreakdown.textContent = breakdownText;
        }
        
        // Update total amount in confirmation
        const confirmAmount = document.getElementById('confirmAmount');
        if (confirmAmount) {
            confirmAmount.textContent = `KES ${vatData.totalWithVAT.toFixed(2)}`;
        }
    }
    
    /**
     * Toggle VAT section display based on whether VAT exists
     */
    toggleVATSectionDisplay(vatData) {
        const vatSection = document.querySelector('.payment-summary-section');
        if (!vatSection) return;
        
        // Only show VAT section if there's actual VAT or special customer status
        if (vatData.hasVAT || vatData.isExempted || vatData.isZeroRated) {
            vatSection.style.display = 'block';
            
            // Add appropriate styling based on VAT status
            if (vatData.isExempted) {
                vatSection.style.borderLeftColor = '#dc3545'; // Red for exempted
                vatSection.querySelector('h6').innerHTML = `
                    <i class="bx bx-shield-x me-2"></i>VAT Exempted Invoice
                `;
            } else if (vatData.isZeroRated) {
                vatSection.style.borderLeftColor = '#ffc107'; // Yellow for zero-rated
                vatSection.querySelector('h6').innerHTML = `
                    <i class="bx bx-shield-alt-2 me-2"></i>Zero-Rated Invoice
                `;
            } else if (vatData.hasVAT) {
                vatSection.style.borderLeftColor = '#198754'; // Green for VAT
                vatSection.querySelector('h6').innerHTML = `
                    <i class="bx bx-receipt me-2"></i>VAT Invoice Breakdown
                `;
            }
        } else {
            // Hide VAT section if no VAT applicable
            vatSection.style.display = 'none';
        }
    }
    
    /**
     * Reset VAT display
     */
    resetVATDisplay() {
        console.log('🔄 Resetting VAT display');
        
        // Reset main totals
        const totalElement = document.getElementById('paymentTotalAmount');
        if (totalElement) totalElement.textContent = 'KES 0.00';
        
        const totalSummaryElement = document.getElementById('paymentTotalAmountSummary');
        if (totalSummaryElement) totalSummaryElement.textContent = 'KES 0.00';
        
        // Reset subtotal
        const subtotalElement = document.getElementById('paymentSubtotalAmount');
        if (subtotalElement) subtotalElement.textContent = 'KES 0.00';
        
        // Reset discount
        const discountElement = document.getElementById('paymentDiscountAmount');
        if (discountElement) {
            discountElement.textContent = '-KES 0.00';
            discountElement.classList.add('d-none');
        }
        
        // Reset VAT amount
        const vatElement = document.getElementById('paymentVATAmount');
        const vatLabel = vatElement?.closest('.d-flex');
        if (vatLabel) vatLabel.classList.add('d-none');
        
        // Reset VAT breakdown
        const vatBreakdownElement = document.getElementById('paymentVATBreakdown');
        if (vatBreakdownElement) {
            vatBreakdownElement.innerHTML = '';
            vatBreakdownElement.classList.add('d-none');
        }
        
        // Hide VAT section
        const vatSection = document.querySelector('.payment-summary-section');
        if (vatSection) vatSection.style.display = 'none';
    }
    
    /**
     * Initialize VAT helper
     */
    init() {
        console.log('✅ Payment VAT Helper initialized');
        
        // Set up event listeners
        this.setupEventListeners();
        
        // Listen for cart changes
        this.setupCartChangeListener();
        
        return this;
    }
    
    /**
     * Set up event listeners
     */
    setupEventListeners() {
        // Update VAT when payment modal is shown
        const paymentModal = document.getElementById('paymentModal');
        if (paymentModal) {
            paymentModal.addEventListener('show.bs.modal', () => {
                setTimeout(() => this.updatePaymentModalVAT(), 100);
            });
        }
        
        // Update VAT when customer changes
        document.addEventListener('customerChanged', () => {
            this.updatePaymentModalVAT();
        });
        
        // Update VAT when discount changes
        document.addEventListener('discountChanged', () => {
            this.updatePaymentModalVAT();
        });
    }
    
    /**
     * Set up cart change listener
     */
    setupCartChangeListener() {
        if (window.posState && window.posState.cart) {
            // Create a proxy to detect cart changes
            const originalCart = window.posState.cart;
            
            window.posState.cart = new Proxy(originalCart, {
                set: (target, property, value) => {
                    target[property] = value;
                    
                    // If payment modal is open, update VAT
                    setTimeout(() => {
                        const modal = document.getElementById('paymentModal');
                        if (modal && modal.classList.contains('show')) {
                            this.updatePaymentModalVAT();
                        }
                    }, 100);
                    
                    return true;
                }
            });
        }
    }
}

// Create global instance
window.PaymentVATHelper = new PaymentVATHelper().init();