// =========================================================
// Base Payment Processor - Abstract class for all processors
// =========================================================

class BasePaymentProcessor {
    constructor(methodCode, methodName) {
        if (new.target === BasePaymentProcessor) {
            throw new Error("Cannot instantiate abstract class");
        }
        
        this.methodCode = methodCode;
        this.methodName = methodName;
        this.config = {};
        this.events = new Map();
    }

    // ========== ABSTRACT METHODS ==========
    async validate(data) {
        throw new Error('validate() must be implemented by subclass');
    }

    async process(data) {
        throw new Error('process() must be implemented by subclass');
    }

    async verify(transactionId) {
        throw new Error('verify() must be implemented by subclass');
    }

    async refund(transactionId, amount, reason = '') {
        throw new Error('refund() must be implemented by subclass');
    }

    // ========== UTILITY METHODS ==========
    formatCurrency(amount) {
        return `KES ${parseFloat(amount).toFixed(2).replace(/\B(?=(\d{3})+(?!\d))/g, ',')}`;
    }

    formatPhone(phone) {
        let clean = phone.replace(/\D/g, '');
        
        if (clean.startsWith('0')) {
            clean = '254' + clean.substring(1);
        } else if (!clean.startsWith('254')) {
            clean = '254' + clean;
        }
        
        return clean;
    }

    validatePhone(phone) {
        const clean = this.formatPhone(phone);
        if (!/^254[17]\d{8}$/.test(clean)) {
            throw new Error('Invalid Kenyan phone number');
        }
        return clean;
    }

    validateAmount(amount) {
        const num = parseFloat(amount);
        if (isNaN(num) || num <= 0) {
            throw new Error('Invalid amount');
        }
        return num;
    }

    validateEmail(email) {
        const regex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!regex.test(email)) {
            throw new Error('Invalid email address');
        }
        return email;
    }

    // ========== EVENT SYSTEM ==========
    on(event, callback) {
        if (!this.events.has(event)) {
            this.events.set(event, []);
        }
        this.events.get(event).push(callback);
    }

    off(event, callback) {
        if (this.events.has(event)) {
            const callbacks = this.events.get(event);
            const index = callbacks.indexOf(callback);
            if (index > -1) {
                callbacks.splice(index, 1);
            }
        }
    }

    trigger(event, data) {
        if (this.events.has(event)) {
            this.events.get(event).forEach(callback => {
                try {
                    callback(data);
                } catch (error) {
                    console.error(`Error in ${event} handler:`, error);
                }
            });
        }
    }

    // ========== API COMMUNICATION ==========
    async callAPI(endpoint, data, options = {}) {
        const controller = new AbortController();
        const timeout = setTimeout(() => controller.abort(), 10000);

        try {
            const csrfToken = document.querySelector('meta[name="csrf-token"]')?.content;
            
            const response = await fetch(endpoint, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                    'X-CSRF-TOKEN': csrfToken || '',
                    'X-Requested-With': 'XMLHttpRequest',
                    ...options.headers
                },
                body: JSON.stringify(data),
                signal: controller.signal
            });

            clearTimeout(timeout);

            if (!response.ok) {
                throw new Error(`HTTP ${response.status}`);
            }

            const result = await response.json();
            
            if (!result.success) {
                throw new Error(result.message || result.error || 'API request failed');
            }

            return result;

        } catch (error) {
            clearTimeout(timeout);
            
            if (error.name === 'AbortError') {
                throw new Error('Request timeout');
            }
            
            throw error;
        }
    }

    // ========== TRANSACTION LOGGING ==========
    logTransaction(type, data) {
        const logEntry = {
            timestamp: new Date().toISOString(),
            processor: this.methodCode,
            type: type,
            data: data,
            userAgent: navigator.userAgent
        };

        console.log(`[${this.methodCode.toUpperCase()}] ${type}:`, data);
        
        // Send to server for logging
        this.sendLogToServer(logEntry).catch(() => {
            // Silently fail logging
        });

        return logEntry;
    }

    async sendLogToServer(logEntry) {
        try {
            await this.callAPI('/api/logs/payment', logEntry);
        } catch (error) {
            // Don't throw errors for logging failures
            console.warn('Failed to send log:', error.message);
        }
    }

    // ========== ERROR HANDLING ==========
    createError(code, message, details = {}) {
        const error = new Error(message);
        error.code = code;
        error.details = details;
        error.processor = this.methodCode;
        error.timestamp = new Date().toISOString();
        return error;
    }

    handleError(error, context = '') {
        console.error(`[${this.methodCode}] Error${context ? ` in ${context}` : ''}:`, error);
        
        // Log the error
        this.logTransaction('error', {
            context: context,
            error: error.message,
            code: error.code,
            details: error.details,
            stack: error.stack
        });

        // Trigger error event
        this.trigger('error', { error, context });

        return error;
    }

    // ========== CONFIGURATION ==========
    setConfig(config) {
        this.config = { ...this.config, ...config };
        this.trigger('configUpdated', { config: this.config });
    }

    getConfig(key = null) {
        if (key === null) {
            return this.config;
        }
        return this.config[key];
    }

    // ========== VALIDATION HELPERS ==========
    validateRequiredFields(data, fields) {
        const missing = fields.filter(field => {
            const value = data[field];
            return value === undefined || value === null || value === '';
        });

        if (missing.length > 0) {
            throw this.createError(
                'MISSING_FIELDS',
                `Missing required fields: ${missing.join(', ')}`,
                { missingFields: missing }
            );
        }
    }

    validateNumericRange(value, min, max, fieldName) {
        const num = parseFloat(value);
        if (isNaN(num)) {
            throw this.createError(
                'INVALID_NUMBER',
                `${fieldName} must be a number`,
                { value, fieldName }
            );
        }
        
        if (num < min || num > max) {
            throw this.createError(
                'OUT_OF_RANGE',
                `${fieldName} must be between ${min} and ${max}`,
                { value, min, max, fieldName }
            );
        }
        
        return num;
    }

    validateStringLength(value, min, max, fieldName) {
        if (typeof value !== 'string') {
            throw this.createError(
                'INVALID_TYPE',
                `${fieldName} must be a string`,
                { value, fieldName }
            );
        }
        
        if (value.length < min || value.length > max) {
            throw this.createError(
                'INVALID_LENGTH',
                `${fieldName} must be between ${min} and ${max} characters`,
                { value, length: value.length, min, max, fieldName }
            );
        }
        
        return value;
    }

    // ========== FORMATTING ==========
    formatResponse(success, data, message = '') {
        return {
            success: success,
            message: message,
            data: data,
            timestamp: new Date().toISOString(),
            processor: this.methodCode
        };
    }

    // ========== CLEANUP ==========
    destroy() {
        this.events.clear();
        this.config = {};
        console.log(`[${this.methodCode}] Processor destroyed`);
    }
}

export default BasePaymentProcessor;